package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.Options;
import org.lsst.ccs.command.SupportedOption;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Option;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.subsystem.common.actions.BulkPsPowerAction;
import org.lsst.ccs.subsystem.common.focalplane.data.HasFocalPlaneData;
import org.lsst.ccs.subsystem.power.data.PowerDataGroup;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.states.PowerSupplyState;
import org.lsst.ccs.subsystem.power.states.RebPowerState;
import org.lsst.ccs.subsystem.power.states.RebDPhiState;
import org.lsst.ccs.subsystem.power.states.HvControlState;
import org.lsst.ccs.subsystem.power.states.RebHvControlState;
import org.lsst.ccs.subsystem.power.states.RebHvBiasState;
import org.lsst.ccs.subsystem.power.RebPsHVRegulator;

/**
 *  Implements the REB PS control subsystem.
 *
 *  @author The LSST CCS Team
 */
public class RebPowerSupplyMain extends Subsystem implements HasLifecycle, HasDataProviderInfos {

    private static final Logger LOG = Logger.getLogger(RebPowerSupplyMain.class.getName());

    @LookupField( strategy=LookupField.Strategy.CHILDREN)
    private List<PowerDevice> mainPsList = new ArrayList<>();
    @LookupField( strategy=LookupField.Strategy.CHILDREN)    
    private List<RebPsDevice> psDeviceList = new ArrayList<>();

        
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private DataProviderDictionaryService dictionaryService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private RebPsHVRegulator RebPsHvRegulator;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private EmergencyResponseManager emergencyResponseManager;
    
    //The list of all the Rebs
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String,RebPowerSupplyNode> rebs = new HashMap<>();
    
    @ConfigurationParameter(units = "ms", description = "During the powerOffAllRebs command this is the sleep time between each node being powered off.")
    private volatile long powerOffSleepMillis = 100;

    @ConfigurationParameter(name="psDeviceReadTimeout", description = "Read Timeout for the underlying PS driver", units = "ms")
    private volatile long psDeviceReadTimeout = 1000;    
    
    @ConfigurationParameter(category = "General", description = "Number of consecutive read failures (exceptions) before gonig Offline. Must be a positive number greater than zero.", range = "1..5", units = "unitless")    
    private volatile int psDeviceExcepNumToOffline = 1;
    
    //The map of all the Channels
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String,Channel> channels = new HashMap<>();

    private final SupportedOption dryRunOption = SupportedOption.getSupportedOption("dryRun");
    
    public RebPowerSupplyMain() {
        super("rebps", AgentInfo.AgentType.WORKER);
    }

    
    @Override
    public void build() {
        if (emergencyResponseManager == null) {
            throw new RuntimeException("Cannot operate the RebPower subsystem without an EmergencyResponseManager");
        }
        
        agentPropertiesService.setAgentProperty(BulkPsPowerAction.class.getCanonicalName(), "uses");
        agentPropertiesService.setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.POWER.name());
        agentPropertiesService.setAgentProperty(HasFocalPlaneData.AGENT_PROPERTY, HasFocalPlaneData.generatePropertyValue(PowerDataGroup.class));
    }
    
    @Override
    public void init()
    {
        for (RebPowerSupplyNode reb : rebs.values()) {
            agentStateService.registerState(RebPowerState.class, "REB power state", reb);        
            agentStateService.registerState(RebHvControlState.class, "REB HV control permit", reb);  // ALLOWED|BLOCKED      
            agentStateService.registerState(RebHvBiasState.class, "REB HV bias state", reb);        
            if ( reb instanceof CornerRaftRebPowerSupplyNode ) {
                agentStateService.registerState(RebDPhiState.class, "REB Dphi state", reb);        
            }
        
        }        
        for (RebPsDevice powerSupplyDevice : psDeviceList ) {
            agentStateService.registerState(PowerSupplyState.class, "Power Supply state", powerSupplyDevice);
        }
        
        setPsDeviceReadTimeout(psDeviceReadTimeout);
        setPsDeviceConsecutiveExceptionsToOffline(psDeviceExcepNumToOffline);
    }
    
    @ConfigurationParameterChanger(propertyName = "psDeviceExcepNumToOffline") 
    public void setPsDeviceConsecutiveExceptionsToOffline(int value) {
        psDeviceExcepNumToOffline = value;
        for (RebPsDevice powerSupplyDevice : psDeviceList ) {
            powerSupplyDevice.setConsecutiveExceptiionsToOffline(psDeviceExcepNumToOffline);
        }
        
    }

    @ConfigurationParameterChanger(propertyName = "psDeviceReadTimeout") 
    public void setPsDeviceReadTimeout(long timeout) {
        if ( timeout <= 0 ) {
            throw new IllegalArgumentException("Negative timeouts are not allowd");
        }
        psDeviceReadTimeout = timeout;
        for (RebPsDevice powerSupplyDevice : psDeviceList ) {
            powerSupplyDevice.setReadTimeout(psDeviceReadTimeout);
        }
        
    }
    
    @Override
    public void finalizeDictionary() {        
        //Manipulate the dictionary here: 
        for ( DataProviderInfo data: dictionaryService.getDataProviderDictionary().getDataProviderInfos() ) {
            PowerDataGroup dataGroup = PowerDataGroup.findPowerDataGroup(data);
            if (dataGroup != null) {
                dataGroup.addAttributesToDataInfo(data);
            }
        }
    }
    
    @Command(type=Command.CommandType.ACTION, description="Set whether REB HV Control is active")
    public void enableHvControl(boolean on) {
        RebPsHvRegulator.enableHvControl(on);
    }

    @Command(type=Command.CommandType.QUERY, description="returns whether REB HV Control is active")
    public boolean isHvControlActive() {
        return RebPsHvRegulator.isHvControlActive();
    }
    
    @Command(description="Emergency power off of all rebs at ~10/s")
    public void powerOffAllRebs() throws InterruptedException {
        String failedRebs = "";
        int count = rebs.size();
        for (RebPowerSupplyNode node: rebs.values()) {
            count--;
            if ( agentStateService.isComponentInState(node.rebPath, RebPowerState.ON) ) {
                LOG.log(Level.INFO, "Powering off reb: {0}", new Object[]{node.rebPath});
                try {
                    node.powerRebOff();
                } catch (PowerException ex) {
                    LOG.log(Level.WARNING, "Failed to power off reb: "+node.rebPath, ex);                    
                    failedRebs += node.rebPath + " ";
                }
                if ( count > 0 ) {
                    Thread.sleep(powerOffSleepMillis);
                }
            }
        }
        if ( ! failedRebs.isEmpty() ) {
            throw new RuntimeException("Failed to turn off the following rebs: "+failedRebs.trim());
        }
    }
    
    @Command(description="Open and disable all RebPS hvbias switches with no delay")
    public void setAllHvBiasToZero() {
        RebPsHvRegulator.enableHvControl(false);
        String failedRebs = "";
        for (RebPowerSupplyNode node: rebs.values()) {
            LOG.log(Level.INFO, "Opening HvBias switch for node: {0}", new Object[]{node.rebPath});
            try {
                node.hvBiasOff(true);
            } catch (PowerException ex) {
                LOG.log(Level.WARNING, "Failed to open HvBias switch for node: " + node.rebPath, ex);
                failedRebs += node.rebPath + " ";
            }
        }
        if (!failedRebs.isEmpty()) { // TODO ask Max
            throw new RuntimeException("Failed to open HvBias switch for rebs: " + failedRebs.trim());
        }
    }

    @Option(name = "dryRun", description = "If true, no action occurs on the Rebs")
    @Command(description = "Power On matching Rebs", type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, autoAck = false)
    public void powerOnRebs(Options options,
            @Argument(description = "Regular expression to match the rebs by path.") String regEx,
            @Argument(description = "Number of seconds to wait between rebs [2:30]") double delay
    ) {

        Pattern rebPattern = Pattern.compile(regEx);
        boolean dryRun = options.hasOption(dryRunOption);

        List<String> rebsToTurnOn = new ArrayList<>();
        List<String> rebsAlreadyOn = new ArrayList<>();
        Map<String, RebPowerState> otherStatesRebs = new HashMap<>();
        for (RebPowerSupplyNode node : rebs.values()) {
            String rebPath = node.rebPath;
            if (rebPattern.matcher(rebPath).matches()) {
                RebPowerState rebState = (RebPowerState)agentStateService.getComponentState(rebPath, RebPowerState.class);
                if ( rebState == RebPowerState.OFF ) {
                    rebsToTurnOn.add(rebPath);
                } else if ( rebState == RebPowerState.ON ) {
                    rebsAlreadyOn.add(rebPath);
                } else {
                    otherStatesRebs.put(rebPath, rebState);
                }
            }
        }

        //Estimate an upper limit for the command duration.
        double rebPowerOnSeconds = 2.5;
        Duration duration = rebsToTurnOn.isEmpty() ? Duration.ofSeconds(1) : Duration.ofSeconds((long) (1.3 * (rebsToTurnOn.size() * rebPowerOnSeconds + (rebsToTurnOn.size() - 1) * delay)));

        helper()
            .precondition(delay >= 2 && delay <= 30, "Power on delay must be in range [2:30] seconds")
            .precondition(otherStatesRebs.isEmpty(), "Bad RebPowerState detected for the rebs: "+otherStatesRebs)
            .duration(duration)
            .action(() -> {

                if (! rebsAlreadyOn.isEmpty()) {
                    LOG.log(Level.INFO, "Skipping Rebs already on: {0}", rebsAlreadyOn);
                }

                if (rebsToTurnOn.isEmpty()) {
                    LOG.log(Level.INFO, "There are no Rebs matching {0}", rebPattern);
                    return;
                }

                LOG.log(Level.INFO, "Turning on Rebs:\n  {0}\nUsing a {1}s pause.\nDry Run = {2}",
                        new Object[]{rebsToTurnOn, delay, dryRun});

                int count = 0;
                for (String rebPath : rebsToTurnOn) {

                    count++;
                    RebPowerSupplyNode node = rebs.get(rebPath);
                    LOG.log(Level.INFO, "Invoking powerRebOn on Reb: {0}", rebPath);
                    if (!dryRun) {
                        node.powerRebOn();
                    }

                    if (count != rebsToTurnOn.size()) {
                        LOG.log(Level.INFO, "Sleep for {0} seconds", delay);
                        //Wait the specified amount of time
                        if (!dryRun) {
                            try {
                                Thread.sleep((long) (delay * 1000L));
                            } catch (InterruptedException ex) {
                            }
                        }
                    }
                }
            } );
    }
    
    @Option(name = "dryRun", description = "If true, no action occurs on the Rebs")
    @Command(description = "Power Off matching Rebs", type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, autoAck = false)
    public void powerOffRebs(Options options,
            @Argument(description = "Regular expression to match the rebs by path.") String regEx,
            @Argument(description = "Number of seconds to wait between rebs [0.5:5]") double delay
    ) {

        Pattern rebPattern = Pattern.compile(regEx);
        boolean dryRun = options.hasOption(dryRunOption);

        //Of the rebs that match the regular expression, get the following lists
        // - the rebs to be turned off 
        // - the rebs that are already off; they will be skipped
        // - the rebs in some other RebPowerState; if any are present, issue a warning
        List<String> rebsToTurnOff = new ArrayList<>();
        List<String> rebsAlreadyOff = new ArrayList<>();
        Map<String, RebPowerState> otherStatesRebs = new HashMap<>();
        for (RebPowerSupplyNode node : rebs.values()) {
            String rebPath = node.rebPath;
            if (rebPattern.matcher(rebPath).matches()) {
                RebPowerState rebState = (RebPowerState)agentStateService.getComponentState(rebPath, RebPowerState.class);
                if ( rebState == RebPowerState.ON ) {
                    rebsToTurnOff.add(rebPath);
                } else if ( rebState == RebPowerState.OFF ) {
                    rebsAlreadyOff.add(rebPath);
                } else {
                    otherStatesRebs.put(rebPath, rebState);
                }
            }
        }

        double rebPowerOffSeconds = 0.1;
        //Estimate an upper limit for the command duration.
        Duration duration = rebsToTurnOff.isEmpty() ? Duration.ofSeconds(1) : Duration.ofSeconds((long) (1.3 * (rebsToTurnOff.size() * rebPowerOffSeconds + (rebsToTurnOff.size() - 1) * delay)));

        helper()
            .precondition(delay >= 0.5 && delay <= 5, "Power off delay must be in range [0.5:5] seconds")
            .duration(duration)
            .action(() -> {

                if (! otherStatesRebs.isEmpty()) {
                    LOG.log(Level.WARNING, "Inconsistent RebPowerState detected for reb nodes: {0}",otherStatesRebs);
                }

                if (! rebsAlreadyOff.isEmpty()) {
                    LOG.log(Level.INFO, "Skipping Rebs already off {0}", rebsAlreadyOff);
                }

                if (rebsToTurnOff.isEmpty()) {
                    LOG.log(Level.INFO, "There are no matching Rebs to power off for {0}", rebPattern);
                    return;
                }

                LOG.log(Level.INFO, "Turning off Rebs:\n  {0}\nUsing a {1}s pause.\nDry Run = {2}", 
                        new Object[]{rebsToTurnOff, delay, dryRun});

                int count = 0;
                for (String rebPath : rebsToTurnOff) {

                    count++;
                    RebPowerSupplyNode node = rebs.get(rebPath);
                    LOG.log(Level.INFO, "Invoking powerRebOff on Reb: {0}", rebPath);
                    if (!dryRun) {
                        node.powerRebOff();
                    }

                    if (count != rebsToTurnOff.size()) {
                        LOG.log(Level.INFO, "Sleep for {0} seconds", delay);
                        //Wait the specified amount of time
                        if (!dryRun) {
                            try {
                                Thread.sleep((long) (delay * 1000L));
                            } catch (InterruptedException ex) {
                            }
                        }
                    }
                }
            } );
    }
    
    
    @Option(name = "dryRun", description = "If true, no action occurs on the Rebs")
    @Command(description = "Close the HvBias switch on matching rebs", type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, autoAck = false)
    public void closeHvBiasSwitches(Options options,
            @Argument(description = "Regular expression to match rebs") String regEx,
            @Argument(description = "Number of seconds to wait between rebs [0.3:5]") double delay
    ) {

        Pattern rebPattern = Pattern.compile(regEx);
        boolean dryRun = options.hasOption(dryRunOption);

        //Of the rebs that match the regular expression, get the following lists
        // - the rebs for which to close the HvBias switch
        // - the rebs for which the HvBias switch is already closed; they will be skipped
        // - the rebs in some other RebHvBiasState; if any are present, fail the command precondition
        List<String> rebsToClose = new ArrayList<>();
        List<String> rebsAlreadyClosed = new ArrayList<>();
        Map<String, RebHvBiasState> otherStatesRebs = new HashMap<>();
        for (RebPowerSupplyNode node : rebs.values()) {
            String rebPath = node.rebPath;
            if (rebPattern.matcher(rebPath).matches()) {
                RebHvBiasState rebState = (RebHvBiasState)agentStateService.getComponentState(rebPath, RebHvBiasState.class);
                if ( rebState == RebHvBiasState.OFF ) {
                    rebsToClose.add(rebPath);
                } else if ( rebState == RebHvBiasState.ON ) {
                    rebsAlreadyClosed.add(rebPath);
                } else {
                    otherStatesRebs.put(rebPath, rebState);
                }
            }
        }

        //Estimate an upper limit for the command duration.
        double rebHvBiasSwitchCloseSeconds = 0.10;
        Duration duration = rebsToClose.isEmpty() ? Duration.ofSeconds(1) : Duration.ofSeconds((long) (1.3 * (rebsToClose.size() * rebHvBiasSwitchCloseSeconds + (rebsToClose.size() - 1) * delay)));

        helper()
            .precondition(delay >= 0.3 && delay <= 5, "Delay must be in range [0.3:5] seconds")
            .precondition(otherStatesRebs.isEmpty(), "Bad RebHvBiasState detected for: "+otherStatesRebs)
            .duration(duration)
            .action(() -> {
                if (! rebsAlreadyClosed.isEmpty()) {
                    LOG.log(Level.INFO, "Skipping Rebs already closed: {0}", rebsAlreadyClosed);
                }
                if (rebsToClose.isEmpty()) {
                    LOG.log(Level.INFO, "There are no Rebs matching {0}", rebPattern);
                    return;
                }
                LOG.log(Level.INFO, "Closing the HvBias switch on:\n  {0} with {1}s pause.\nDryRun:{2}\n",
                        new Object[]{rebsToClose, delay, dryRun});

                int count = 0;
                for (String rebPath : rebsToClose) {

                    count++;
                    RebPowerSupplyNode node = rebs.get(rebPath);
                    LOG.log(Level.INFO, "Invoking hvBiasOn() on Reb: {0}", rebPath);
                    if (!dryRun) {
                        node.hvBiasOn();
                    }

                    if (count != rebsToClose.size()) {
                        LOG.log(Level.INFO, "Sleep for {0} seconds", delay);
                        //Wait the specified amount of time
                        if (!dryRun) {
                            try {
                                Thread.sleep((long) (delay * 1000L));
                            } catch (InterruptedException ex) {
                            }
                        }
                    }
                }
            } );
    }
    
    // for selected rebs: open hv switches, block control, set DACs-->0, there is minimal delay and no checking is needed
    @Option(name = "dryRun", description = "If true, no action occurs on the Rebs")
    @Command(description = "Open hv switch, set volts=0 and block control on a specified set of Rebs", type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, autoAck = false)
    public void setHvBiasToZero(Options options,
            @Argument(description = "Regular expression to match rebs") String regEx) {

        Pattern rebPattern = Pattern.compile(regEx);
        boolean dryRun = options.hasOption(dryRunOption);
        long delay = 10;

        for (RebPowerSupplyNode node : rebs.values()) {
            String rebPath = node.rebPath;
            if (rebPattern.matcher(rebPath).matches()) {
                if (!dryRun) {
                    try {
                        node.hvBiasOff();
                        node.allowHvControl(false);
                        node.setHvBiasDac(0);
                    } catch (PowerException pe) {
                        LOG.log(Level.SEVERE, "Could not turn HvBias OFF for reb " + rebPath, pe);
                    }
                }
                LOG.log(Level.INFO, String.format("%sinvoked setHvBiasToZero on Reb: %s", dryRun ? "would have " : "", rebPath));
                if (!dryRun) {
                    try {
                        Thread.sleep((long) delay);
                    } catch (InterruptedException ex) {
                    }
                }
            }
        }
    }

    // open hvbias switches, there is minimal delay and no checking is needed
    @Option(name = "dryRun", description = "If true, no action occurs on the Rebs")
    @Command(description = "Open and (re-)enable the HvBias switch on a specified set of Rebs", type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, autoAck = false)
    public void openHvBiasSwitches(Options options,
            @Argument(description = "Regular expression to match rebs") String regEx) {

        Pattern rebPattern = Pattern.compile(regEx);
        boolean dryRun = options.hasOption(dryRunOption);
        long delay = 10;

        for (RebPowerSupplyNode node : rebs.values()) {
            String rebPath = node.rebPath;
            if (rebPattern.matcher(rebPath).matches()) {
                if (!dryRun) {
                    try {
                        node.hvBiasOff();
                    } catch (PowerException pe) {
                        LOG.log(Level.SEVERE, "Could not turn HvBias OFF for reb " + rebPath, pe);
                        // TODO raise alert ?  I don't think this should fail anymore
                    }
                }
                LOG.log(Level.INFO, String.format("%sinvoked hvBiasOff on Reb: %s", dryRun ? "would have " : "", rebPath));
                if (!dryRun) {
                    try {
                        Thread.sleep((long) delay);
                    } catch (InterruptedException ex) {
                    }
                }
            }
        }
    }

    @Option(name = "dryRun", description = "If true, no action occurs on the Rebs")
    @Command(description = "Set allowHvControl(true|false) on matching Rebs", type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, autoAck = false)
    public void allowHvControl(Options options,
            @Argument(description = "Regular expression to match rebs") String regEx,
            @Argument(description = "true|false to allow|disallow HvControl") boolean allow) {

            Pattern rebPattern = Pattern.compile(regEx);
            boolean dryRun = options.hasOption(dryRunOption);
            int count = 0;

            // - the rebs for which AllowHvControl() is called
            List<String> rebsToCommand = new ArrayList<>();
            for (RebPowerSupplyNode node : rebs.values()) {
                String rebPath = node.rebPath;
                if (rebPattern.matcher(rebPath).matches()) {
                    rebsToCommand.add(rebPath);
                    count++;
                    if (!dryRun) {
                        node.allowHvControl(allow);
                        LOG.log(Level.FINE, "Invoking allowHvControl({0}) on Reb: {1}", new Object[]{allow, rebPath});
                    }
                }
            }
            LOG.log(Level.INFO, String.format("allowHvControl() %s called on %d Rebs", 
                        dryRun ? "would have been (DRYRUN) " : "", count));
            LOG.log(Level.FINE, "Rebs matched: {0}\n", new Object[]{rebsToCommand});
    }

}
