package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.LinkedHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.monitor.Channel;
import java.util.Collections;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.power.states.RebPowerState;
import org.lsst.ccs.subsystem.power.states.RebHVRegulationState;
import org.lsst.ccs.subsystem.power.data.RebPsState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.data.RebHVBiasInfo;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.bus.data.Alert;
import java.time.format.DateTimeFormatter;
import org.lsst.ccs.bus.states.AlertState;

/**
 *  Implements the REB PS HV regulation system
 *
 *  @author The LSST CCS Team
 */
public class RebPsHVRegulator implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    
    private static final boolean TESTMODE=false;

    private static final Logger LOG = Logger.getLogger(RebPsHVRegulator.class.getName());
    //    private StateBundle deviceAndRebsState = new StateBundle();
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    //The list of all the Rebs
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String, RebPowerSupplyNode> rebPowers = new HashMap<>();
    
    @LookupField(strategy = LookupField.Strategy.TREE,pathFilter = "(R\\d\\d)/(Reb\\d|Reb[WG])/hvbias/(DAC|[VI]befSwch)" )
    private List<Channel> rebPowerHVChannels = new ArrayList<>();
    private static final Map<String,Channel> rebPowerHVChannelsMap = new HashMap<>();

    
    int maxstep = 25;
    @ConfigurationParameter(name="initial_hv_step", category="HVRegulation", isFinal=true, units = "unitless", description = "Initial HV regulation step size")
    int initial_stepsize = 15;
    @ConfigurationParameter(name="volts_per_hv_step", category="HVRegulation", isFinal=true, units = "V", description = "Volts per HV regulation step")
    double volts_per_step0 = 0.125;

    boolean enable_hv_regulation = false;
    
    private static final Map<String,RebHVBiasInfo> hvBiasInfoMap = new HashMap<>();


    private double small_delay = 0.05;

    private int    limit_steps = 0;
    private double hvCsetpt = 30.0;
    private double hvSsetpt = 50.0;
    private double volts_max = 52.0;
    private double volts_min = 20.0;
    private final int dac_max_acceptable = 3200;
    private double current_max = 0.150;
    private int hvbias_dac_min = 1100;
    private boolean hv_enable = true;

    private long updateHVMillis = 10000;

    List<String> allrebs = new ArrayList<>(hvBiasInfoMap.keySet());
    

    // functions

    public RebPsHVRegulator() {
    }

    @Command(type=CommandType.QUERY, description="get hvBias volts config value")
    public double get_hvbiasVolts_config(String ch) {
	final double hvconf = Double.parseDouble(sce.getConfigurationParameterValue(ch, "hvBiasVolts"));

	double value = hvconf;
	
	if (hvconf < volts_min || hvconf > volts_max) {
	    LOG.log(Level.SEVERE,
		    () -> String.format("hvconf volts:%s out of allowed range: %f--%f",hvconf, volts_min, volts_max));
	    value = 0.0; // this will now also disable the regulation for the channel
	    if (get_chan_hvbias_dac(ch)>0) {
		set_hvbias_dac(ch, 0);
	    }
	}
	return value;
    }

    @Command(type=CommandType.QUERY, description="get hvBias VbefSwch value")
    public double get_hvbias_volts(String ch) {
	double volts = rebPowerHVChannelsMap.get(ch+"/hvbias/VbefSwch").getValue();;
	if (volts < 0 || volts > volts_max) {
	    LOG.log(Level.SEVERE,
			() -> String.format("hvbias value:%s out of allowed range: 0--%f",volts, volts_max));
	 }
	return volts;
    }


    @Command(type=CommandType.QUERY, description="get hvBias current")
    public double get_hvbias_current(String ch) {
	double current = rebPowerHVChannelsMap.get(ch+"/hvbias/IbefSwch").getValue();
	if (current_max < 0 || current > current_max) {
	                        LOG.log(Level.SEVERE,
					() -> String.format("hvbias current:%s out of allowed range: 0--%f",
							     current, current_max));
	}
	return current;
    }

    /**
     *  Set the HV bias DAC if safety checks OK 
     */
    void set_hvbias_dac(String reb, int dac)  {
	if (dac < 0 || dac > dac_max_acceptable) {
	    LOG.log(Level.SEVERE,
		    () -> String.format("reb:%s - DAC request %d out of allowed range: 0--%d   CHANGE REJECTED!",reb,dac, dac_max_acceptable));
	} else {
	    try {
		if (agentStateService.isComponentInState(reb, RebHVRegulationState.ALLOWED)) {
		    ((RebPowerSupplyNode)(hvBiasInfoMap.get(reb).getTarget())).setHVBiasDac(dac,true);
		} else {
		    LOG.log(Level.SEVERE, "Set HV bias dac aborted because regulation of this channel has been disabled.");
		}
	    } catch (PowerException ex) {
		LOG.log(Level.SEVERE, "Failed to set HV bias dac: " + ex); 
	    }
	}
    }

    /**
     *  Get channel HV bias DAC value
     */
    @Command(type=CommandType.QUERY, description="get channel hvBias dac rebps value")
    public int get_chan_hvbias_dac(String ch) {
	int hvbias_dac = (int)rebPowerHVChannelsMap.get(ch+"/hvbias/DAC").getValue();
	return hvbias_dac;
    }


    /**
     *  disable/enable the HV regulation of a given reb
     */
    @Command(type=CommandType.ACTION, description="disable/enable the HV regulation of a given reb")
    public void hv_regulation_on(String reb, boolean on) throws PowerException {
	hvBiasInfoMap.get(reb).setEnable(on);
    }
    
    void setup_reb_states()
    {

	double setpt = 0.0;
	boolean enable = false;
	double volts = 0.0;
	double last_volts = 0.0;
	double current = 0.0;
	int dac = 0;
	int last_dac = 0;
	int delta_dac = 0;
	double delta_volts = 0.0;
	double volts_per_step = volts_per_step0;

	for (Entry<String, RebPowerSupplyNode> rebNode : rebPowers.entrySet()) {
	    String rebpath = rebNode.getKey();

	    LOG.log(Level.INFO, String.format("rebpath: %s",rebpath));

	    allrebs.add(rebpath);
	    
	    RebPowerSupplyNode target = rebNode.getValue();
	    boolean rebstate = false;
	    /*
	    deviceAndRebsState = (target.getRebPsDevice()).getRebPsDeviceState();
	    RebPowerState rebCurrentState = (RebPowerState)deviceAndRebsState.getComponentState(rebpath, RebPowerState.class);
	    if ( rebCurrentState == RebPowerState.ON ) {
		 rebstate = true;
	    }
	    */
	    
	    synchronized (agentStateService.getStateLock()) {
		if (agentStateService.isComponentInState(rebpath, RebPowerState.ON)) {
		    rebstate = true;
		}
	    }
	    
	    hvBiasInfoMap.put(rebpath, new RebHVBiasInfo(target, rebstate, setpt, enable, volts, last_volts, current, dac, last_dac, delta_dac, delta_volts, volts_per_step0));

	}

	Collections.sort(allrebs);

    } 


    /**
     *  Update the dictionary of HV bias info
     */
    void update_hvBiasInfoMap() throws PowerException {

	for (String reb : allrebs) {

	    //	    LOG.log(Level.INFO, String.format("hvBiasInfoMap update for reb %s",reb));


	    /*
	    deviceAndRebsState = ((RebPowerSupplyNode)(hvBiasInfoMap.get(reb).getTarget())).getRebPsDevice().getRebPsDeviceState();
            RebPowerState rebCurrentState = (RebPowerState)deviceAndRebsState.getComponentState(reb, RebPowerState.class);

	    if ( rebCurrentState == RebPowerState.ON ) {
		 hvBiasInfoMap.get(reb).setState(true);
	    } else {
		 hvBiasInfoMap.get(reb).setState(false);
	    }
	    */
	    
	    synchronized (agentStateService.getStateLock()) {
		if (agentStateService.isComponentInState(reb, RebPowerState.ON)) {
		    hvBiasInfoMap.get(reb).setState(true);
		} else {
		    hvBiasInfoMap.get(reb).setState(false);
		}
	    }

	    if (get_hvbiasVolts_config(reb)>0.0) {
		hvBiasInfoMap.get(reb).setEnable(true);
	    } else {
		hvBiasInfoMap.get(reb).setEnable(false);
		continue;
	   }
	    

	    hvBiasInfoMap.get(reb).setSetPt(get_hvbiasVolts_config(reb));
	    
	    int biasDac = (int)get_chan_hvbias_dac(reb);
	    LOG.log(Level.FINE, String.format("biasDac value read for reb %s is %d",reb,biasDac));
	    hvBiasInfoMap.get(reb).setDac(biasDac);

	    hvBiasInfoMap.get(reb).setLastVolts(hvBiasInfoMap.get(reb).getVolts());

	    hvBiasInfoMap.get(reb).setVolts(get_hvbias_volts(reb));
        
     	    hvBiasInfoMap.get(reb).setCurrent(get_hvbias_current(reb));
        
	    hvBiasInfoMap.get(reb).setLastDac((int)hvBiasInfoMap.get(reb).getDac());
	    
	    hvBiasInfoMap.get(reb).setDeltaDac(0);
	    hvBiasInfoMap.get(reb).setDeltaVolts(0.0);
	    if (hvBiasInfoMap.get(reb).getLastDac() != 0 && (hvBiasInfoMap.get(reb).getDac() - hvBiasInfoMap.get(reb).getLastDac()) != 0) {
    		double old = hvBiasInfoMap.get(reb).getVoltsPerStep();

		
	    	double newval = (hvBiasInfoMap.get(reb).getVolts() -
				 hvBiasInfoMap.get(reb).getLastVolts()) /
		    (hvBiasInfoMap.get(reb).getDac() - hvBiasInfoMap.get(reb).getLastDac());
		
                if (newval < 0.025) {
                    newval = 0.025;
                }   
                hvBiasInfoMap.get(reb).setVoltsPerStep((2.0 * old + newval) / 3.0);
	    }
	}
    }


    /**
     *  Return DAC step size to be used
     */
    int get_hvbias_dac_steps(double setpt, double volts, double volts_per_step){
        int local_max = (int)(maxstep / (volts_per_step / 0.04));
        int steps = (int)(Math.round((setpt - volts) / volts_per_step));
        if (steps > local_max) {
            steps = local_max;
        }
        if (steps < -local_max) {
            steps = -local_max;
        }
        return steps;
    }


    public void build()
    {

	AgentPeriodicTask pt;

	//Create and schedule an AgentPeriodicTask to update the RebPowerState

	pt = new AgentPeriodicTask("reb-hv-update",
				   () -> updateRebHVSettings()).withPeriod(Duration.ofMillis(updateHVMillis));
	periodicTaskService.scheduleAgentPeriodicTask(pt);
	
	
	 // Place chnnels in a map for faster access
	for (Channel ch : rebPowerHVChannels) {
	    rebPowerHVChannelsMap.put(ch.getPath(), ch);
	}

	LOG.log(Level.INFO, String.format("The length of rebPowerHVChannelsMap is %d",rebPowerHVChannelsMap.size()));


	setup_reb_states();
    
    }


    /**
     *  Initialize reb hvbias regulator components
     */
    public void init()
    {

	LOG.log(Level.INFO, String.format("The length of allrebs is %d",allrebs.size()));
    }

    @Command(type=CommandType.ACTION, description="Set whether REB HV regulation is active")
    public void enable_hv_regulation(boolean on) {
	enable_hv_regulation = on;
        for (String reb : allrebs) {
	    ((RebPowerSupplyNode)(hvBiasInfoMap.get(reb).getTarget())).setActiveHVControlState(on);
	}
    }

    @Command(type=CommandType.QUERY, description="returns whether REB HV regulation is active")
    public boolean isHvRegulationActive() {
	return(enable_hv_regulation);
    }

    
    /**
     *  Updates the REB power state periodically.                                                   
     */
    //    @Command(type=CommandType.ACTION, description="updates REB HV settings")
    public void updateRebHVSettings() {

	long t0 = System.currentTimeMillis();
	double max_delta = 0.0;
	int changes = 0;

	if (!enable_hv_regulation) {
	    return;
	}
	
	try {
	    update_hvBiasInfoMap();
	} catch (PowerException ex) {
	    LOG.log(Level.SEVERE, "Failed to update HV bias dict"); // a more severe action should occur
	}


	for (String reb : allrebs) {
	    LOG.log(Level.INFO, String.format("Working on update for %s",reb));
      	    if ((TESTMODE || (hvBiasInfoMap.get(reb).getState() && hvBiasInfoMap.get(reb).getEnable())) &&
		agentStateService.isComponentInState(reb, RebHVRegulationState.ALLOWED)) {

		LOG.log(Level.INFO, String.format("%s: state:%s  enable:%s",reb,hvBiasInfoMap.get(reb).getState()?"true":"false",
						    hvBiasInfoMap.get(reb).getEnable()?"true":"false"));
		double volts = hvBiasInfoMap.get(reb).getVolts();
		double setpt = hvBiasInfoMap.get(reb).getSetPt();
		double delta = setpt - volts;
		if (Math.abs(delta) > max_delta) {
		    max_delta = Math.abs(delta);
		}

		
		if (hvBiasInfoMap.get(reb).getDac() < hvbias_dac_min) {
		    LOG.log(Level.INFO, () -> String.format(
							      "%s: Configure dac to min=%s",reb, hvbias_dac_min));

		    if (TESTMODE) {
			LOG.log(Level.INFO, () -> String.format(
								  "If not in test mode, reb %s would have had its set point changed from %d to %d (min)",
								  reb, hvBiasInfoMap.get(reb).getDac(), hvbias_dac_min));
		    } else {
			LOG.log(Level.INFO, String.format("reb %s set point changing from %d to %d (min)",
								  reb, hvBiasInfoMap.get(reb).getDac(), hvbias_dac_min));
			set_hvbias_dac(reb,hvbias_dac_min);
		    }

		    changes += 1;
		} else {
		    double volts_per_step = hvBiasInfoMap.get(reb).getVoltsPerStep();
		    int steps = get_hvbias_dac_steps(setpt, volts, volts_per_step);
		    hvBiasInfoMap.get(reb).setDeltaDac(steps);
		    if (steps != 0) {

			int new_dac = hvBiasInfoMap.get(reb).getDac() + steps;
			if (new_dac < hvbias_dac_min + maxstep) {
			    steps = initial_stepsize;
			    new_dac = hvBiasInfoMap.get(reb).getDac() + steps;
			}

			LOG.log(Level.INFO, String.format(
							      "%s: hvBias %4d->%4d steps={%3d} for delta=%7.3f volts/step=%5.3f volts=%6.3f",
							      reb,
							      hvBiasInfoMap.get(reb).getDac(),
							      new_dac,
							      steps,
							      delta,
							      volts_per_step,
							      hvBiasInfoMap.get(reb).getVolts()
							      )
			    );

		    
			if (TESTMODE) {
			    LOG.log(Level.INFO, String.format("If not in test mode, reb %s would have had its set point changed from %d to %d",
								      reb, hvBiasInfoMap.get(reb).getDac(), new_dac));
			} else {
			    LOG.log(Level.INFO, String.format("reb %s set point changing from %d to %d",
								      reb, hvBiasInfoMap.get(reb).getDac(), new_dac));

			    set_hvbias_dac(reb,new_dac);
			}

			changes += 1;
		    }
		}

	    } else {
		continue;
	    }

	    try {
		Thread.sleep((long)(1000*small_delay));
	    } catch (InterruptedException ex) {
		LOG.log(Level.SEVERE, "Sleep interrupted");
	    }
	}
	
	long t1 = System.currentTimeMillis();
	long delta_time = t1 - t0;
	if (changes>0) {
	    
	    DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm:ss");
	    
	    LOG.log(Level.INFO,
		    String.format(
				       "loop_time=%f change_count=%d at %s",
	       delta_time/1000.,changes, LocalTime.now().format(formatter) ));
	}
    }
}
