package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverConstants.ConnType;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 *  Interface to an ASCII-commanded power supply device.
 *
 *  @author Owen Saxton
 */
public abstract class PowerDevice extends Device {

    /**
     *  Constants.
     */
    protected static final Logger LOG = Logger.getLogger(PowerDevice.class.getName());

    protected static final int
        NETWORK_CONN_INVALID = 0x01,  // Network connection type is invalid
        CONN_TYPE_NOT_CONFIG = 0x02,  // Don't get connection type from configuration
        DEVC_ID_NOT_CONFIG   = 0x04,  // Don't get device ID from configuration
        DEVC_PARM_NOT_CONFIG = 0x08;  // Don't get the device parameter from configuration

    private static final int
        MON_TYPE_VOLTAGE = 0,
        MON_TYPE_CURRENT = 1;

    private static final String
        CONN_TYPE = "connType",
        DEVC_ID   = "devcId",
        DEVC_PARM = "devcParm";

    private static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
    }

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name=DEVC_ID, category=Power.POWER, isFinal=true, units = "unitless", description = "The device ID: host name; serial device name")
    protected volatile String devcId;
    @ConfigurationParameter(name=DEVC_PARM, category=Power.POWER, isFinal=true, units = "unitless", description = "The device parameter: port number; baud rate")
    protected volatile int devcParm;
    @ConfigurationParameter(name=CONN_TYPE, category=Power.POWER, isFinal=true, units = "unitless", description = "The connection type: network; serial")
    protected volatile ConnType connType;

    private final String devcName;
    protected PowerSupplyDriver psd;
    private final int options;
    private int minChan, maxChan;
    private boolean initError = false;


    /**
     *  Constructor.
     *
     *  @param  name      The device name
     *  @param  psd       The power supply device driver object
     *  @param  options   The options word
     *  @param  minChan   The minimum channel number
     *  @param  maxChan   The maximum channel number
     */
    public PowerDevice(String name, PowerSupplyDriver psd, int options, int minChan, int maxChan)
    {
        super();
        this.devcName = name;
        this.psd = psd;
        this.options = options;
        this.minChan = minChan;
        this.maxChan = maxChan;
    }


    /**
     *  Constructor.
     *
     *  @param  name      The device name
     *  @param  psd       The power supply device driver object
     *  @param  connType  The connection type: network; serial
     *  @param  devcId    The device ID: host name; serial device name
     *  @param  devcParm  The device parameter: port number; baud rate
     *  @param  options   The options word
     *  @param  minChan   The minimum channel number
     *  @param  maxChan   The maximum channel number
     */
    public PowerDevice(String name, PowerSupplyDriver psd, ConnType connType, String devcId,
                       int devcParm, int options, int minChan, int maxChan)
    {
        super();
        this.devcName = name;
        this.psd = psd;
        if ((options & CONN_TYPE_NOT_CONFIG) != 0) {
            this.connType = connType;
        }
        if ((options & DEVC_ID_NOT_CONFIG) != 0) {
            this.devcId = devcId;
        }
        if ((options & DEVC_PARM_NOT_CONFIG) != 0) {
            this.devcParm = devcParm;
        }
        this.options = options;
        this.minChan = minChan;
        this.maxChan = maxChan;
    }


    /**
     *  Sets the minimum channel number.
     *
     *  @param  chan  The minimum channel number
     */
    public void setMinChannel(int chan)
    {
        minChan = chan;
    }


    /**
     *  Sets the maximum channel number.
     *
     *  @param  chan  The maximum channel number
     */
    public void setMaxChannel(int chan)
    {
        maxChan = chan;
    }


    /**
     *  Gets the minimum channel number.
     *
     *  @return  The minimum channel number
     */
    public int getMinChannel()
    {
        return minChan;
    }


    /**
     *  Gets the power supply driver.
     *
     *  @return  The driver
     */
    public PowerSupplyDriver getPowerDriver()
    {
        return psd;
    }


    /**
     *  Gets the maximum channel number.
     *
     *  @return  The maximum channel number
     */
    public int getMaxChannel()
    {
        return maxChan;
    }

   
    /**
     *  Gets the connection type.
     *
     *  @return  The connection type string
     */
    public ConnType getConnType()
    {
        return connType;
    }


    /**
     *  Gets the device ID.
     *
     *  @return  The device identification string
     */
    public String getDevcId()
    {
        return devcId;
    }


    /**
     *  Gets the device parameter.
     *
     *  @return  The device parameter
     */
    public int getDevcParm()
    {
        return devcParm;
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (connType == null) {
            ErrorUtils.reportConfigError(LOG, name, "connType", "is missing");
        }
        if (connType == ConnType.NET && (options & NETWORK_CONN_INVALID) != 0) {
            ErrorUtils.reportConfigError(LOG, name, "connType", "is invalid");
        }
        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        fullName = devcName + (devcId.isEmpty() ? "" : " (" + devcId + ")");
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            psd.open(connType, devcId, devcParm);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            psd.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}:{1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  ch  The channel to check
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        String type = ch.getTypeStr();
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            ErrorUtils.reportChannelError(LOG, ch.getPath(), "type", type);
        }
        int hwChan = ch.getHwChan();
        if (hwChan < minChan || hwChan > maxChan) {
            ErrorUtils.reportChannelError(LOG, ch.getPath(), "hw channel number", hwChan);
        }
        
        return new int[]{mType, 0};
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  ch  The channel to read
     *  @return  The read value
     */
    @Override
    protected double readChannel(Channel ch)
    {
        double value = super.readChannel(ch);
        if (isOnline()) {
            int type = ch.getType();
            int hwChan = ch.getHwChan();
            try {
                value = (type == MON_TYPE_VOLTAGE) ? psd.readVoltage(hwChan) : psd.readCurrent(hwChan);
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading from {0}: {1}", new Object[]{fullName, e});
                setOnline(false);
            }
        }

        return value;
    }


    /**
     *  Writes the voltage to the hardware.
     *
     *  @param  voltage  The voltage to set
     *  @param  hwChan   The hardware channel number
     *  @throws  PowerException
     */
    protected void writeVoltage(double voltage, int hwChan) throws PowerException
    {
        try {
            psd.setVoltage(voltage, hwChan);
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Reads the voltage from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The read voltage
     *  @throws  PowerException
     */
    protected double readVoltage(int hwChan) throws PowerException
    {
        try {
            return psd.readVoltage(hwChan);
        }
        catch (DriverException e) {
            handleException(e);
            return 0;
        }
    }


    /**
     *  Writes the current to the hardware.
     *
     *  @param  current  The current to set
     *  @param  hwChan   The hardware channel number
     *  @throws  PowerException
     */
    protected void writeCurrent(double current, int hwChan) throws PowerException
    {
        try {
            psd.setCurrent(current, hwChan);
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Reads the current from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The read current
     *  @throws  PowerException
     */
    protected double readCurrent(int hwChan) throws PowerException
    {
        try {
            return psd.readCurrent(hwChan);
        }
        catch (DriverException e) {
            handleException(e);
            return 0;
        }
    }


    /**
     *  Writes the power-on delay to the hardware.
     *
     *  @param  onDelay  The delay to set
     *  @param  hwChan   The hardware channel number
     *  @throws  PowerException
     */
    protected void writeOnDelay(double onDelay, int hwChan) throws PowerException
    {
        try {
            psd.setOnDelay(onDelay, hwChan);
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Writes the power-off delay to the hardware.
     *
     *  @param  offDelay  The delay to set
     *  @param  hwChan    The hardware channel number
     *  @throws  PowerException
     */
    protected void writeOffDelay(double offDelay, int hwChan) throws PowerException
    {
        try {
            psd.setOffDelay(offDelay, hwChan);
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Writes the power output state to the hardware.
     *
     *  @param  value   The output state to set
     *  @param  hwChan  The hardware channel number
     *  @throws  PowerException
     */
    protected void writeOutput(boolean value, int hwChan) throws PowerException
    {
        try {
            psd.setOutput(value, hwChan);
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Reads the power output state from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The power output state, or null if device is offline
     */
    protected Boolean readOutput(int hwChan)
    {
        if (!isOnline()) return null;
        try {
            return psd.getOutput(hwChan);
        }
        catch (DriverException e) {
            try {
                handleException(e);
            }
            catch (PowerException pe) {
            }
            return null;
        }
    }


    /**
     *  Writes a group of voltages to the hardware.
     *
     *  May be overridden for devices with this capability
     *
     *  @throws  DriverException
     */
    protected void writeVoltageGroup() throws DriverException
    {
    }


    /**
     *  Reads a group of voltages from the hardware.
     *
     *  May be overridden for devices with this capability
     */
    protected void readVoltageGroup()
    {
    }


    /**
     *  Writes a group of currents to the hardware.
     *
     *  May be overridden for devices with this capability
     *
     *  @throws  DriverException
     */
    protected void writeCurrentGroup() throws DriverException
    {
    }


    /**
     *  Reads a group of currents from the hardware.
     *
     *  May be overridden for devices with this capability
     */
    protected void readCurrentGroup()
    {
    }


    /**
     *  Writes a group of power-on delays to the hardware.
     *
     *  May be overridden for devices with this capability
     *
     *  @throws  DriverException
     */
    protected void writeOnDelayGroup() throws DriverException
    {
    }


    /**
     *  Writes a group of power-off delays to the hardware.
     *
     *  May be overridden for devices with this capability
     *
     *  @throws  DriverException
     */
    protected void writeOffDelayGroup() throws DriverException
    {
    }


    /**
     *  Writes a group of power output states to the hardware.
     *
     *  May be overridden for devices with this capability
     *
     *  @throws  DriverException
     */
    protected void writeOutputGroup() throws DriverException
    {
    }


    /**
     *  Reads a group of power output states from the hardware.
     *
     *  May be overridden for devices with this capability
     */
    protected void readOutputGroup()
    {
    }


    /**
     *  Handle driver exceptions.
     * 
     * @param e The exception
     * @throws PowerException 
     */
    protected void handleException(DriverException e) throws PowerException
    {
        if (e instanceof DriverTimeoutException) {
            LOG.log(Level.SEVERE, "Timeout while accessing power supply: {0}", fullName);
            setOnline(false);
        }
        throw new PowerException(e.toString());
    }

}
