package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;
import org.lsst.ccs.subsystem.power.constants.PathFinderQuadBoxSwitches;
import org.lsst.ccs.subsystem.power.constants.SwitchState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.data.PathFinderQuadBoxState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the Quad Box control subsystem.
 *
 *  @author Owen Saxton
 */
public class PathFinderQuadBox extends Subsystem implements HasLifecycle {

    static class DummyDevice implements SwitchControl {
        @Override
        public int getSwitchDevice() { return 0; }
        @Override
        public Boolean isSwitchOn(int sw) { return null; }
        @Override
        public void switchOn(int sw) {}
        @Override
        public void switchOff(int sw) {}
    }

    private static final Map<Integer, String> typeMap = new HashMap<>();
    static {
        typeMap.put(PathFinderQuadBoxSwitches.DEVC_BFR, "BFR");
        typeMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_5V, "5V PDU");
        typeMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_24VC, "24V clean PDU");
        typeMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_24VD, "24V dirty PDU");
        typeMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_48V, "48V PDU");
	//        typeMap.put(PathFinderQuadBoxSwitches.DEVC_REB_BULK, "REB bulk PS");
    }

    private static final Logger LOG = Logger.getLogger(PathFinderQuadBox.class.getName());
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService aps;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService apts;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<SwitchControl> swDeviceList = new ArrayList<>();

    private final SwitchControl[] swDevices = new SwitchControl[PathFinderQuadBoxSwitches.NUM_DEVICES];
    private final PathFinderQuadBoxState qbState = new PathFinderQuadBoxState();


    /**
     *  Constructor.
     */
    public PathFinderQuadBox() {
        super("pathfinderquadbox", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a Quad Box controller.
        aps.setAgentProperty(PowerAgentProperties.PATHFINDER_QUAD_BOX_AGENT, getClass().getCanonicalName());

        // Checks power devices and control channels
        boolean error = false;
        for (SwitchControl sd : swDeviceList) {
	    LOG.error("sd = " + sd);
            swDevices[sd.getSwitchDevice()] = sd;
        }
        for (int j = 0; j < swDevices.length; j++) {
            if (swDevices[j] == null) {
                LOG.error("The " + typeMap.get(j) + " device has not been specified");
                //swDevices[j] = new DummyDevice();
                error = true;
            }
        }
        if (error) {
            throw new RuntimeException("Fatal initialization error");
        }
    }


    /**
     *  Starts the subsystem.
     *
     */
    @Override
    public void postStart()
    {
        updateState();
        publishState();        // For any GUIs
        LOG.info("PathFinder Quad Box subsystem started");
    }


    /**
     *  Performs periodic power data broadcast.
     */
    public void periodicBroadcast()
    {
        /*
        if (pwrDevices.isEmpty()) return;
        ArrayList<PowerChanState> pState = new ArrayList<>();
        for (PowerDevice pDevc : pwrDevices.values()) {
            try {
                pState.addAll(pDevc.getPowerState());
            }
            catch (Exception e) {
            }
        }
        KeyValueData kvd = new KeyValueData(PowerChanState.KEY, pState);
        subsys.publishSubsystemDataOnStatusBus(kvd);
        */
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a switch")
    public void setSwitchOn(@Argument(description="The switch number") int sw,
                            @Argument(description="Whether to turn on") boolean on) throws PowerException
    {
        int devNum = sw >> 8;
        int swNum = (byte)sw;  // Sign extend byte value
        if (devNum < 0 || devNum >= PathFinderQuadBoxSwitches.NUM_DEVICES || swNum >= PathFinderQuadBoxSwitches.NUM_SWITCHES[devNum]) {
            throw new PowerException("Invalid switch number: " + sw);
        }
        if (on) {
            swDevices[devNum].switchOn(swNum);
        }
        else {
            swDevices[devNum].switchOff(swNum);
        }
        updateState();
        publishState();
    }


    /**
     *  Turns a named switch on or off.
     *
     *  @param  name  The switch name.
     *  @param  on  Whether to turn on or off
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description="The switch name") String name,
                                 @Argument(description="Whether to turn on") boolean on) throws PowerException
    {
        Integer sw = PathFinderQuadBoxSwitches.switchNameToId.get(name);
        if (sw == null) {
            throw new PowerException("Invalid switch name: " + name);
        }
        setSwitchOn(sw, on);
    }


    /**
     *  Gets the list of switch names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of switch names")
    public List<String> getSwitchNames()
    {
        return new ArrayList<>(PathFinderQuadBoxSwitches.switchNameToId.keySet());
    }


    /**
     *  Sets the update (tick) period.
     *
     *  @param  period  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)")
                                int period)
    {
        setTickPeriod(period);
        publishState();    // Must do this for GUI
    }


    /**
     *  Gets the full quad box state.
     *
     *  @return  The full quad box state
     *  @throws  PowerException
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public PathFinderQuadBoxState getFullState() throws PowerException
    {
        updateState();
        qbState.setTickMillis(getTickPeriod());
        return qbState;
    }


    /**
     *  Updates the quad box state.
     */
    private void updateState()
    {
        for (int devNum = 0; devNum < PathFinderQuadBoxSwitches.NUM_DEVICES; devNum++) {
	    if (devNum == PathFinderQuadBoxSwitches.DEVC_REB_BULK) continue;
            for (int swNum = -1; swNum < PathFinderQuadBoxSwitches.NUM_SWITCHES[devNum]; swNum++) {
		LOG.error("sd = " + swDevices[devNum] + " , swnum = " + swNum);
                Boolean st = swDevices[devNum].isSwitchOn(swNum);
                SwitchState ss = st == null ? SwitchState.OFFLINE : st ? SwitchState.ON : SwitchState.OFF;
                qbState.setSwitchState(devNum, swNum, ss);
            }
        }
    }


    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        apts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the tick period
     */
    private int getTickPeriod()
    {
        return (int)apts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }
    

    /**
     *  Publishes the state of the quad box system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        qbState.setTickMillis(getTickPeriod());
        KeyValueData kvd = new KeyValueData(PathFinderQuadBoxState.KEY, qbState);
        publishSubsystemDataOnStatusBus(kvd);
    }    

}
