package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;
import org.lsst.ccs.subsystem.power.constants.ComCamQuadBoxSwitches;
import org.lsst.ccs.subsystem.power.constants.SwitchState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.data.ComCamQuadBoxState;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.subsystem.common.actions.ComCamSharedVacState;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the Quad Box control subsystem.
 *
 *  @author CCS Team
 */
public class ComCamQuadBox extends Subsystem implements HasLifecycle, AgentPresenceListener, StatusMessageListener {

    static class DummyDevice implements SwitchControl {
        @Override
        public int getSwitchDevice() { return 0; }
        @Override
        public Boolean isSwitchOn(int sw) { return null; }
        @Override
        public void switchOn(int sw) {}
        @Override
        public void switchOff(int sw) {}
    }

    private static final Map<Integer, String> typeMap = new HashMap<>();
    static {
        typeMap.put(ComCamQuadBoxSwitches.DEVC_BFR, "BFR");
        typeMap.put(ComCamQuadBoxSwitches.DEVC_PDU_5V, "5V PDU");
        typeMap.put(ComCamQuadBoxSwitches.DEVC_PDU_24VC, "24V clean PDU");
        typeMap.put(ComCamQuadBoxSwitches.DEVC_PDU_24VD, "24V dirty PDU");
        typeMap.put(ComCamQuadBoxSwitches.DEVC_PDU_48V, "48V PDU");
	//        typeMap.put(ComCamQuadBoxSwitches.DEVC_REB_BULK, "REB bulk PS");
    }

    private static final Logger LOG = Logger.getLogger(ComCamQuadBox.class.getName());
    
    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService aps;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService apts;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<SwitchControl> swDeviceList = new ArrayList<>();

    private final SwitchControl[] swDevices = new SwitchControl[ComCamQuadBoxSwitches.NUM_DEVICES];
    private final ComCamQuadBoxState qbState = new ComCamQuadBoxState();

    private ComCamSharedVacState comCamSharedVacState = new ComCamSharedVacState();



    /**
     *  Constructor.
     */
    public ComCamQuadBox() {
        super("comcamquadbox", AgentInfo.AgentType.WORKER);
    }



    /**
     *  Subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a Quad Box controller.
        aps.setAgentProperty(PowerAgentProperties.COMCAM_QUAD_BOX_AGENT, getClass().getCanonicalName());


        // Add an agent present listener
        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);


        // Checks power devices and control channels
        boolean error = false;
        for (SwitchControl sd : swDeviceList) {
	    if (sd.getSwitchDevice() == ComCamQuadBoxSwitches.DEVC_REB_BULK) continue;

            swDevices[sd.getSwitchDevice()] = sd;
        }
        for (int j = 0; j < swDevices.length; j++) {
            if (swDevices[j] == null) {
                LOG.error("The " + typeMap.get(j) + " device has not been specified");
                //swDevices[j] = new DummyDevice();
                error = true;
            }
        }
        if (error) {
            throw new RuntimeException("Fatal initialization error");
        }
    }


    /**
     *  Starts the subsystem.
     *
     */
    @Override
    public void postStart()
    {
        updateState();
        publishState();        // For any GUIs
        LOG.info("Quad Box subsystem started");
    }



    /**
     *  Performs periodic power data broadcast.
     */
    public void periodicBroadcast()
    {
        /*
        if (pwrDevices.isEmpty()) return;
        ArrayList<PowerChanState> pState = new ArrayList<>();
        for (PowerDevice pDevc : pwrDevices.values()) {
            try {
                pState.addAll(pDevc.getPowerState());
            }
            catch (Exception e) {
            }
        }
        KeyValueData kvd = new KeyValueData(PowerChanState.KEY, pState);
        subsys.publishSubsystemDataOnStatusBus(kvd);
        */
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a switch")
    public void setSwitchOn(@Argument(description="The switch number") int sw,
                            @Argument(description="Whether to turn on") boolean on) throws PowerException
    {
        int devNum = sw >> 8;
        int swNum = (byte)sw;  // Sign extend byte value
        if (devNum < 0 || devNum >= ComCamQuadBoxSwitches.NUM_DEVICES || swNum >= ComCamQuadBoxSwitches.NUM_SWITCHES[devNum]) {
            throw new PowerException("Invalid switch number: " + sw);
        }

	LOG.info("request to flip switch = " + sw + " *********************************");

	if (sw != ComCamQuadBoxSwitches.SW_RELAY_AUX_CMC_DPP_01) {
 
	    if (on) {
		swDevices[devNum].switchOn(swNum);
	    }
	    else {
		swDevices[devNum].switchOff(swNum);
	    }

	} else { //HN special handling of the vat valve
	    Boolean validVacInfo =  comCamSharedVacState.getVacuumState()!= ComCamSharedVacState.VacuumState.UNKNOWN;
	    Boolean forceVATclosed = comCamSharedVacState.isForceValveClosing();
	    Boolean VATStateOpenRequest = comCamSharedVacState.isValveOpenStateRequested();
	    if (validVacInfo) {
		if (forceVATclosed) {
		    LOG.info("Forcing VAT valve closed!");
		    swDevices[devNum].switchOff(swNum);
		} else if (VATStateOpenRequest == true) {
		    LOG.info("Opening of the VAT valve requested.");
		    swDevices[devNum].switchOn(swNum);
		} else if (VATStateOpenRequest == false) {
		    LOG.info("Closing of the VAT valve requested.");
		    swDevices[devNum].switchOff(swNum);
		}
	    }
	}
        updateState();
        publishState();
    }


    /**
     *  Turns a named switch on or off.
     *
     *  @param  name  The switch name.
     *  @param  on  Whether to turn on or off
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description="The switch name") String name,
                                 @Argument(description="Whether to turn on") boolean on) throws PowerException
    {
        Integer sw = ComCamQuadBoxSwitches.switchNameToId.get(name);
        if (sw == null) {
            throw new PowerException("Invalid switch name: " + name);
        }
        setSwitchOn(sw, on);
    }


    /**
     *  Gets the list of switch names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of switch names")
    public List<String> getSwitchNames()
    {
        return new ArrayList<>(ComCamQuadBoxSwitches.switchNameToId.keySet());
    }


    /**
     *  Sets the update (tick) period.
     *
     *  @param  period  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)")
                                int period)
    {
        setTickPeriod(period);
        publishState();    // Must do this for GUI
    }


    /**
     *  Gets the full quad box state.
     *
     *  @return  The full quad box state
     *  @throws  PowerException
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public ComCamQuadBoxState getFullState() throws PowerException
    {
        updateState();
        qbState.setTickMillis(getTickPeriod());
        return qbState;
    }


    /**
     *  Updates the quad box state.
     */
    private void updateState()
    {
        for (int devNum = 0; devNum < ComCamQuadBoxSwitches.NUM_DEVICES; devNum++) {
            for (int swNum = -1; swNum < ComCamQuadBoxSwitches.NUM_SWITCHES[devNum]; swNum++) {
                Boolean st = swDevices[devNum].isSwitchOn(swNum);
                SwitchState ss = st == null ? SwitchState.OFFLINE : st ? SwitchState.ON : SwitchState.OFF;
                qbState.setSwitchState(devNum, swNum, ss);
            }
        }
    }


    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        apts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the tick period
     */
    private int getTickPeriod()
    {
        return (int)apts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }
    

    /**
     *  Publishes the state of the quad box system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        qbState.setTickMillis(getTickPeriod());
        KeyValueData kvd = new KeyValueData(ComCamQuadBoxState.KEY, qbState);
        publishSubsystemDataOnStatusBus(kvd);
    }    


    @Override
    public void connected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
	    LOG.info("agent = "+agent);
	    //            if (agent.hasAgentProperty(ComCamSharedVacState.class.getCanonicalName())) {
            if (agent.getName().contains("comcam-vacuum")) {
                String agentName = agent.getName();
                subsys.getMessagingAccess().addStatusMessageListener(this, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName));
            }
        }
    }


	/**
	 *  Listens for the departure of the companion refrigeration subsystem.
	 *
	 *  @param  agents  Agents going absent
	 */
    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(ComCamSharedVacState.class.getCanonicalName()) ) {
                subsys.getMessagingAccess().removeStatusMessageListener(this);
                //refrigColdState = ColdState.UNKNOWN;
            }
        }
    }


	/**
	 *  Handles vacuum status messages.
	 *
	 *  @param  msg  The status message
	 */
    @Override
    public void onStatusMessage(StatusMessage msg) {
	//        comCamSharedVacState = msg.getState().getState(ComCamSharedVacState.class);

	/*      msg.getObject() = KeyValueData{ key=ComCamSharedVacState, value=org.lsst.ccs.subsystem.common\
				 .actions.ComCamSharedVacState@7917c270, timestamp=CCSTimeStamp{utc=2020-02-13T07:20:20.986\
				 568451Z}, type=KeyValueTrendingData } (org.lsst.ccs.subsystem.power.ComCamQuadBox onStatusMessage)*/

	/*
	    if (msg.getObject() instanceof ComCamSharedVacState) {
		comCamSharedVacState = (ComCamSharedVacState) msg.getObject();
		LOG.info("vacuum state = "+comCamSharedVacState.getVacuumState());
	    } else {
		LOG.info("object not an instance of ComCamSharedVacState -- msg.getObject() = "+msg.getObject());
		return;
	    }
	*/

	try {


	    StatusSubsystemData sd = (StatusSubsystemData)msg;
	
	
	    //	    LOG.info("heard a message ... checking ... key = [ " + sd.getDataKey().toString() + " ] "); 
	    if (!sd.getDataKey().equals(ComCamSharedVacState.KEY)) return;

	    comCamSharedVacState = (ComCamSharedVacState)((KeyValueData)sd.getSubsystemData()).getValue();
	    LOG.info("vacuum state = "+comCamSharedVacState.getVacuumState() +
		     " | vacuum state OK = " + (comCamSharedVacState.getVacuumState()!= ComCamSharedVacState.VacuumState.UNKNOWN) +
		     " | forceValveClosing = " + comCamSharedVacState.isForceValveClosing() +
		     " | valveOpenStateRequested = "+comCamSharedVacState.isValveOpenStateRequested());


	} catch (Exception e) {

	    LOG.fine("not StatusSubsystemData :" + e);

	}

	int sw = ComCamQuadBoxSwitches.SW_RELAY_AUX_CMC_DPP_01; // VAT valve
        int devNum = sw >> 8;
        int swNum = (byte)sw;  // Sign extend byte value

	Boolean st = swDevices[devNum].isSwitchOn(swNum);

	if (st == true && comCamSharedVacState.isForceValveClosing()) {
	    try {
		setSwitchOn(sw, false) ;
	    } catch (PowerException e) {
		LOG.error("Failed to set switch to force the VAT valve closed! - " + e);
	    }
	} else if (!comCamSharedVacState.isForceValveClosing()) {
	    try {
		if (st == false && comCamSharedVacState.isValveOpenStateRequested() == true) setSwitchOn(sw, true) ;
		if (st == true && comCamSharedVacState.isValveOpenStateRequested() == false) setSwitchOn(sw, false) ;
	    } catch (PowerException e) {
		LOG.error("Failed to set VAT valve switch! - " + e);
	    }
	}
	
    }



}
