package org.lsst.ccs.subsystem.power;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.drivers.auxelex.RebPS;
import org.lsst.ccs.drivers.auxelex.SrpException;
import org.lsst.ccs.drivers.auxelex.Srp;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorUpdateTask;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.constants.RebPsEnum;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.states.PowerSupplyState;
import org.lsst.ccs.subsystem.power.states.RebDPhiState;
import org.lsst.ccs.subsystem.power.states.RebHvBiasState;
import org.lsst.ccs.subsystem.power.states.RebPowerState;

/**
 *  Interface to a REB power supply board.
 *
 *  @author Owen Saxton
 */
public class RebPsDevice extends Device {

    /**
     *  Inner class for holding power supply parameters.
     */
    static class SeqParams {

        final int    psNum;
        final int    rebOff;
        final double minGood;
        final double maxGood;

        SeqParams(int psNum, double minGood, double maxGood)
        {
            this(psNum, 0, minGood, maxGood);
        }

        SeqParams(int psNum, int rebOff, double minGood, double maxGood)
        {
            this.psNum = psNum;
            this.rebOff = rebOff;
            this.minGood = minGood;
            this.maxGood = maxGood;
        }

    }

    /**
     *  Interface for open/close event listener.
     */
    public interface Event {

        public void opened();

        public void closed();

    }

    /**
     *  Public constants.
     */
    public enum PsName {
        DIGITAL(RebPS.PS_DIGITAL),
        ANALOG(RebPS.PS_ANALOG),
        CLOCKHI(RebPS.PS_CLK_HIGH),
        CLOCKLO(RebPS.PS_CLK_LOW),
        OD(RebPS.PS_OD),
        HEATER(RebPS.PS_HEATER),
        DPHI(RebPS.PS_DPHI);

        int value;

        PsName(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }
    }

    public enum ChanName {
        VBEFLDO(RebPS.CHAN_VOLT_BEF_LDO),
        VAFTLDO(RebPS.CHAN_VOLT_AFT_LDO),
        IBEFLDO(RebPS.CHAN_CURR_BEF_LDO);

        int value;

        ChanName(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }
    }

    /**
     *  Private constants.
     */
    private static final Logger LOG = Logger.getLogger(RebPsDevice.class.getName());

    private static final String
        UNKN_PSID = "PS-XX";

    private static final int
        PS_POWER = 255;

    private static final double
        LIM_LOW_DIGITAL = 4.5,
        LIM_HIGH_DIGITAL = 6.0,
        LIM_LOW_ANALOG = 6.5,
        LIM_HIGH_ANALOG = 8.5,
        LIM_LOW_CLKL_SR = 12.0,
        LIM_HIGH_CLKL_SR = 17.0,
        LIM_LOW_CLKL_CR = 10.0,
        LIM_HIGH_CLKL_CR = 12.0,
        LIM_LOW_CLKH_SR = 15.0,
        LIM_HIGH_CLKH_SR = 17.0,
        LIM_LOW_CLKH_CR = 10.0,
        LIM_HIGH_CLKH_CR = 12.0,
        LIM_LOW_OD_SR = 35.0,
        LIM_HIGH_OD_SR = 42.0,
        LIM_LOW_OD_CR = 35.0,
        LIM_HIGH_OD_CR = 37.0,
        LIM_LOW_HEATER_SR = 7.0,
        LIM_HIGH_HEATER_SR = 13.0,
        LIM_LOW_HEATER_CR = 5.0,
        LIM_HIGH_HEATER_CR = 7.0,
        LIM_LOW_DPHI = 4.0,
        LIM_HIGH_DPHI = 13.0;

    private static final Map<String, Integer> psMap = new HashMap<>();
    static {
        psMap.put("DIGITAL", RebPS.PS_DIGITAL);
        psMap.put("ANALOG", RebPS.PS_ANALOG);
        psMap.put("CLOCKHI", RebPS.PS_CLK_HIGH);
        psMap.put("CLOCKLO", RebPS.PS_CLK_LOW);
        psMap.put("DPHI", RebPS.PS_DPHI);
        psMap.put("HEATER", RebPS.PS_HEATER);
        psMap.put("HVBIAS", RebPS.PS_HV_BIAS);
        psMap.put("OD", RebPS.PS_OD);
        psMap.put("POWER", PS_POWER);
    }

    private static final SeqParams[]
        onSeqSR02 = {new SeqParams(RebPS.PS_DIGITAL, LIM_LOW_DIGITAL, LIM_HIGH_DIGITAL),
                     new SeqParams(RebPS.PS_ANALOG, LIM_LOW_ANALOG, LIM_HIGH_ANALOG),
                     new SeqParams(RebPS.PS_CLK_LOW, LIM_LOW_CLKL_SR, LIM_HIGH_CLKL_SR),
                     new SeqParams(RebPS.PS_CLK_HIGH, LIM_LOW_CLKH_SR, LIM_HIGH_CLKH_SR),
                     new SeqParams(RebPS.PS_OD, LIM_LOW_OD_SR, LIM_HIGH_OD_SR),
                     new SeqParams(RebPS.PS_HEATER, LIM_LOW_HEATER_SR, LIM_HIGH_HEATER_SR)};

    private static final SeqParams[]
        onSeqSR1 = {new SeqParams(RebPS.PS_DIGITAL, LIM_LOW_DIGITAL, LIM_HIGH_DIGITAL),
                    new SeqParams(RebPS.PS_ANALOG, LIM_LOW_ANALOG, LIM_HIGH_ANALOG),
                    new SeqParams(RebPS.PS_CLK_LOW, LIM_LOW_CLKL_SR, LIM_HIGH_CLKL_SR),
                    new SeqParams(RebPS.PS_CLK_HIGH, LIM_LOW_CLKH_SR, LIM_HIGH_CLKH_SR),
                    new SeqParams(RebPS.PS_OD, LIM_LOW_OD_SR, LIM_HIGH_OD_SR)};

    private static final SeqParams[]
        onSeqCR0 = {new SeqParams(RebPS.PS_DIGITAL, LIM_LOW_DIGITAL, LIM_HIGH_DIGITAL),
                    new SeqParams(RebPS.PS_ANALOG, LIM_LOW_ANALOG, LIM_HIGH_ANALOG),
                    new SeqParams(RebPS.PS_CLK_LOW, LIM_LOW_CLKL_CR, LIM_HIGH_CLKL_CR),
                    new SeqParams(RebPS.PS_CLK_HIGH, LIM_LOW_CLKH_CR, LIM_HIGH_CLKH_CR),
                    new SeqParams(RebPS.PS_OD, LIM_LOW_OD_CR, LIM_HIGH_OD_CR),
                    new SeqParams(RebPS.PS_DIGITAL, 2, LIM_LOW_HEATER_CR, LIM_HIGH_HEATER_CR)};

    private static final SeqParams[]
        onSeqCR1 = {new SeqParams(RebPS.PS_DIGITAL, LIM_LOW_DIGITAL, LIM_HIGH_DIGITAL),
                    new SeqParams(RebPS.PS_ANALOG, LIM_LOW_ANALOG, LIM_HIGH_ANALOG),
                    new SeqParams(RebPS.PS_CLK_LOW, LIM_LOW_CLKL_CR, LIM_HIGH_CLKL_CR),
                    new SeqParams(RebPS.PS_CLK_HIGH, LIM_LOW_CLKH_CR, LIM_HIGH_CLKH_CR),
                    new SeqParams(RebPS.PS_OD, LIM_LOW_OD_CR, LIM_HIGH_OD_CR)};
/*
    private static final SeqParams[]
        onSeqCRL = {new SeqParams(RebPs.PS_DIGITAL, 5.0, 7.0),
                    new SeqParams(RebPs.PS_ANALOG, 5.0, 7.0)};

    private static final int[]
        offSeqSR = {RebPs.PS_HV_BIAS, RebPs.PS_HEATER, RebPs.PS_OD, RebPs.PS_CLK_HIGH,
                    RebPs.PS_CLK_LOW, RebPs.PS_ANALOG, RebPs.PS_DIGITAL};

    private static final int[]
        offSeqCR = {RebPs.PS_HV_BIAS, RebPs.PS_DPHI, RebPs.PS_OD, RebPs.PS_CLK_HIGH,
                    RebPs.PS_CLK_LOW, RebPs.PS_ANALOG, RebPs.PS_DIGITAL};

    private static final int[]
        offSeqCRL = {RebPs.PS_ANALOG, RebPs.PS_DIGITAL};
*/

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name="ipAddr", category=Power.POWER, isFinal=true)
    private volatile String ipAddr;

    @ConfigurationParameter(name="nPowerOnPub", category=Power.POWER, description = "Number of publications after a power on sequence")
    private volatile int nPowerOnPub = 15;

    @ConfigurationParameter( isReadOnly = true, maxLength = 6, description = "List of Rebs powered by this PS")
    private volatile List<String> rebs = new ArrayList<>();
    private List<String> rebsList;

    @ConfigurationParameter(name="switchName", category="build")
    private volatile String switchName;
    
    private int psType = RebPS.TYPE_UNKNOWN;
    private final RebPS ps = new RebPS(null);
    private int psVersion = RebPS.VERSION_PROD;
    private Event listener;
    private String psId = UNKN_PSID;
    private Properties props;
    private boolean isSimulated;
    private double[] power = new double[RebPS.MAX_REBS];

    //Variable that control how many consecutive caught exceptions are needed
    //to put the Device in the Offline state
    //This is the exception counter
    private volatile int caughtExceptionNumber = 0;
    //This is the number of consecutive exception that will put the device offline
    private volatile int numberOfConsectiveExceptionsToOffline = 1;
    
    private boolean initError = false;
    private final RebTrippedState noTrip = new RebTrippedState(false);
    
    //This state bundle contains the state information for this Device
    //and for all the rebs controlled by this rebPsDevice
    private final StateBundle deviceAndRebsState = new StateBundle();
    
    //Is this needed? Can we rely on the StateBundle synchronization?
    private final Object stateLock = new Object();

    //Is this a corner raft power supply?
    private boolean isCornerRaft = false;
    
    //The index of the reb currently being powered on.
    //A negative number means no rebs are being powered on.
    private volatile int rebBeingPoweredOn = -1;

    //The last read for the tripped state.
    //We use it to skip processing the state if the value is unchanged.
    private volatile int lastFail = Integer.MIN_VALUE;
        
    //A List containing the RebPowerState; this list is used exclusively to control
    //the log level of InvalidAddressExceptions for reb specific channels
    //We keep this information separate from the StateBundle variable deviceAndRebsState
    //since it's updated more frequently to account for the reb being powered off.
    private final List<RebPowerState> rebPowerStates = new CopyOnWriteArrayList<>();

    private int simulatedPsType = RebPS.TYPE_UNKNOWN;
    
    @LookupPath
    protected String psPath;    
    
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS) 
    private List<RebPsBoardAlarm> boardTemperatureAlarms = new ArrayList<>();
    
    private List<MonitorUpdateTask> tasksForDevice = null;

    
    @Override
    public void build() {
        super.build();
        if ( rebsList != null ) {
            rebs.addAll(rebsList);
        } else {
            LOG.log(Level.WARNING, "The list of Rebs has not been provided; read-only configuration parameter \"rebs\" will be empty.");            
        }
    }
    
    @Override
    protected void initDevice()
    {
        tasksForDevice = getDeviceMonitorUpdateTasks();
        
        isSimulated = RunMode.isSimulation();
        String fileName = "psid.properties";
        if (BootstrapResourceUtils.getBootstrapResource(fileName) == null) {
            LOG.log(Level.WARNING, "PS ID properties file ({0}) not found", fileName);
        }
        props = BootstrapResourceUtils.getBootstrapProperties(fileName);
        if (ipAddr == null && !isSimulated) {
            ErrorUtils.reportConfigError(LOG, name, "ipAddr", "is missing");
        }
        fullName = "REB PS board (" + (isSimulated ? "simulated" : ipAddr) + ")";
        ps.setLogName(fullName);        
    }


    @Override
    protected void initialize()
    {
        try {
            ps.open(isSimulated ? "" : ipAddr);            
            ps.configTemperature();
            psVersion = ps.getVersion();
            psType = ps.getType();
            generatePsId();
            caughtExceptionNumber = 0;
            initSensors();
            if (listener != null) {
                listener.opened();
            }
            String buildStamp = ps.getBuildStamp();
            Srp.BoardType boardType = ps.getBoardType();
            long serialNumber = ps.getSerialNo();
            int fwVersion = ps.getFwVersion();
            LOG.log(Level.INFO, () -> String.format("Connected to %s with buildstamp=\"%s\" boardType=%s serial=%x fwVersion=%d psVersion=%d psType=%d",
                    fullName, buildStamp, boardType, serialNumber, fwVersion, psVersion, psType));
            
            initError = false;
            
            //On successful initialization we reset the temperature alarms
            for ( Alarm a : boardTemperatureAlarms ) {
                a.clearState();
            }
            setOnline(true);            
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                ps.close();
            }
            catch (DriverException ce) {
                // Will happen if open failed
            }
        }
    }

    @Override
    protected void setOnline(boolean online) {
        initializeState(online);
        super.setOnline(online); //To change body of generated methods, choose Tools | Templates.
    }
    
    
    private void initializeState(boolean online) {

        synchronized(stateLock) {
            deviceAndRebsState.setComponentState(path, online ? PowerSupplyState.ON : PowerSupplyState.OFF);

            for (String path : rebs) {
                //We don't assign a state to the Heater channels attached to this RebPS
                if (path.contains("Heater") || path.isEmpty()) {
                    continue;
                }
                deviceAndRebsState.setComponentState(path, RebPowerState.UNKNOWN);
                deviceAndRebsState.setComponentState(path, RebHvBiasState.UNKNOWN);
                if (isCornerRaft) {
                    deviceAndRebsState.setComponentState(path, RebDPhiState.UNKNOWN);
                }
            }
        }
        
        //Reset the local state variables.
        lastFail = Integer.MIN_VALUE;        
        
        rebPowerStates.clear();        
        rebPowerStates.addAll(Collections.nCopies(6, RebPowerState.UNKNOWN));
        
    }
    
    //Update the trip state
    private void updateTripState(int rebNumber, RebTrippedState trippedState) {
        synchronized(stateLock) {
            String rebPath = rebs.get(rebNumber);
            //We don't assign a state to the Heater channels attached to this RebPS
            if (path.contains("Heater") || path.isEmpty()) {
                return;
            }
            RebPowerState rebCurrentState = (RebPowerState)deviceAndRebsState.getComponentState(rebPath, RebPowerState.class);
            if ( trippedState.hasTripped() ) {
                //Here is a brand new trip!
                if ( rebCurrentState != RebPowerState.TRIPPED ) {
                    deviceAndRebsState.setComponentState(rebPath, RebPowerState.TRIPPED);            
                    //ADD SOME LOGGING??
                    //RAISE AN ALERT?
                }
            } else {
                //And here is the clearing of a trip
                if ( rebCurrentState == RebPowerState.TRIPPED ) {
                    deviceAndRebsState.setComponentState(rebPath, RebPowerState.UNKNOWN);            
                }
            }
        }
    }
        

    //Update the state of a reb
    private void updateRebState(int rebNumber, int rebPowerState) {
        synchronized(stateLock) {
            
            String rebPath = rebs.get(rebNumber);
            //We don't assign a state to the Heater channels attached to this RebPS
            if (path.contains("Heater") || path.isEmpty()) {
                return;
            }
            RebPowerState state = RebPowerState.TRIPPED;
            if ( deviceAndRebsState.getComponentState(rebPath, RebPowerState.class) != RebPowerState.TRIPPED ) {
                state = getRebPowerState(rebPowerState);
            }
            deviceAndRebsState.setComponentState(rebPath, state);
            rebPowerStates.set(rebNumber, state);
            deviceAndRebsState.setComponentState(rebPath, getRebHvBiasState(rebPowerState));

            if (isCornerRaft) {
                deviceAndRebsState.setComponentState(rebPath, getRebDphiState(rebPowerState));
            }
        }
    }
    
    @Override
    protected void close()
    {
        if (listener != null) {
            listener.closed();
        }
        psId = UNKN_PSID;
        try {
            ps.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}:{1}", new Object[]{fullName, e});
        }
    }


    /**
     * Set the timeout for the underlying driver.
     * @param timeout Read timeout in ms.
     */
    public void setReadTimeout(long timeout) {
        LOG.log(Level.FINE, "Updating read timeout for {0} to {1} ms", new Object[]{fullName, timeout});
        ps.setReadTimeout((int)timeout);
    }    
    
    /**
     * Set the number of consecutive exceptions that will put the device offline.
     * @param nExceptions The number of consecutive exceptions.
     */
    public void setConsecutiveExceptiionsToOffline(int nExceptions) {
        LOG.log(Level.FINE, "Updating the number of consecutive exceptions to go Offline for Device {0} to {1}", new Object[]{fullName, nExceptions});
        this.numberOfConsectiveExceptionsToOffline = nExceptions;
    }    

    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer rebNum = null, psNum = null;
        String[] flds = type.split(":");
        if (flds.length == 1) {
            if (flds[0].toUpperCase().equals("TEMP")) {
                psNum = -1;
                rebNum = 0;
            } else if (flds[0].toUpperCase().equals("STATE")) {
                psNum = -2;
                rebNum = 0;                
            }
        }
        else if (flds.length == 2) {
            try {
                rebNum = Integer.valueOf(flds[0]);
                psNum = psMap.get(flds[1].toUpperCase());
            }
            catch (NumberFormatException e) {
            }
        }
        if (psNum == null) {
            String text = rebNum == null ? "type (REB number)" : "type (PS name)";
            ErrorUtils.reportChannelError(LOG, name, text, type);
        }
        if (psNum >= 0 && psNum != PS_POWER && !RebPS.testChannelNumber(psNum, hwChan)) {
            ErrorUtils.reportChannelError(LOG, name, "hwChan", hwChan);
        }
        
        return new int[]{(rebNum << 8) | psNum, 0};
    }


    @Override
    protected void initChannel(String name, int id, int hwChan, int type, int subtype)
    {
        try {
            if (type == -1) {
                if (hwChan > ps.getNumTemperatures()) {
                    ErrorUtils.reportChannelError(LOG, name, "hwChan", hwChan);
                }
            } else if ( type == -2) {
                if (hwChan < -1 || hwChan > ps.getNumRebs()) {
                    ErrorUtils.reportChannelError(LOG, name, "hwChan", hwChan);
                }                
            }
            else {
                int rebNum = type >> 8, psNum = type & 0xff;
                if (psNum != PS_POWER && !ps.testRebNumber(rebNum)) {
                    ErrorUtils.reportChannelError(LOG, name, "REB number", rebNum);
                }
            }
        }
        catch (Exception e) {
            dropChannel(id);
        }
    }


    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        if (isOnline()) {
            final boolean isTempChan = type == -1;
            final boolean isStateChan = type == -2;
            final int rebNum = type >> 8;
            final int psNum = type & 0xff;
            try {
                if (isTempChan) {
                    if (hwChan == ps.getNumTemperatures()) {
                        value = ps.readFPGATemperature();
                    } else {
                        value = ps.readTemperature(hwChan);
                    }                    
                } else if (isStateChan) {
                    if ( hwChan == -1 ) {
                        int nReb = ps.getNumRebs();
                        int rawValue = ps.getFailureSummary();
                        int convertedValue = rawValue & ((1 << nReb) - 1);
                        LOG.log(Level.FINER, "PS {0} tripped state value: {1} (failure sumary: {2})",new Object[] {path, convertedValue, rawValue});
                        if (convertedValue != lastFail) {
                            //Make sure this is not the first valid value read.
                            if ( lastFail != Integer.MIN_VALUE ) {
                                LOG.log(Level.INFO, "PS {0} tripped state value changed from {1} to {2}",new Object[] {path, lastFail, convertedValue});
                            }
                            lastFail = convertedValue;
                            for (int reb = 0; reb < nReb; reb++) {
                                RebTrippedState currentState;
                                if ((lastFail & (1 << reb)) != 0) {
                                    String tripReason = String.format("REB PS %s tripped: %s", reb, ps.getFailureDetailString(reb));
                                    LOG.severe(tripReason);
                                    currentState = new RebTrippedState(true, tripReason);
                                } else {
                                    currentState = noTrip;
                                }
                                updateTripState(reb, currentState);
                            }
                        }
                        value = convertedValue;
                    } else {
                        int state = ps.getPower(hwChan);
                        updateRebState(hwChan, state);          
                        value = state;
                    }
                }
                else {
                    if (psNum == PS_POWER) {
                        if (rebNum >= RebPS.MAX_REBS) {
                            value = 0.0;
                            for (int j = 0; j < ps.getNumRebs(); j++) {
                                value += power[j];
                            }
                        }
                        else {
                            value = 0.0;
                            for (int psN = 0; psN < RebPS.NUM_PS; psN++) {
                                if (psN == RebPS.PS_HV_BIAS || !ps.isPowerOn(rebNum, psN)) continue;
                                double volts = ps.readChannel(rebNum, psN, RebPS.CHAN_VOLT_AFT_LDO);
                                if (psN == RebPS.PS_CLK_LOW) {
                                    volts = ps.readChannel(rebNum, psN, RebPS.CHAN_VOLT_BEF_LDO) - volts;
                                }
                                value += volts * ps.readChannel(rebNum, psN, RebPS.CHAN_CURR_BEF_LDO);
                            }
                            power[rebNum] = value;
                        }
                    }
                    else if (psNum != RebPS.PS_CLK_LOW || hwChan != RebPS.CHAN_VOLT_AFT_SW
                               || ps.isPowerOn(rebNum, psNum)) {   // Clock low V after switch is meaningless if powered off
                        value = ps.readChannel(rebNum, psNum, hwChan);
                    }
                }
                caughtExceptionNumber = 0;
            }
            catch (DriverException e) {
                Level logLevel = Level.SEVERE;
                Supplier<String> msgSupplier;
                if (psNum == RebPS.PS_OD || psNum == RebPS.PS_HV_BIAS) { // These should always be on
                    //If the reb is being powered on and the exception is SrpException.SrpInvalidRegisterAddressException,
                    //we change the log level to FINER
                    //see https://jira.slac.stanford.edu/browse/LSSTCCSPOWER-120
                    if ( rebBeingPoweredOn == rebNum && e instanceof SrpException.SrpInvalidRegisterAddressException ) {
                        logLevel = Level.FINER;
                    }
                    msgSupplier = () -> String.format("Error reading %s: reb=%d ps=%d", fullName, rebNum, psNum);
                } else if (e instanceof SrpException.SrpInvalidRegisterAddressException) { // expected if ps is off
                    if ( rebPowerStates.get(rebNum) != RebPowerState.ON ) {
                        logLevel = Level.FINER;
                    }
                    msgSupplier = () -> String.format("Error reading %s: reb=%d ps=%d (not powered on?)", fullName, rebNum, psNum);
                } else {
                    msgSupplier = () -> String.format("Error reading %s: reb=%d ps=%d", fullName, rebNum, psNum);
                }
                processCaughtException(e, logLevel, msgSupplier);
            } 
        }

        return value;
    }

    private void processCaughtException(Exception e) {
        processCaughtException(e, Level.SEVERE,null);
    }

    private void processCaughtException(Exception e, Level level, Supplier<String> msgSupplier) {
        boolean countTowardOfflineState = e instanceof DriverTimeoutException || e.getCause() instanceof IOException;
        if ( countTowardOfflineState ) {
            ++caughtExceptionNumber;        
        }
        if ( countTowardOfflineState || msgSupplier != null ) {
            LOG.log(level, e,
                    () -> {
                        String defaultMsg = countTowardOfflineState ? String.format("Caught exception (%s) for %s", caughtExceptionNumber, fullName) : "";
                        if (msgSupplier != null) {
                            if (!defaultMsg.isEmpty()) {
                                defaultMsg += "\n";
                            }
                            defaultMsg += msgSupplier.get();
                        }
                        return defaultMsg;
                    });
        }
        if (caughtExceptionNumber >= numberOfConsectiveExceptionsToOffline) {
            setOnline(false);  // No power to the supply
        }
    }
    

    /**
     *  Checks whether a power supply has tripped.
     *
     *  @return  Whether a trip has occurred
     */
    public boolean checkPsTripped()
    {
        
        if (! isOnline() ) {
            return false;
        }
        try {
            int nReb = ps.getNumRebs();
            int fail = ps.getFailureSummary() & ((1 << nReb) - 1);
            if (fail == lastFail) return false;
            lastFail = fail;
            if (fail == 0) return false;
            for (int reb = 0; reb < nReb; reb++) {
                if ((fail & (1 << reb)) != 0) {
                    LOG.severe(String.format("REB PS %s tripped: %s", reb, ps.getFailureDetailString(reb)));
                }
            }
            return true;
        }
        catch (DriverException e) {
            processCaughtException(e);
            return false;  // Ignore it
        }
    }

    public StateBundle getRebPsDeviceState() {
        synchronized(stateLock) {
            return deviceAndRebsState.clone();
        }
    }
    
    /**
     *  Gets a set of DPHI or HV bias DAC values.
     *
     *  @param  getHV  If true, get HV Bias DACs; otherwise get DPHI DACs 
     *  @return  An array of DAC values
     */
    public double[] getDacs(boolean getHV)
    {
        double[] dacs = new double[ps.getNumRebs()];
        int psNum = getHV ? RebPS.PS_HV_BIAS : RebPS.PS_DPHI;
        for (int reb = 0; reb < dacs.length; reb++) {
            dacs[reb] = Double.NaN;
            try {
                if (isOnline()) {
                    dacs[reb] = ps.readChannel(reb, psNum, RebPS.CHAN_VOLT_DAC);
                }
            }
            catch (DriverException e) {
                processCaughtException(e, Level.SEVERE, () -> String.format("REB PS %s getDacs exception", fullName));                    
            }
        }
        return dacs;
    }


   /**
     *  Sets a HV bias DAC value.
     *
     *  @param  rebNum  The REB number
     *  @param  value   The value to set
     *  @throws  PowerException
     */
    public void setBiasDac(int rebNum, double value) throws PowerException
    {
        try {
            ps.writeDac(rebNum, RebPS.PS_HV_BIAS, value);
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Sets a DPHI DAC value.
     *
     *  @param  rebNum  The REB number
     *  @param  value   The value to set
     *  @throws  PowerException
     */
    public void setDphiDac(int rebNum, double value) throws PowerException
    {
        try {
            ps.writeDac(rebNum, RebPS.PS_DPHI, value);
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Turns a power supply on or off.
     *
     *  Currently the only allowed operations are turning the master on or off,
     *  the bias HV supply on, or the CR DPHI supply on; all others are ignored.
     *
     *  @param  reb    The REB number
     *  @param  psNum  The power supply number, or -1 for master switch
     *  @param  on     Turn on if true, off if false
     *  @throws  PowerException
     */
    public void setPowerOn(int reb, int psNum, boolean on) throws PowerException
    {
        //A reb is being powered on when psNum == -1 and on == true.
        //When this happens we set the rebBeingPoweredOn flag to the value of reb
        if ( psNum == -1 && on ) {
            rebBeingPoweredOn = reb;
        }
        //If a reb is being powered off, we switch the internal reb state variable to OFF
        if ( psNum == -1 && !on ) {
            rebPowerStates.set(reb, RebPowerState.OFF);
        }
        
        if (psNum >= 0 && psNum != RebPS.PS_HV_BIAS
              && !(psType == RebPS.TYPE_CORNER && psNum == RebPS.PS_DPHI)) return;
        try {
            if (psNum < 0) {
                ps.setPower(reb, on ? 1 : 0);
            }
            else {  // HV bias or DPHI
                if (psType == RebPS.TYPE_CORNER && (reb % RebPS.REB_QUANTUM) == RebPS.REB_QUANTUM - 1) return;
                int value = ps.getPower(reb);
                if ((value & 1) == 0) return;  // Not powered
                int mask = 1 << (psNum + 1);
                int newValue = on ? value | mask : value & ~mask;
                if ((mask & (value ^ newValue)) == 0) return;  // No change
                ps.setPower(reb, newValue);
                //ps.writeDac(reb, psNum, psNum == RebPs.PS_HV_BIAS ? 0 : RebPs.DAC_MAXIMUM);
            }
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage(), e);
        } finally {
            rebBeingPoweredOn = -1;
        }
        if (on) {
            forceDataPublicationForNextUpdateIterations();
        }
    }


    /**
     *  Toggles the on/off state of a power supply.
     *
     *  @param  reb    The REB number
     *  @param  psNum  The power supply number, or -1 for master switch
     *  @throws  PowerException
     */
    public void togglePower(int reb, int psNum) throws PowerException
    {
        try {
            int psn = (psNum < 0) ? 0 : psNum + 1;
            setPowerOn(reb, psNum, ((ps.getPower(reb) >> psn) & 1) == 0);
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Sequences a power supply section on or off.
     *
     *  @param  reb  The REB number
     *  @param  on   Turns on if true, off if false
     *  @throws  PowerException
     */
    public void sequencePower(int reb, boolean on) throws PowerException
    {
        if (psVersion == RebPS.VERSION_PROD) {
            setPowerOn(reb, -1, on);  // PS does the sequencing
            return;
        }
        try {
            SeqParams[] onSeq;
            if (psType == RebPS.TYPE_CORNER) {
                if (reb == RebPS.REB_QUANTUM - 1) return;
                onSeq = reb == 0 ? onSeqCR0 : onSeqCR1;
            }
            else {
                onSeq = reb == 1 ? onSeqSR1 : onSeqSR02;
            }
            if (on) {
                ps.writeDac(reb, RebPS.PS_HV_BIAS, 0);
                if (psType == RebPS.TYPE_CORNER) {
                    ps.writeDac(reb, RebPS.PS_DPHI, RebPS.DAC_MAXIMUM);
                }
                for (SeqParams parm : onSeq) {
                    int effReb = reb + parm.rebOff;
                    int value = ps.getPower(effReb);
                    if ((value & 1) == 0) {
                        value |= 1;
                        ps.setPower(effReb, value);
                    }
                    value |= 1 << (parm.psNum + 1);
                    ps.setPower(effReb, value);
                    if (!checkPower(effReb, parm)) {
                        value &= ~(1 << (parm.psNum + 1));
                        ps.setPower(effReb, value);
                        throw new PowerException("REB " + effReb + " power supply "
                                                 + parm.psNum + " failed to stabilize");
                    }
                }
            }
            else {
                ps.setPower(reb, ps.getPower(reb) & ~(1 << (RebPS.PS_HV_BIAS + 1)) & ~(1 << (RebPS.PS_DPHI + 1)));
                for (int j = onSeq.length - 1; j >= 0; j--) {
                    SeqParams parm = onSeq[j];
                    int effReb = reb + parm.rebOff;
                    int value = ps.getPower(effReb) & ~(1 << (parm.psNum + 1));
                    ps.setPower(effReb, value);
                    if (value == 1) {
                        ps.setPower(effReb, 0);
                    }
                }
            }
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Sequences a whole power supply on or off.
     *
     *  @param  on   Turns on if true, off if false
     *  @throws  PowerException
     */
    public void sequencePower(boolean on) throws PowerException
    {
        for (int reb = 0; reb < ps.getNumRebs(); reb++) {
            sequencePower(reb, on);
        }
    }


    /**
     *  Reads extrema for a power supply channel
     *
     *  @param  reb     The REB number
     *  @param  psName  The power supply enum
     *  @param  chan    The channel enum 
     *  @return  A two-element array containing the maximum (0) and minimum (1) values
     *  @throws  PowerException
     */
    public double[] readChanExtrema(int reb, PsName psName, ChanName chan) throws PowerException
    {
        try {
            double[] value = ps.readChanExtended(reb, psName.getValue(), chan.getValue());
            return new double[]{value[RebPS.EXT_VALUE_MAX], value[RebPS.EXT_VALUE_MIN]};
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Resets extrema for a power supply channel
     *
     *  @param  reb     The REB number
     *  @param  psName  The power supply enum
     *  @param  chan    The channel enum
     *  @throws  PowerException
     */
    public void resetChanExtrema(int reb, PsName psName, ChanName chan) throws PowerException
    {
        try {
            ps.resetChanExtrema(reb, psName.getValue(), chan.getValue());
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Sets an event listener.
     *
     *  @param  listen  The listener object
     */
    public void setListener(Event listen)
    {
        listener = listen;
    }


    /**
     *  Clears the event listener.
     */
    public void clearListener()
    {
        listener = null;
    }


    /**
     *  Gets the error counters.
     *
     *  @return  A two-element array containing the number of network timeouts
     *           and the number of sequence errors
     */
    public int[] getErrors()
    {
        return new int[]{ps.getNumTimeout(), ps.getNumSeqErr()};
    }


    /**
     *  Gets the power supply driver.
     *
     *  @return  The driver
     */
    public RebPS getPs()
    {
        return ps;
    }


    /**
     *  Gets the power supply type.
     *
     *  @return  The type
     */
    public int getPsType()
    {
        return psType;
    }


    /**
     *  Gets the power supply version.
     *
     *  @return  The version
     */
    public int getPsVersion()
    {
        return psVersion;
    }


    /**
     *  Gets the power supply ID.
     *
     *  @return  The ID
     */
    public String getPsId()
    {
        return psId;
    }


    /**
     *  Checks that a PS voltage has reached the correct value.
     *
     *  @param  parms  The power supply parameters
     *  @return  Whether the correct value was reached
     */
    private boolean checkPower(int reb, SeqParams parms) throws PowerException
    {
        boolean okay = false;
        int nGood = 0;
        for (int count = 0; count < 20 && !okay; count++) {
            try {
                Thread.sleep(50);
            }
            catch (InterruptedException e) {
            }
            try {
                double voltage = ps.readChannel(reb, parms.psNum, RebPS.CHAN_VOLT_AFT_SW);
                nGood = (voltage >= parms.minGood && voltage <= parms.maxGood) ? nGood + 1 : 0;
                okay = nGood >= 3;
            }
            catch (DriverException e) {
                throw new PowerException(e.getMessage());
            }
        }

        return okay;
    }


    /**
     *  Generates the power supply ID and type.
     */
    private void generatePsId() throws DriverException
    {
        if (isSimulated){
            psId = "PS-SIM";
            psType = simulatedPsType != RebPS.TYPE_UNKNOWN ? simulatedPsType : RebPS.TYPE_SCIENCE;
            return;
        }
        String serial = String.format("%016x", ps.getSerialNo());
        psId = props.getProperty("org.lsst.ccs.power.psid." + serial);
        if (psId == null) {
            if (ps.getNumRebs() == 3) {
                LOG.log(Level.WARNING, "Unknown PS serial number: {0}", serial);
                psId = serial.substring(5, 10);
            }
            else {
                psId = (psType == RebPS.TYPE_CORNER ? "CR-" : "SR-");
                try {
                    int id = Integer.valueOf(ipAddr.split("\\.")[3]);
                    psId += String.format("%03d", id);
                }
                catch (IndexOutOfBoundsException | NumberFormatException e) {
                    psId += "XXX";
                }
            }
        }
        psType = psType == RebPS.TYPE_UNKNOWN ? psId.substring(0, 2).equals("CR")
                   ? RebPS.TYPE_CORNER : RebPS.TYPE_SCIENCE : psType;
    }
    
    private void forceDataPublicationForNextUpdateIterations() {
        for (MonitorUpdateTask task: tasksForDevice) {
            task.forceDataPublicationOnNextUpdates(nPowerOnPub);
        }
    }
    
    void setSimulatedPsType(int simulatedType) {
        if (!RunMode.isSimulation()) {
            throw new RuntimeException("What are you doing? This cannot be done unless in simulation!!!");
        }
        if ( simulatedType != RebPS.TYPE_CORNER && simulatedType != RebPS.TYPE_SCIENCE ) {
            throw new RuntimeException("What are you doing? Setting simulated type to unknown value!!! "+simulatedType);            
        }
        if ( simulatedPsType == RebPS.TYPE_UNKNOWN ) {
            simulatedPsType = simulatedType;            
        } else if ( simulatedPsType != simulatedType ) {
            throw new RuntimeException("What are you doing? Overriding already assigned value: "+simulatedPsType+" != "+simulatedType);            
        }
    }
    
    @Command(name = "dump", type = Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Dump info about power supply")
    public String dumpInfo() throws DriverException {
        long serialNo = ps.getSerialNo();
        String buildStamp = ps.getBuildStamp();
        double fPGATemperature = ps.readFPGATemperature();
        double fPGAMinTemperature = ps.readMinFPGATemperature();
        double fPGAMaxTemperature = ps.readMaxFPGATemperature();
        int numSeqErr = ps.getNumSeqErr();
        int numTimeout = ps.getNumTimeout();

        return String.format("%s\n\tbuildStamp: %s\n\tserialNumber: %x\n\tfpga(temp,min,max): [%g,%g,%g]\n\tnumSeqErr: %d\n\tnumTimeout: %d", 
                fullName, buildStamp, serialNo, fPGATemperature, fPGAMinTemperature, fPGAMaxTemperature, numSeqErr, numTimeout);
    }

    @Command(name = "readReg", type = Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Query one or more registers")
    public int[] readRegs(int addr, @Argument(defaultValue = "1") int count) throws DriverException {
        return ps.readRegs(addr, count);
    }    
    
    /**
     * Get the name of the switch name controlling this PS board.
     * 
     */
    public String getBulkPsSwitchName() {
        return switchName;
    }
    
    //Methods to convert integer state flags to Power states
    private RebHvBiasState getRebHvBiasState(int powerState) {
        int hvBiasState = decodePowerState(powerState, RebPsEnum.HVBIAS);
        if ( hvBiasState < 0 ) {
            return RebHvBiasState.UNKNOWN;
        }
        if ( hvBiasState == 0 ) {
            return RebHvBiasState.OFF;
        } else if ( hvBiasState == 1 ) {
            return RebHvBiasState.ON;
        }
        
        throw new RuntimeException("Unknown HVBias state "+hvBiasState);        
    }
    
    private RebDPhiState getRebDphiState(int powerState) {        
        int dphiState = decodePowerState(powerState, RebPsEnum.DPHI);
        if ( dphiState < 0 ) {
            return RebDPhiState.UNKNOWN;
        }
        if ( dphiState == 0 ) {
            return RebDPhiState.OFF;
        } else if ( dphiState == 1 ) {
            return RebDPhiState.ON;
        }
        
        throw new RuntimeException("Unknown Dphi state "+dphiState);
        
    }
    
    
    /**
     *  Generates the REB power state.
     * 
     *  @param value The power value
     *  @return The generated power state
     */
    private RebPowerState getRebPowerState(int powerState) {
        int pState = decodePowerState(powerState, RebPsEnum.MASTER);
        if ( pState < 0 ) {
            return RebPowerState.UNKNOWN;
        } 
        if ( pState == 0 ) {
            return RebPowerState.OFF;            
        }
        
        return RebPowerState.ON;
    }
    
    private int decodePowerState(int powerState, RebPsEnum id) {
        if (powerState < 0) {
            return -1;
        }
        int psNum = id.getNumber() +1;
        int r = (powerState >> psNum) & 1;
        return r;
                
    }
    
    
}
