package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.actions.RebPowerAction;
import org.lsst.ccs.subsystem.power.alerts.RebPowerAlertType;
import org.lsst.ccs.subsystem.power.constants.RebPsEnum;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.states.RebDPhiState;
import org.lsst.ccs.subsystem.power.states.RebHvBiasState;
import org.lsst.ccs.subsystem.power.states.RebPowerState;

/**
 * A class representing an individual Reb.
 * It contains the information regarding the power supply powering this reb.
 * It provides commands to control the power to this reb.
 *
 * @author The LSST CCS Team
 */
public class RebPowerSupplyNode implements HasLifecycle {

    protected final PowerSupplyConfiguration powerSupplyConfiguration;
    protected final int channel;
    protected static final Logger LOG = Logger.getLogger(RebPowerSupplyNode.class.getName());

    @LookupName
    protected String rebName;

    @LookupPath
    protected String rebPath;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String, RebPsDevice> psDevices = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.TOP)
    protected Agent agent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter = "Power")
    private Channel powerChannel;

    @ConfigurationParameter( range = "0..2500")
    protected double hvBias;

    protected RebPsDevice powerSupplyDevice;

    private volatile boolean disableRebPowerOn = false;
    private volatile boolean disableRebHvBiasOn = false;

    //Lock used to prefent interference between the reb power On/Off sequences.
    private final Object powerOnOffLock = new Object();

    //Lock used to prefent interference between the hvBias On/Off sequences.
    private final Object hvBiasOnOffLock = new Object();

    private final Map<RebPowerAction.Type, List<String>> outstandingAlerts = new ConcurrentHashMap<>();
    
    public RebPowerSupplyNode(PowerSupplyConfiguration powerSupplyConfiguration, int channel) {
        this.channel = channel;
        this.powerSupplyConfiguration = powerSupplyConfiguration;
    }

    @Override
    public void build() {
        String psName = powerSupplyConfiguration.getName();
        for (Entry<String,RebPsDevice> e : psDevices.entrySet()) {
            if ( e.getKey().endsWith(psName)) {
                if ( powerSupplyDevice != null ) {
                    throw new RuntimeException("Found more than one power supply that matches name "+powerSupplyConfiguration.getName());                
                }
                powerSupplyDevice = e.getValue();
            }
        }
        if ( powerSupplyDevice == null ) {
            throw new RuntimeException("Could not find power supply device "+powerSupplyConfiguration.getName()+" corresponding to this reb "+rebName);
        }
    }

    @ConfigurationParameterChanger(propertyName = "hvBias")
    public void setHVBias(double hvBias) {
        this.hvBias = hvBias;
        //Write configuration value to DAC only if RebPower is ON
        //Synchornize on StateLock to avoid race conditions with thread that
        //updates states
        synchronized (agentStateService.getStateLock()) {
            if (agentStateService.isComponentInState(rebPath, RebPowerState.ON)) {
                writeHvBiasConfigurationValueToDac();
            }
        }
    }

    private void writeHvBiasConfigurationValueToDac() {
        try {
            powerSupplyDevice.setBiasDac(channel, hvBias);
            LOG.log(Level.INFO, "{0}: set hvBias DAC value of {1}", new Object[]{rebPath, hvBias});
        } catch (PowerException ex) {
            LOG.log(Level.WARNING,"Could not set the HVBias DAC value",ex);
            throw new RuntimeException(ex);
        }
    }

    @Command(type = Command.CommandType.ACTION, description = "Turns on the power to this Reb", autoAck = false)
    public void powerRebOn() throws PowerException {
        synchronized (powerOnOffLock) {
            agent.helper()
                    .precondition(!disableRebPowerOn, rebPath + ": powering on has been disabled due to outstanding alerts: "+outstandingAlerts.get(RebPowerAction.Type.TURN_REB_OFF))
                    .precondition(agentStateService.isComponentInState(rebPath, RebHvBiasState.OFF), "The Reb hvBias switch must be open.")
                    .precondition((this instanceof CornerRaftRebPowerSupplyNode) ? agentStateService.isComponentInState(rebPath, RebDPhiState.OFF) : true, "The Reb dphi switch must be open.")
                    .enterFaultOnException(true).action(
                    () -> {
                        powerSupplyDevice.sequencePower(channel, true);
                        writeHvBiasConfigurationValueToDac();
                        if (this instanceof CornerRaftRebPowerSupplyNode) {
                            ((CornerRaftRebPowerSupplyNode) this).writeDphiConfigurationValueToDac();
                        }

                    });
        }
    }

    @Command(type = Command.CommandType.ACTION, description = "Turns off the power to this Reb")
    public void powerRebOff() throws PowerException {
        powerRebOff(false);
    }

    private void powerRebOff(boolean disable) throws PowerException {
        synchronized(powerOnOffLock) {
            disableRebPowerOn = disable;
            powerSupplyDevice.sequencePower(channel, false);
        }
    }
    
    @Command(type = Command.CommandType.ACTION, description = "Turns on the HV bias to the configuration value", autoAck = false)
    public void hvBiasOn() throws PowerException {
        synchronized (hvBiasOnOffLock) {
            agent.helper()
                    .precondition(!disableRebHvBiasOn, rebPath + ": HvBias on has been disabled due to outstanding alerts: "+outstandingAlerts.get(RebPowerAction.Type.TURN_REB_HVBIAS_OFF))
                    .precondition(agentStateService.isComponentInState(rebPath, RebPowerState.ON), "The Reb main power must be ON.").enterFaultOnException(true).action(
                    () -> {
                        powerSupplyDevice.setPowerOn(channel, RebPsEnum.HVBIAS.getNumber(), true);
                    });
        }
    }

    @Command(type=Command.CommandType.ACTION, description="Turns off the HV bias")
    public void hvBiasOff() throws PowerException {
        hvBiasOff(false);
    }

    private void hvBiasOff(boolean disable) throws PowerException {
        synchronized(hvBiasOnOffLock) {
            disableRebHvBiasOn = disable;
            powerSupplyDevice.setPowerOn(channel, RebPsEnum.HVBIAS.getNumber(), false);
        }
    }
    
    @Command(type = Command.CommandType.ACTION, description = "Read the HvBias DAC value")
    public double readHvBiasDac() throws PowerException {
        return powerSupplyDevice.getDacs(true)[channel];
    }

    /**
     *  Reads the maximum and minimum values of a channel.
     *
     *  @param  psName  The power supply enum
     *  @param  chan    The channel enum
     *  @return  A two-element array containing maximum (0) & minimum (0) values
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.QUERY, description="Read a channel's maximum & minimum values")
    public double[] getChanExtrema(@Argument(description="PS name")
                                   RebPsDevice.PsName psName,
                                   @Argument(description="Channel name")
                                   RebPsDevice.ChanName chan) throws PowerException {
        return powerSupplyDevice.readChanExtrema(channel, psName, chan);
    }


    /**
     *  Resets the maximum and minimum values of a channel.
     *
     *  @param  psName  The power supply enum
     *  @param  chan    The channel enum
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.ACTION, description="Reset a channel's maximum & minimum values")
    public void resetChanExtrema(@Argument(description="PS name")
                                 RebPsDevice.PsName psName,
                                 @Argument(description="Channel name")
                                 RebPsDevice.ChanName chan) throws PowerException {
        powerSupplyDevice.resetChanExtrema(channel, psName, chan);
    }

    RebPsDevice getRebPsDevice() {
        return powerSupplyDevice;
    }

    Channel getPowerChannel() {
        return powerChannel;
    }

    synchronized void processEmergencyResponse(String origin, Alert alert, RebPowerAction rebPowerAction) {
        if (rebPowerAction != null) {
            //If there is an action process it on a separate thread.
            if (rebPath.equals(rebPowerAction.getRebPath())) {
                agent.getScheduler().execute(() -> {
                    //If there is an action to be carried out, do it right away.
                    switch (rebPowerAction.getType()) {
                        case TURN_REB_HVBIAS_OFF:
                            LOG.log(Level.WARNING, "Turning HVBias OFF for reb {0}", rebPath);
                            try {
                                hvBiasOff(true);
                            } catch (PowerException pe) {
                                LOG.log(Level.SEVERE, "Could not turn HVBias OFF for reb " + rebPath, pe);
                                alertService.raiseAlert(RebPowerAlarm.getRebPowerAlert(RebPowerAlertType.EMERGENCY_ACTION_FAILED, rebPath), AlertState.ALARM, "Could not turn HVBias OFF for reb " + rebPath);
                            }
                            break;
                        case TURN_REB_OFF:
                            LOG.log(Level.WARNING, "Turning Power OFF for reb {0}", rebPath);
                            try {
                                powerRebOff(true);
                            } catch (PowerException pe) {
                                LOG.log(Level.SEVERE, "Could not turn Power OFF for reb " + rebPath, pe);
                                alertService.raiseAlert(RebPowerAlarm.getRebPowerAlert(RebPowerAlertType.EMERGENCY_ACTION_FAILED, rebPath), AlertState.ALARM, "Could not turn Power OFF for reb " + rebPath);
                            }
                            break;
                        default:
                            LOG.log(Level.SEVERE, "Unknown Reb Power Action {0} for Reb path {1}", new Object[]{rebPowerAction.getType(), rebPath});
                    }
                });
            }
        } 
    }
    
    void updateEmergencyState(RebPowerAction.Type actionType, List<String> outstandingAlerts) {
        this.outstandingAlerts.put(actionType,outstandingAlerts);
        if (outstandingAlerts.isEmpty()) {
            switch (actionType) {
                case TURN_REB_HVBIAS_OFF:
                    LOG.log(Level.INFO, "Enabling hvBiasOn command for {0}", rebPath);
                    disableRebHvBiasOn = false;
                    break;
                case TURN_REB_OFF:
                    LOG.log(Level.INFO, "Enabling powerRebOn command for {0}", rebPath);
                    disableRebPowerOn = false;
                    break;
            }
        }
    }
    
    
        
}
