package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.power.constants.RebPsEnum;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.states.RebPowerState;

/**
 * A class representing an individual Reb.
 * It contains the information regarding the power supply powering this reb.
 * It provides commands to control the power to this reb.
 * 
 * @author The LSST CCS Team
 */
public class RebPowerSupplyNode implements HasLifecycle {
    
    protected final PowerSupplyConfiguration powerSupplyConfiguration;
    protected final int channel;
    protected static final Logger LOG = Logger.getLogger(RebPowerSupplyNode.class.getName());
    
    @LookupName
    protected String rebName;
    
    @LookupPath
    protected String rebPath;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String, RebPsDevice> psDevices = new HashMap<>();
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    protected Agent agent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter = "Power")
    private Channel powerChannel;

    @ConfigurationParameter( range = "0..3500")
    protected double hvBias;

    protected RebPsDevice powerSupplyDevice;
        
    public RebPowerSupplyNode(PowerSupplyConfiguration powerSupplyConfiguration, int channel) {
        this.channel = channel;
        this.powerSupplyConfiguration = powerSupplyConfiguration;        
    }

    @Override
    public void build() {
        String psName = powerSupplyConfiguration.getName();
        for (Entry<String,RebPsDevice> e : psDevices.entrySet()) {
            if ( e.getKey().endsWith(psName)) {
                if ( powerSupplyDevice != null ) {
                    throw new RuntimeException("Found more than one power supply that matches name "+powerSupplyConfiguration.getName());                
                }
                powerSupplyDevice = e.getValue();
            }
        }
        if ( powerSupplyDevice == null ) {
            throw new RuntimeException("Could not find power supply device "+powerSupplyConfiguration.getName()+" corresponding to this reb "+rebName);
        }
    }
    
    @ConfigurationParameterChanger(propertyName = "hvBias")
    public void setHVBias(double hvBias) {
        //TO-DO: what should we do when setting the dac value fails?
        //When can this operation fail?
        try {
            powerSupplyDevice.setBiasDac(channel, hvBias);        
            this.hvBias = hvBias;
        } catch (PowerException ex) {
            LOG.log(Level.WARNING,"Could not set the HVBias DAC value",ex);
            throw new RuntimeException(ex);
        }
    }
    
    //TO-DO: Should these commands be at level 0?
    @Command(type=Command.CommandType.ACTION, description="Turns on the power to this Reb")
    public void powerRebOn() throws PowerException {
        powerSupplyDevice.sequencePower(channel, true);
    }
    
    @Command(type=Command.CommandType.ACTION, description="Turns off the power to this Reb")
    public void powerRebOff() throws PowerException {
        powerSupplyDevice.sequencePower(channel, false);
    }
    
    @Command(type=Command.CommandType.ACTION, description="Turns on the HV bias to the configuration value", autoAck = false)
    public void hvBiasOn() throws PowerException {
        agent.helper().precondition(agentStateService.isComponentInState(rebPath, RebPowerState.ON),"The Reb main power must be ON.").enterFaultOnException(true).action(
                () -> {
                    powerSupplyDevice.setPowerOn(channel, RebPsEnum.HVBIAS.getNumber(), true);
                    powerSupplyDevice.setBiasDac(channel, hvBias);                    
                });
    }
    
    @Command(type=Command.CommandType.ACTION, description="Turns off the HV bias")
    public void hvBiasOff() throws PowerException {
        powerSupplyDevice.setPowerOn(channel, RebPsEnum.HVBIAS.getNumber(), false);
    }
    
    /**
     *  Reads the maximum and minimum values of a channel.
     *
     *  @param  psName  The power supply enum
     *  @param  chan    The channel enum
     *  @return  A two-element array containing maximum (0) & minimum (0) values
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.QUERY, description="Read a channel's maximum & minimum values")
    public double[] getChanExtrema(@Argument(description="PS name")
                                   RebPsDevice.PsName psName,
                                   @Argument(description="Channel name")
                                   RebPsDevice.ChanName chan) throws PowerException {
        return powerSupplyDevice.readChanExtrema(channel, psName, chan);
    }


    /**
     *  Resets the maximum and minimum values of a channel.
     *
     *  @param  psName  The power supply enum
     *  @param  chan    The channel enum
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.ACTION, description="Reset a channel's maximum & minimum values")
    public void resetChanExtrema(@Argument(description="PS name")
                                 RebPsDevice.PsName psName,
                                 @Argument(description="Channel name")
                                 RebPsDevice.ChanName chan) throws PowerException {
        powerSupplyDevice.resetChanExtrema(channel, psName, chan);
    }
  
    RebPsDevice getRebPsDevice() {
        return powerSupplyDevice;
    }
    
    Channel getPowerChannel() {
        return powerChannel;
    }
}
