package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystem.common.focalplane.data.HasFocalPlaneData;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;
import org.lsst.ccs.subsystem.power.constants.RebPsEnum;
import org.lsst.ccs.subsystem.power.data.PowerDataGroup;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.states.PowerSupplyState;
import org.lsst.ccs.subsystem.power.states.RebDPhiState;
import org.lsst.ccs.subsystem.power.states.RebHvBiasState;
import org.lsst.ccs.subsystem.power.states.RebPowerState;

/**
 *  Implements the REB PS control subsystem.
 *
 *  @author The LSST CCS Team
 */
public class RebPowerSupplyMain extends Subsystem implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(RebPowerSupplyMain.class.getName());

    @LookupField( strategy=LookupField.Strategy.CHILDREN)
    private List<PowerDevice> mainPsList = new ArrayList<>();
    @LookupField( strategy=LookupField.Strategy.CHILDREN)    
    private List<RebPsDevice> psDeviceList = new ArrayList<>();

        
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private DataProviderDictionaryService dictionaryService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    //The list of all the Rebs
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private List<RebPowerSupplyNode> rebs = new ArrayList<>();
    
    @ConfigurationParameter
    private long powerOffSleepMillis = 100;
    
    //The map of all the Channels
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String,Channel> channels = new HashMap<>();
        
    public RebPowerSupplyMain() {
        super("rebps", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }

    
    @Override
    public void build() {
        agentPropertiesService.setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.POWER.name());
        agentPropertiesService.setAgentProperty(HasFocalPlaneData.AGENT_PROPERTY, HasFocalPlaneData.generatePropertyValue(PowerDataGroup.class));
        AgentPeriodicTask pt;
        //Create and schedule an AgentPeriodicTask to update the RebPowerState
        pt = new AgentPeriodicTask("reb-power-state",
                                   () -> updateRebPowerState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }
    
    @Override
    public void init()
    {
        for (RebPowerSupplyNode reb : rebs) {
            stateService.registerState(RebPowerState.class, "REB power state", reb);        
            stateService.registerState(RebHvBiasState.class, "REB HV bias state", reb);        
            if ( reb instanceof CornerRaftRebPowerSupplyNode ) {
                stateService.registerState(RebDPhiState.class, "REB Dphi state", reb);        
            }
        
        }        
        for (RebPsDevice powerSupplyDevice : psDeviceList ) {
            stateService.registerState(PowerSupplyState.class, "Power Supply state", powerSupplyDevice);
        }
    }
    
    @Override
    public void preStart() {        
        //Manipulate the dictionary here: 
        for ( DataProviderInfo data: dictionaryService.getDataProviderDictionary().getDataProviderInfos() ) {
            PowerDataGroup dataGroup = PowerDataGroup.findPowerDataGroup(data);
            if ( dataGroup != null ) {
                dataGroup.addAttributesToDataInfo(data);
            }
        }
    }
    
    

    /**
     *  Updates the REB power state periodically.
     */
    private void updateRebPowerState() {
        //Temporary maps to store the power supply state
        Map<RebPsDevice, int[]> powerSupplyStateMap = new HashMap<>();
        Map<RebPsDevice, RebTrippedState[]> rebTrippedStateMap = new HashMap<>();
        
        StateBundle sb = new StateBundle();        
        
        //Update the Power Supply state
        for (RebPsDevice powerSupplyDevice : psDeviceList ) {
            sb.setComponentState(powerSupplyDevice.psPath, powerSupplyDevice.isOnline() ? PowerSupplyState.ON : PowerSupplyState.OFF);
        }
        
        //Update the Reb Power states
        for (RebPowerSupplyNode reb : rebs) {

            if ( !reb.getRebPsDevice().isOnline() ) {
                sb.setComponentState(reb.rebPath, RebPowerState.UNKNOWN);
                sb.setComponentState(reb.rebPath, RebHvBiasState.UNKNOWN);
                if ( reb instanceof CornerRaftRebPowerSupplyNode ) {
                    sb.setComponentState(reb.rebPath, RebDPhiState.UNKNOWN);
                }                
                continue;
            }

            int[] powerState = powerSupplyStateMap.getOrDefault(reb.getRebPsDevice(), reb.getRebPsDevice().getState());
            powerSupplyStateMap.put(reb.getRebPsDevice(), powerState);
            
            RebTrippedState[] rebTrippedStateArray = rebTrippedStateMap.getOrDefault(reb.getRebPsDevice(), reb.getRebPsDevice().getPsTrippedStatus());
            rebTrippedStateMap.put(reb.getRebPsDevice(), rebTrippedStateArray);
            
            if ( powerState.length != 6 ) {
                throw new RuntimeException("This length should have been 6 "+powerState.length);
            }
            
            int rebPowerState = powerState[reb.channel];
            
            sb.setComponentState(reb.rebPath, getRebPowerState(rebPowerState, rebTrippedStateArray[reb.channel]));
            sb.setComponentState(reb.rebPath, getRebHvBiasState(rebPowerState));
            if ( reb instanceof CornerRaftRebPowerSupplyNode ) {
                sb.setComponentState(reb.rebPath, getRebDphiState(rebPowerState));
            }
        }
                
        stateService.updateAgentState(sb);
    }

    private int decodePowerState(int powerState, RebPsEnum id) {
        if (powerState < 0) {
            return -1;
        }
        int psNum = id.getNumber() +1;
        int r = (powerState >> psNum) & 1;
        return r;
                
    }
    
    private RebHvBiasState getRebHvBiasState(int powerState) {
        int hvBiasState = decodePowerState(powerState, RebPsEnum.HVBIAS);
        if ( hvBiasState < 0 ) {
            return RebHvBiasState.UNKNOWN;
        }
        if ( hvBiasState == 0 ) {
            return RebHvBiasState.OFF;
        } else if ( hvBiasState == 1 ) {
            return RebHvBiasState.ON;
        }
        
        throw new RuntimeException("Unknown HVBias state "+hvBiasState);        
    }
    
    private RebDPhiState getRebDphiState(int powerState) {        
        int dphiState = decodePowerState(powerState, RebPsEnum.DPHI);
        if ( dphiState < 0 ) {
            return RebDPhiState.UNKNOWN;
        }
        if ( dphiState == 0 ) {
            return RebDPhiState.OFF;
        } else if ( dphiState == 1 ) {
            return RebDPhiState.ON;
        }
        
        throw new RuntimeException("Unknown Dphi state "+dphiState);
        
    }
    
    
    /**
     *  Generates the REB power state.
     * 
     *  @param value The power value
     *  @return The generated power state
     */
    private RebPowerState getRebPowerState(int powerState, RebTrippedState rebTrippedState) {
        if ( rebTrippedState.hasTripped() ) {
            return RebPowerState.TRIPPED;
        }
        int pState = decodePowerState(powerState, RebPsEnum.MASTER);
        if ( pState < 0 ) {
            return RebPowerState.UNKNOWN;
        } 
        if ( pState == 0 ) {
            return RebPowerState.OFF;            
        }
        
        return RebPowerState.ON;
    }
    
    @Command
    public void powerOffAllRebs() throws InterruptedException {
        String failedRebs = "";
        int count = rebs.size();
        for (RebPowerSupplyNode node: rebs) {
            count--;
            if ( stateService.isComponentInState(node.rebPath, RebPowerState.ON) ) {
                LOG.log(Level.INFO, "Powering off reb: {0}", new Object[]{node.rebPath});
                try {
                    node.powerRebOff();
                } catch (PowerException ex) {
                    LOG.log(Level.WARNING, "Failed to power off reb: "+node.rebPath, ex);                    
                    failedRebs += node.rebPath + " ";
                }
                if ( count > 0 ) {
                    Thread.sleep(powerOffSleepMillis);
                }
            }
        }
        if ( ! failedRebs.isEmpty() ) {
            throw new RuntimeException("Failed to turn off the following rebs: "+failedRebs.trim());
        }
    }
        
}
