package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.auxelex.RebBulkPS;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.power.constants.PowerConfig;
import org.lsst.ccs.subsystem.power.constants.QuadBoxSwitches;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Interface to the SLAC REB bulk PS controller.
 *
 *  @author Owen Saxton
 */
public class RebBulkPsDevice extends Device implements SwitchControl {

    /**
     *  Constants.
     */
    private static final Logger LOG = Logger.getLogger(RebBulkPsDevice.class.getName());

    protected static final int
        MON_TYPE_VOLTAGE = 0,
        MON_TYPE_CURRENT = 1,
        MON_TYPE_TEMP = 2,
        MON_TYPE_BOARD_TEMP = 3;

    private static final String
        NODE   = "node";

    private static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
        mTypeMap.put("TEMP", MON_TYPE_TEMP);
        mTypeMap.put("BOARDTEMP", MON_TYPE_BOARD_TEMP);
    }

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name=NODE, category=PowerConfig.CATEGORY_QUADBOX, isFinal=true)
    protected Integer node;

    private RebBulkPS ps = new RebBulkPS();


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (node == null) {
            MonitorLogUtils.reportConfigError(LOG, name, NODE, "is missing");
        }
        fullName = "REB bulk PS board (" + node + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    public void initialize()
    {
        try {
            ps.open(node);
            setOnline(true);
            initSensors();
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            ps.close();
        }
        catch (DriverException e) {
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  Only the type can be checked at this time.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            MonitorLogUtils.reportError(LOG, name, "type", type);
        }
        if (mType != MON_TYPE_BOARD_TEMP && (hwChan < 0 || hwChan >= RebBulkPS.NUM_BULKS)) {
            MonitorLogUtils.reportError(LOG, name, "hwChan", hwChan);
        }
        return new int[]{mType, 0};
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        String item = null;
        if (online) {
            try {
                switch (type) {
                case MON_TYPE_VOLTAGE:
                    item = "voltage";
                    value = ps.readVoltage(hwChan);
                    break;

                case MON_TYPE_CURRENT:
                    item = "current";
                    value = ps.readCurrent(hwChan);
                    break;

                case MON_TYPE_TEMP:
                    item = "temperature";
                    value = ps.readTemperature(hwChan);
                    break;

                case MON_TYPE_BOARD_TEMP:
                    item = "board temperature";
                    value = ps.readBoardTemperature();
                    break;
                }
            }
            catch (DriverException e) {
                LOG.error("Error reading " + fullName + " " + item + ": " + e);
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Gets the switch device type
     * 
     *  @return  The switch device type
     */
    @Override
    public int getSwitchDevice()
    {
        return QuadBoxSwitches.DEVC_REB_BULK;
    }


    /**
     *  Turns a bulk power supply on.
     *
     *  @param  psNum  The PS number
     */
    @Override
    public void switchOn(int psNum)
    {
        try {
            ps.setPowerOn(psNum, true);
        }
        catch (DriverException e) {
            LOG.error("Error switching on PS " + psNum + " on " + fullName + ": " + e);
        }
    }


    /**
     *  Turns a bulk power supply off.
     *
     *  @param  psNum  The PS number
     */
    @Override
    public void switchOff(int psNum)
    {
        try {
            ps.setPowerOn(psNum, false);
        }
        catch (DriverException e) {
            LOG.error("Error switching off PS " + psNum + " on " + fullName + ": " + e);
        }
    }


    /**
     *  Gets the state of a bulk power supply.
     *
     *  @param  psNum  The PS number
     *  @return  Whether the PS is on or off
     */
    @Override
    public Boolean isSwitchOn(int psNum)
    {
        if (!online || psNum < 0) return null;
        try {
            return (ps.getIoStatus(psNum) & RebBulkPS.IOSTAT_REM_ON) != 0;
        }
        catch (DriverException e) {
            LOG.error("Error getting IO status for PS " + psNum + " on " + fullName + ": " + e);
            return null;
        }
    }

}
