package org.lsst.ccs.subsystem.power;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20DiscControl;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.power.constants.QuadBoxSwitches;

/**
 *  Handles MAQ20 DIOL modules.
 *
 *  @author Owen Saxton
 */
public class QbMaq20Device extends Maq20Device implements SwitchControl {

    /**
     *  Inner class for storing switch information.
     */
    static class SwitchData {

        Maq20DiscControl control;
        int channel;

        SwitchData(Maq20DiscControl control, int channel) {
            this.control = control;
            this.channel = channel;
        }

    }

    /**
     *  Data fields.
     */
    private Maq20DiscControl utHcuPower, mechHcuPower, fesProt;  // Groovy file

    private static final Logger LOG = Logger.getLogger(QbMaq20Device.class.getName());
    private final SwitchData[] switchData = new SwitchData[QuadBoxSwitches.NUM_MAQ20_SWITCHES];


    /**
     *  Device initialization - checks configuration.
     */
    @Override
    public void initDevice()
    {
        super.initDevice();
        if (utHcuPower == null) {
            ErrorUtils.reportConfigError(LOG, path, "utHcuPower", "not defined");
        }
        switchData[QuadBoxSwitches.SW_MAQ20_REB_PS_HCU] = new SwitchData(utHcuPower, 3);
        switchData[QuadBoxSwitches.SW_MAQ20_CRYOSTAT_HCU] = new SwitchData(utHcuPower, 4);
        if (mechHcuPower == null) {
            ErrorUtils.reportConfigError(LOG, path, "mechHcuPower", "not defined");
        }
        switchData[QuadBoxSwitches.SW_MAQ20_FES_HCU] = new SwitchData(mechHcuPower, 3);
        switchData[QuadBoxSwitches.SW_MAQ20_SHUTTER_HCU] = new SwitchData(mechHcuPower, 4);
        if (fesProt == null) {
            ErrorUtils.reportConfigError(LOG, path, "fesProt", "not defined");
        }
        switchData[QuadBoxSwitches.SW_MAQ20_FES_PLC] = new SwitchData(fesProt, 4);
    }


    /**
     *  Gets the switch device type.
     * 
     *  @return  The switch device type enumeration
     */
    @Override
    public int getSwitchDevice()
    {
        return QuadBoxSwitches.DEVC_MAQ20;
    }


    /**
     *  Sets a switch on.
     *
     *  @param  sw  The switch number.
     */
    @Override
    public void switchOn(int sw)
    {
        SwitchData swData = switchData[sw];
        try {
            swData.control.setLineOn(swData.channel, false);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error writing channel {0} on {1}: {2}", new Object[]{swData.channel, getPath(), e});
        }
    }


    /**
     *  Sets a switch off.
     *
     *  @param  sw  The switch number.
     */
    @Override
    public void switchOff(int sw)
    {
        SwitchData swData = switchData[sw];
        try {
            swData.control.setLineOn(swData.channel, true);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error writing channel {0} on {1}: {2}", new Object[]{swData.channel, getPath(), e});
        }
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        if (sw < 0) return null;  // No "main" switch
        SwitchData swData = switchData[sw];
        Boolean state = swData.control.isLineOn(swData.channel);
        return state == null ? null : !state;
    }

}
