package org.lsst.ccs.subsystem.power;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.auxelex.BfrComCam;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.power.constants.PowerConfig;
import org.lsst.ccs.subsystem.power.constants.ComCamQuadBoxSwitches;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Interface to the SLAC BFR.
 *
 *  @author Owen Saxton
 */
public class BfrComCamDevice extends Device implements SwitchControl {

    /**
     *  Constants.
     */
    private static final Logger LOG = Logger.getLogger(BfrComCamDevice.class.getName());

    private static final String
        NODE   = "node";

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name=NODE, category=PowerConfig.CATEGORY_QUADBOX, isFinal=true)
    private volatile Integer node;

    private final BfrComCam bfr = new BfrComCam();
    private double currents[] = null;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (node == null) {
            MonitorLogUtils.reportConfigError(LOG, name, NODE, "is missing");
        }
        fullName = "BFR board (" + node + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    public void initialize()
    {
        try {
            bfr.open(node);
            setOnline(true);
            initSensors();
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            bfr.close();
        }
        catch (DriverException e) {
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= BfrComCam.NUM_SENSORS) {
            MonitorLogUtils.reportError(LOG, name, "hw channel number", hwChan);
        }
        return new int[]{0, 0};
    }


    /**
     *  Reads all monitored channels.
     *
     */
    @Override
    protected void readChannelGroup()
    {
        currents = null;
        if (isOnline()) {
            try {
                currents = bfr.readCurrent(BfrComCam.CurrentType.AC);
            }
            catch (DriverException e) {
                LOG.error("Error reading " + fullName + " currents: " + e);
                setOnline(false);
            }
        }
    }


    /**
     *  Reads a monitored channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return currents == null ? Double.NaN : currents[hwChan];
    }


    /**
     *  Gets the switch device type.
     * 
     *  @return  The switch device type enumeration
     */
    @Override
    public int getSwitchDevice()
    {
        return ComCamQuadBoxSwitches.DEVC_BFR;
    }


    /**
     *  Turns a relay on.
     *
     *  @param  relay  The relay number
     */
    @Override
    public void switchOn(int relay)
    {
        try {
            bfr.setRelayOn(relay);
        }
        catch (DriverException e) {
            LOG.error("Error setting relay " + relay + " on " + fullName + ": " + e);
        }
        
    }


    /**
     *  Turns a relay off.
     *
     *  @param  relay  The relay number
     */
    @Override
    public void switchOff(int relay)
    {
        try {
            bfr.setRelayOff(relay);
        }
        catch (DriverException e) {
            LOG.error("Error setting relay " + relay + " on " + fullName + ": " + e);
        }
        
    }


    /**
     *  Gets the state of a relay.
     *
     *  @param  relay  The relay number
     *  @return  Whether the relay is on or off
     */
    @Override
    public Boolean isSwitchOn(int relay)
    {
        if (!isOnline() || relay < 0) return null;
        try {
            return bfr.isRelayOn(relay);
        }
        catch (DriverException e) {
            LOG.error("Error getting state for relay " + relay + " on " + fullName + ": " + e);
            return null;
        }
    }

}
