package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.power.constants.ATSChannels;
import org.lsst.ccs.subsystem.power.data.ATSPowerState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;

/**
 *  Implements the Auxiliary Telescope power control subsystem.
 *
 *  @author Owen Saxton
 */
public class ATSPowerMain implements HasLifecycle {

    private static final String
        DIGITAL_CHAN = "Digital",
        ANALOG_CHAN = "Analog",
        CLK_HIGH_CHAN = "ClockHigh",
        CLK_LOW_CHAN = "ClockLow",
        OD_CHAN = "OD",
        HV_BIAS_CHAN = "HVBias",
        DPHI_CHAN = "DPHI",
        OTM_CHAN = "OTM",
        FAN_CHAN = "Fan",
        AUX_CHAN = "Aux";

    private static final String[] pwrChannels = new String[ATSChannels.NUM_CHANNELS];
    static {
        pwrChannels[ATSChannels.CHAN_DIGITAL] = DIGITAL_CHAN;
        pwrChannels[ATSChannels.CHAN_ANALOG] = ANALOG_CHAN;
        pwrChannels[ATSChannels.CHAN_CLK_HIGH] = CLK_HIGH_CHAN;
        pwrChannels[ATSChannels.CHAN_CLK_LOW] = CLK_LOW_CHAN;
        pwrChannels[ATSChannels.CHAN_OD] = OD_CHAN;
        pwrChannels[ATSChannels.CHAN_DPHI] = DPHI_CHAN;
        pwrChannels[ATSChannels.CHAN_HV_BIAS] = HV_BIAS_CHAN;
        pwrChannels[ATSChannels.CHAN_OTM] = OTM_CHAN;
        pwrChannels[ATSChannels.CHAN_FAN] = FAN_CHAN;
        pwrChannels[ATSChannels.CHAN_AUX] = AUX_CHAN;
    }
    private static final int POWER_TIMEOUT = 2000;  // Power on/off timeout per channel
    private static final int UPDATE_TIME = 3000;    // Power state update period
    private static final int[] rebChannels = {ATSChannels.CHAN_DIGITAL, ATSChannels.CHAN_ANALOG, ATSChannels.CHAN_OD,
                                              ATSChannels.CHAN_CLK_HIGH, ATSChannels.CHAN_CLK_LOW};

    private static final Logger LOG = Logger.getLogger(ATSPowerMain.class.getName());
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, PowerControl> pwrControls = new LinkedHashMap<>();
    
    private PowerControl dphiControl, hvBiasControl, otmControl, fanControl, auxControl;
    private final Boolean[] powerOn = new Boolean[ATSChannels.NUM_CHANNELS];
    private final PowerGroup rebGroup = new PowerGroup(), dphiGroup = new PowerGroup(), hvBiasGroup = new PowerGroup(),
                             otmGroup = new PowerGroup(), fanGroup = new PowerGroup(), auxGroup = new PowerGroup();


    /**
     *  Power subsystem initialization
     */
    @Override
    public void build()
    {
        AgentPeriodicTask pt = new AgentPeriodicTask("power-state",
                                                     () -> updatePowerState()).withPeriod(Duration.ofMillis(UPDATE_TIME));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Power subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a Power Supply controller.
        agentPropertiesService.setAgentProperty(PowerAgentProperties.ATS_POWER_AGENT, getClass().getCanonicalName());

        // Checks power control channels
        boolean error = false;
        for (String chan : pwrChannels) {
            if (pwrControls.get(chan) == null) {
                LOG.log(Level.SEVERE, "Required power channel {0} has not been defined", chan);
                error = true;
            }
        }
        if (error) {
            throw new RuntimeException("Fatal initialization error");
        }

        dphiControl = pwrControls.get(DPHI_CHAN);
        hvBiasControl = pwrControls.get(HV_BIAS_CHAN);
        otmControl = pwrControls.get(OTM_CHAN);
        fanControl = pwrControls.get(FAN_CHAN);
        auxControl = pwrControls.get(AUX_CHAN);
        for (int chan : rebChannels) {
            rebGroup.addControl(pwrControls.get(pwrChannels[chan]));
        }
        dphiGroup.addControl(dphiControl);
        hvBiasGroup.addControl(hvBiasControl);
        otmGroup.addControl(otmControl);
        fanGroup.addControl(fanControl);
        auxGroup.addControl(auxControl);
    }


    /**
     *  Starts the subsystem.
     *
     */
    @Override
    public void postStart()
    {
        LOG.info("ATS power subsystem started");
    }


    /**
     *  Sets the update (tick) period.
     *
     *  @param  period  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)")
                                int period)
    {
        setTickPeriod(period);
        publishState();    // Must do this for GUI
    }


    /**
     *  Turns on the REB power, except for HV bias.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the power")
    public void powerOn() throws PowerException
    {
        try {
            if (!isPowerOn()) {
                //System.out.println("Power on start time = " + System.currentTimeMillis());
                rebGroup.powerOn();
                rebGroup.waitPowerOn(POWER_TIMEOUT);
                //System.out.println("Power on end time = " + System.currentTimeMillis());
            }
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off all the REB power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the power")
    public void powerOff() throws PowerException
    {
        PowerException excp = null;
        //System.out.println("Power off start time = " + System.currentTimeMillis());
        try {
            hvBiasGroup.powerOff();
        }
        catch (PowerException e) {
            excp = e;
        }
        try {
            dphiGroup.powerOff();
        }
        catch (PowerException e) {
            excp = excp == null ? e : excp;
        }
        try {
            rebGroup.powerOff();
            rebGroup.waitPowerOff(POWER_TIMEOUT);
            if (excp != null) {
                throw excp;
            }
        }
        finally {
        //System.out.println("Power off end time = " + System.currentTimeMillis());
            updatePowerState();
        }
    }


    /**
     *  Turns on the DPHI voltage.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the DPHI voltage")
    public void dphiOn() throws PowerException
    {
        if (!isPowerOn()) return;
        try {
            dphiGroup.powerOn();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off the DPHI voltage.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the DPHI voltage")
    public void dphiOff() throws PowerException
    {
        try {
            dphiGroup.powerOff();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Queries the DPHI voltage state.
     *
     *  @return  Whether the DPHI voltage is on
     */
    @Command(type=CommandType.QUERY, description="Query the DPHI voltage state", level=0)
    public boolean isDphiOn()
    {
        return powerOn[ATSChannels.CHAN_DPHI] == Boolean.TRUE;
    }


    /**
     *  Sets the DPHI voltage
     *
     *  @param  value  The voltage to set
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Set the DPHI voltage")
    public void setDphi(@Argument(description="Voltage value") double value) throws PowerException
    {
        dphiControl.setVoltage(value);
        try {
            if (powerOn[ATSChannels.CHAN_DPHI] == Boolean.TRUE) {
                dphiControl.writeVoltage();
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Turns on the HV bias.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the HV bias")
    public void hvBiasOn() throws PowerException
    {
        if (!isPowerOn()) return;
        try {
            hvBiasGroup.powerOn();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off the HV bias.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the HV bias")
    public void hvBiasOff() throws PowerException
    {
        try {
            hvBiasGroup.powerOff();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Queries the HV bias voltage state.
     *
     *  @return  Whether the HV bias voltage is on
     */
    @Command(type=CommandType.QUERY, description="Query the HV bias voltage state", level=0)
    public boolean isHvBiasOn()
    {
        return powerOn[ATSChannels.CHAN_HV_BIAS] == Boolean.TRUE;
    }


    /**
     *  Sets the HV bias voltage
     *
     *  @param  value  The voltage to set
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Set the HV bias voltage")
    public void setHvBias(@Argument(description="Voltage value") double value) throws PowerException
    {
        hvBiasControl.setVoltage(-Math.abs(value));
        try {
            if (powerOn[ATSChannels.CHAN_HV_BIAS] == Boolean.TRUE) {
                hvBiasControl.writeVoltage();
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Turns on the OTM power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the OTM")
    public void otmOn() throws PowerException
    {
        try {
            otmGroup.powerOn();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off the OTM power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the OTM")
    public void otmOff() throws PowerException
    {
        try {
            otmGroup.powerOff();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns on the fan power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the fan")
    public void fanOn() throws PowerException
    {
        try {
            fanGroup.powerOn();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off the fan power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the fan")
    public void fanOff() throws PowerException
    {
        try {
            fanGroup.powerOff();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns on the auxiliary power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the aux power")
    public void auxOn() throws PowerException
    {
        try {
            auxGroup.powerOn();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off the auxiliary power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the aux power")
    public void auxOff() throws PowerException
    {
        try {
            auxGroup.powerOff();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Gets the full power state.
     *
     *  @return  The full power state
     *  @throws  PowerException
     */
    @Command(type=CommandType.QUERY, description="Get the full state", level=0)
    public ATSPowerState getFullState() throws PowerException
    {
        return new ATSPowerState(getTickPeriod(), powerOn, dphiControl.getVoltage(), hvBiasControl.getVoltage());
    }


    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the tick period
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }
    

    /**
     *  Publishes the state of the power device.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        ATSPowerState ps = new ATSPowerState(getTickPeriod(), powerOn, dphiControl.getVoltage(), hvBiasControl.getVoltage());
        KeyValueData kvd = new KeyValueData(ATSPowerState.KEY, ps);
        subsys.publishSubsystemDataOnStatusBus(kvd);
    }    


    /**
     *  Updates the power channel states.
     *
     *  Called periodically on a timer thread
     */
    private synchronized void updatePowerState()
    {
        boolean changed = false;
        for (int chan = 0; chan < ATSChannels.NUM_CHANNELS; chan++) {
            Boolean state, oldState = powerOn[chan];
            try {
                state = pwrControls.get(pwrChannels[chan]).readOutput();
            }
            catch (Exception e) {  // Undefined channel or read error
                state = null;
            }
            if (state != oldState) {
                powerOn[chan] = state;
                changed = true;
            }
        }
        if (changed) {
            publishState();
        }
    }


    /**
     *  Determines whether the non-HV REB power is fully on.
     */
    private boolean isPowerOn()
    {
        boolean isOn = true;
        for (int chan : rebChannels) {
            if (powerOn[chan] != Boolean.TRUE) {
                isOn = false;
                break;
            }
        }
        return isOn;
    }

}
