package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.auxelex.RebBulkPS;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.power.constants.PowerConfig;
import org.lsst.ccs.subsystem.power.constants.QuadBoxSwitches;

/**
 *  Interface to the SLAC REB bulk PS controller.
 *
 *  @author Owen Saxton
 */
public class RebBulkPsDevice extends Device implements SwitchControl {

    /**
     *  Constants.
     */
    private static final Logger LOG = Logger.getLogger(RebBulkPsDevice.class.getName());

    protected static final int
        MON_TYPE_VOLTAGE = 0,
        MON_TYPE_CURRENT = 1,
        MON_TYPE_TEMP = 2,
        MON_TYPE_BOARD_TEMP = 3;

    private static final String
        NODE   = "node";

    private static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
        mTypeMap.put("TEMP", MON_TYPE_TEMP);
        mTypeMap.put("BOARDTEMP", MON_TYPE_BOARD_TEMP);
    }

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name=NODE, category=PowerConfig.CATEGORY_QUADBOX, isFinal=true)
    protected volatile Integer node;

    private final RebBulkPS ps = new RebBulkPS();
    private boolean initError = false;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (node == null) {
            ErrorUtils.reportConfigError(LOG, name, NODE, "is missing");
        }
        fullName = "REB bulk PS board (" + node + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    public void initialize()
    {
        try {
            ps.open(node);
            setOnline(true);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            ps.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}:{1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  Only the type can be checked at this time.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }
        if (mType != MON_TYPE_BOARD_TEMP && (hwChan < 0 || hwChan >= RebBulkPS.NUM_BULKS)) {
            ErrorUtils.reportChannelError(LOG, name, "hwChan", hwChan);
        }
        return new int[]{mType, 0};
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        String item = null;
        if (online) {
            try {
                switch (type) {
                case MON_TYPE_VOLTAGE:
                    item = "voltage";
                    value = ps.readVoltage(hwChan);
                    break;

                case MON_TYPE_CURRENT:
                    item = "current";
                    value = ps.readCurrent(hwChan);
                    break;

                case MON_TYPE_TEMP:
                    item = "temperature";
                    value = ps.readTemperature(hwChan);
                    break;

                case MON_TYPE_BOARD_TEMP:
                    item = "board temperature";
                    value = ps.readBoardTemperature();
                    break;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0} {1}: {2}", new Object[]{fullName, item, e});
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Gets the switch device type
     * 
     *  @return  The switch device type
     */
    @Override
    public int getSwitchDevice()
    {
        return QuadBoxSwitches.DEVC_REB_BULK;
    }


    /**
     *  Turns a bulk power supply on.
     *
     *  @param  psNum  The PS number
     */
    @Override
    public void switchOn(int psNum)
    {
        try {
            ps.setPowerOn(psNum, true);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error switching on PS {0} on {1}: {2}", new Object[]{psNum, fullName, e});
        }
    }


    /**
     *  Turns a bulk power supply off.
     *
     *  @param  psNum  The PS number
     */
    @Override
    public void switchOff(int psNum)
    {
        try {
            ps.setPowerOn(psNum, false);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error switching off PS {0} on {1}: {2}", new Object[]{psNum, fullName, e});
        }
    }


    /**
     *  Gets the state of a bulk power supply.
     *
     *  @param  psNum  The PS number
     *  @return  Whether the PS is on or off
     */
    @Override
    public Boolean isSwitchOn(int psNum)
    {
        if (!online || psNum < 0) return null;
        try {
            return (ps.getIoStatus(psNum) & RebBulkPS.IOSTAT_REM_ON) != 0;
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error getting IO status for PS {0} on {1}: {2}", new Object[]{psNum, fullName, e});
            return null;
        }
    }

}
