package org.lsst.ccs.subsystem.power;

import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.drivers.commons.DriverConstants.ConnType;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;

/**
 *  Implementation of a simulated power supply device.
 *
 *  @author Owen Saxton
 */
public class SimPowerDevice extends PowerDevice {

    /**
     *  Inner class to implement a simulated power supply driver.
     *
     *  A simulated load is applied to each hardware channel
     */
    static class SimPowerSupply implements PowerSupplyDriver {

        private final int nChans;
        private final double[] loads, voltages, currents, onDelays, offDelays;
        private final boolean[] powerOn;
        private boolean open;


        /**
         *  Constructor.
         */
        public SimPowerSupply(double[] loads)
        {
            nChans = loads.length;
            this.loads = loads;
            voltages = new double[nChans];
            currents = new double[nChans];
            onDelays = new double[nChans];
            offDelays = new double[nChans];
            powerOn = new boolean[nChans];
        }


        /**
         *  Opens a connection.
         */
        @Override
        public void open(DriverConstants.ConnType connType, String id, int parm)
            throws DriverException
        {
            if (open) {
                throw new DriverException("Device already open");
            }
            open = true;
        }


        /**
         *  Closes the connection.
         */
        @Override
        public void close() throws DriverException
        {
            checkOpen();
            open = false;
        }


        /**
         *  Sets the voltage.
         *
         *  @param  voltage  The voltage to set
         *  @param  hwChan   The hardware channel number
         *  @throws  DriverException
         */
        @Override
        public void setVoltage(double voltage, int hwChan)
            throws DriverException
        {
            checkOpen();
            voltages[hwChan] = voltage;
        }


        /**
         *  Gets the set voltage.
         *
         *  @param  hwChan   The hardware channel number
         *  @return  The voltage
         *  @throws  DriverException
         */
        @Override
        public double getVoltage(int hwChan) throws DriverException
        {
            checkOpen();
            return voltages[hwChan];
        }


        /**
         *  Reads the voltage.
         *
         *  @param  hwChan   The hardware channel number
         *  @return  The voltage
         *  @throws  DriverException
         */
        @Override
        public double readVoltage(int hwChan) throws DriverException
        {
            checkOpen();
            double value = 0;
            if (powerOn[hwChan]) {
                if (loads[hwChan] > 0) {
                    value = voltages[hwChan] / loads[hwChan];
                    if (value > currents[hwChan]) {
                        value = currents[hwChan];
                    }
                    value *= loads[hwChan];
                }
                else {
                    value = voltages[hwChan];
                }
                value += 0.01 * Math.random();
            }

            return value;
        }


        /**
         *  Sets the current.
         *
         *  @param  current  The current to set
         *  @param  hwChan   The hardware channel number
         *  @throws  DriverException
         */
        @Override
        public void setCurrent(double current, int hwChan)
            throws DriverException
        {
            checkOpen();
            currents[hwChan] = current;
        }


        /**
         *  Gets the set current.
         *
         *  @param  hwChan   The hardware channel number
         *  @return  The current
         *  @throws  DriverException
         */
        @Override
        public double getCurrent(int hwChan) throws DriverException
        {
            checkOpen();
            return currents[hwChan];
        }


        /**
         *  Reads the current.
         *
         *  @param  hwChan   The hardware channel number
         *  @return  The current
         *  @throws  DriverException
         */
        @Override
        public double readCurrent(int hwChan) throws DriverException
        {
            checkOpen();
            double value = 0;
            if (powerOn[hwChan] && loads[hwChan] > 0) {
                value = voltages[hwChan] / loads[hwChan];
                if (value <= currents[hwChan]) {
                    value += 0.01 * Math.random();
                }
                else {
                    value = currents[hwChan];
                }
            }

            return value;
        }


        /**
         *  Sets the power-on delay.
         *
         *  @param  onDelay  The power-on delay (sec)
         *  @param  hwChan   The hardware channel number
         *  @throws  DriverException
         */
        @Override
        public void setOnDelay(double onDelay, int hwChan)
            throws DriverException
        {
            checkOpen();
            onDelays[hwChan] = onDelay;
        }


        /**
         *  Sets the power-off delay.
         *
         *  @param  offDelay  The power-off delay (sec)
         *  @param  hwChan    The hardware channel number
         *  @throws  DriverException
         */
        @Override
        public void setOffDelay(double offDelay, int hwChan)
            throws DriverException
        {
            checkOpen();
            offDelays[hwChan] = offDelay;
        }


        /**
         *  Sets the power output state.
         *
         *  @param  onState  The state to set: on if true; off if false
         *  @param  hwChan   The hardware channel number
         *  @throws  DriverException
         */
        @Override
        public void setOutput(boolean onState, int hwChan)
            throws DriverException
        {
            checkOpen();
            powerOn[hwChan] = onState;
        }


        /**
         *  Gets the power output state.
         *
         *  @param  hwChan   The hardware channel number
         *  @return  The power output state
         *  @throws  DriverException
         */
        @Override
        public boolean getOutput(int hwChan) throws DriverException
        {
            checkOpen();
            return powerOn[hwChan];
        }


        /**
         *  Checks whether connection is open.
         *
         *  @throws  DriverException
         */
        private void checkOpen() throws DriverException
        {
            if (!open) {
                throw new DriverException("Connection not open");
            }
        }

    }


    /**
     *  Constructor.
     *
     *  @param  loads  The array of load resistances, one per channel
     */
    public SimPowerDevice(double[] loads)
    {
        super("Simulated PS", new SimPowerSupply(loads), ConnType.NET, "", 0,
              CONN_TYPE_NOT_CONFIG | DEVC_ID_NOT_CONFIG | DEVC_PARM_NOT_CONFIG,
              0, loads.length);
    }

}