package org.lsst.ccs.subsystem.power;

import java.util.logging.Logger;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 *  Implements power supply control functions.
 *
 *  @author Owen Saxton
 */
public class PowerControlOld extends Control {

    /**
     *  Constants.
     */
    private static final Logger LOG = Logger.getLogger(PowerControlOld.class.getName());

    private static final String
        VOLTAGE   = "voltage",
        CURRENT   = "current",
        ON_DELAY  = "onDelay",
        OFF_DELAY = "offDelay";

    /**
     *  Data fields.
     */
    // Supplied configurable fields
    @ConfigurationParameter(name=VOLTAGE, category=Power.POWER)
    private double      voltage;       // Voltage
    @ConfigurationParameter(name=CURRENT, category=Power.POWER)
    private double      current;       // Current
    @ConfigurationParameter(name=ON_DELAY, category=Power.POWER)
    private double      onDelay;       // Turn-on delay
    @ConfigurationParameter(name=OFF_DELAY, category=Power.POWER)
    private double      offDelay;      // Turn-off delay

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subs;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService agentConfigurationService;
    
    
    // Derived fields
    private String      shortName;     // Last part of dot-separated name
    private PowerDevice pDevc;         // Parent (power supply) device


    /**
     *  Configures channel description.
     *
     *  @param  mon   The associated monitor
     *  @param  devc  The parent Device
     */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        pDevc = (PowerDevice)devc;
        try {
            if (hwChan < pDevc.getMinChannel() || hwChan > pDevc.getMaxChannel()) {
                ErrorUtils.reportChannelError(LOG, getName(), "hw channel number", hwChan);
            }
        }
        catch (Exception e) {
        }
        String[] parts = getName().split("\\.");
        shortName = parts[parts.length - 1];
    }


    /**
     *  Sets/gets the voltage.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setVoltage(double value)
    {
        voltage = value;
    }

    @Command(type=Command.CommandType.QUERY, description="Gets the set voltage")
    public double getVoltage()
    {
        return voltage;
    }


    /**
     *  Sets/gets the current.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setCurrent(double value)
    {
        current = value;
    }

    @Command(type=Command.CommandType.QUERY, description="Gets the set current")
    public double getCurrent()
    {
        return current;
    }


    /**
     *  Sets/gets the power-on delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setOnDelay(double value)
    {
        onDelay = value;
    }

    @Command(type=Command.CommandType.QUERY, description="Gets the power-on delay")
    public double getOnDelay()
    {
        return onDelay;
    }


    /**
     *  Sets/gets the power-off delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setOffDelay(double value)
    {
        offDelay = value;
    }

    @Command(type=Command.CommandType.QUERY, description="Gets the power-off delay")
    public double getOffDelay()
    {
        return offDelay;
    }


    /**
     *  Changes the voltage.
     *
     *  @param  value  The value to set
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.ACTION, description="Changes the voltage")
    public void changeVoltage(double value) throws PowerException
    {
        pDevc.writeVoltage(value, hwChan);
    }


    /**
     *  Sets all the configuration data.
     */
    void setConfig(Power.Channel chan)
    {
        String name = getName();
        agentConfigurationService.change(name, VOLTAGE, chan.getVoltage());
        agentConfigurationService.change(name, CURRENT, chan.getCurrent());
        agentConfigurationService.change(name, ON_DELAY, chan.getOnDelay());
        agentConfigurationService.change(name, OFF_DELAY, chan.getOffDelay());
    }


    /**
     *  Gets all configuration data.
     */
    Power.Channel getConfig()
    {
        Power.Channel chan = new Power.Channel();
        chan.setName(shortName);
        chan.setVoltage(voltage);
        chan.setCurrent(current);
        chan.setOnDelay(onDelay);
        chan.setOffDelay(offDelay);

        return chan;
    }


    /**
     *  Gets state data from the hardware.
     */
    PowerChanState getState()
    {
        PowerChanState pwr = new PowerChanState();
        pwr.setName(shortName);
        try {
            pwr.setVoltage(readVoltage());
            pwr.setCurrent(readCurrent());
            pwr.setState(readOutput() ? PowerChanState.PWR_STATE_ON : PowerChanState.PWR_STATE_OFF);
        }
        catch (PowerException e) {
            pwr.setVoltage(0);
            pwr.setCurrent(0);
            pwr.setState(PowerChanState.PWR_STATE_OFFLINE);
        }

        return pwr;
    }


    /**
     *  Writes all configured quantities to the hardware.
     */
    void writeAll() throws PowerException
    {
        writeVoltage();
        writeCurrent();
        writeOnDelay();
        writeOffDelay();
    }


    /**
     *  Writes the voltage to the hardware.
     */
    void writeVoltage() throws PowerException
    {
        pDevc.writeVoltage(voltage, hwChan);
    }


    /**
     *  Reads the voltage from the hardware.
     */
    double readVoltage() throws PowerException
    {
        return pDevc.readVoltage(hwChan);
    }


    /**
     *  Writes the current to the hardware.
     */
    void writeCurrent() throws PowerException
    {
        pDevc.writeCurrent(current, hwChan);
    }


    /**
     *  Reads the current from the hardware.
     */
    double readCurrent() throws PowerException
    {
        return pDevc.readCurrent(hwChan);
    }


    /**
     *  Writes the power-on delay to the hardware.
     */
    void writeOnDelay() throws PowerException
    {
        pDevc.writeOnDelay(onDelay, hwChan);
    }


    /**
     *  Writes the power-off delay to the hardware.
     */
    void writeOffDelay() throws PowerException
    {
        pDevc.writeOffDelay(offDelay, hwChan);
    }


    /**
     *  Writes the power output state to the hardware.
     */
    void writeOutput(boolean value) throws PowerException
    {
        pDevc.writeOutput(value, hwChan);
    }


    /**
     *  Reads the power output state from the hardware.
     */
    boolean readOutput() throws PowerException
    {
        return pDevc.readOutput(hwChan);
    }


    /**
     *  Gets the associated device
     */
    PowerDevice getDevice()
    {
        return pDevc;
    }

}
