package org.lsst.ccs.subsystem.power;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 *  Implements a power supply control channel group.
 *
 *  @author Owen Saxton
 */
public class PowerGroup {

    /**
     *  Constants.
     */
    public static final Logger LOG = Logger.getLogger(PowerGroup.class.getName());

    /**
     *  Data fields.
     */
    private final List<PowerControl> controls = new ArrayList<>();


    /**
     *  Constructor.
     *
     *  @param  controls  The members of the group
     */
    public PowerGroup(PowerControl... controls)
    {
        Collections.addAll(this.controls, controls);
    }


    /**
     *  Add a control channel to the group.
     *
     *  @param  control  The control to add
     */
    public void addControl(PowerControl control)
    {
        if (control != null) {
            controls.add(control);
        }
    }


    /**
     *  Gets the list of power state data.
     *
     *  @return  A list of power state data
     */
    public List<PowerChanState> getPowerState()
    {
        List<PowerChanState> pState = new ArrayList<>();
        for (PowerControl ctl : controls) {
            pState.add(ctl.getState());
        }
        return pState;
    }


    /**
     *  Powers on all channels.
     *
     *  @throws  PowerException
     */
    public void powerOn() throws PowerException
    {
        for (PowerControl ctl : controls) {
            ctl.writeAll();
            ctl.writeOutput(true);
        }
    }


    /**
     *  Waits for power-on to complete.
     *
     *  @param  timeout  The timeout period (ms)
     *  @throws  PowerException
     */
    public void waitPowerOn(int timeout) throws PowerException
    {
        waitOutputSet(true, timeout);
    }


    /**
     *  Powers off all channels.
     *
     *  @throws  PowerException
     */
    public void powerOff() throws PowerException
    {
        for (PowerControl ctl : controls) {
            ctl.writeOutput(false);
        }
    }


    /**
     *  Waits for power-off to complete.
     *
     *  @param  timeout  The timeout period (ms)
     *  @throws  PowerException
     */
    public void waitPowerOff(int timeout) throws PowerException
    {
        waitOutputSet(false, timeout);
    }


    /**
     *  Waits for the power output state to be set for all channels.
     *
     *  @param  value  The output state that was set
     *  @param  timeout  The maximum time to wait (ms)
     *  @throws  PowerException
     */
    private void waitOutputSet(boolean value, int timeout) throws PowerException
    {
        if (timeout <= 0) return;
        long maxTime = System.currentTimeMillis() + timeout;
        int numDone = 0;
        for (PowerControl ctl : controls) {
            while (System.currentTimeMillis() <= maxTime) {
                if (value == ctl.readOutput()) {
                    numDone++;
                    break;
                }
                try {
                    Thread.sleep(10);
                }
                catch (InterruptedException e) {}
            }
        }
        if (numDone < controls.size()) {
            throw new PowerException("Output wait timeout");
        }
    }

}
