package org.lsst.ccs.subsystem.power;

import org.lsst.ccs.drivers.bk.Model9130;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 *  Interface to a B&K model 9130 power supply device.
 *
 *  @author Owen Saxton
 */
public class BK9130Device extends PowerDevice {

    /**
     *  Data fields.
     */
    private final Model9130 ps;
    private final double[] outVoltages = new double[Model9130.NUM_CHANNELS];
    private final double[] outCurrents = new double[Model9130.NUM_CHANNELS];
    private final boolean[] outOutputs = new boolean[Model9130.NUM_CHANNELS];
    private double[] inVoltages, inCurrents;
    private boolean[] inOutputs;
    private DriverException voltageException, currentException, outputException;


    /**
     *  Constructor.
     */
    public BK9130Device()
    {
        super("B&K 9130 PS", new Model9130(), NETWORK_CONN_INVALID, 0, Model9130.MAX_CHANNEL);
        ps = (Model9130)psd;
    }


    /**
     *  Writes the voltage to the hardware.
     *
     *  @param  voltage  The voltage to write
     *  @param  hwChan   The hardware channel number
     */
    @Override
    protected void writeVoltage(double voltage, int hwChan)
    {
        //ps.setVoltage(voltage, hwChan);
        outVoltages[hwChan - Model9130.MIN_CHANNEL] = voltage;
    }


    /**
     *  Reads the voltage from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The voltage
     *  @throws  PowerException
     */
    @Override
    protected double readVoltage(int hwChan) throws PowerException
    {
        //return ps.readVoltage(hwChan);
        if (voltageException != null) {
            handleException(voltageException);
        }
        return inVoltages[hwChan - Model9130.MIN_CHANNEL];
    }


    /**
     *  Writes the current to the hardware.
     *
     *  @param  current  The current to write
     *  @param  hwChan   The hardware channel number
     */
    @Override
    protected void writeCurrent(double current, int hwChan)
    {
        //ps.setCurrent(current, hwChan);
        outCurrents[hwChan - Model9130.MIN_CHANNEL] = current;
    }


    /**
     *  Reads the current from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The current
     *  @throws  PowerException
     */
    @Override
    protected double readCurrent(int hwChan) throws PowerException
    {
        //return ps.readCurrent(hwChan);
        if (currentException != null) {
            handleException(currentException);
        }
        return inCurrents[hwChan - Model9130.MIN_CHANNEL];
    }


    /**
     *  Writes the power output state to the hardware.
     *
     *  @param  onState  The state to set: on if true; off if false
     *  @param  hwChan   The hardware channel number
     */
    @Override
    protected void writeOutput(boolean onState, int hwChan)
    {
        //ps.setOutput(onState, hwChan);
        outOutputs[hwChan - Model9130.MIN_CHANNEL] = onState;
    }


    /**
     *  Reads the power output state from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The power output state
     *  @throws  PowerException
     */
    @Override
    protected boolean readOutput(int hwChan) throws PowerException
    {
        //return ps.getOutput(hwChan);
        if (outputException != null) {
            handleException(outputException);
        }
        return inOutputs[hwChan - Model9130.MIN_CHANNEL];
    }


    /**
     *  Reads all the voltages from the hardware.
     */
    @Override
    protected void readVoltageGroup()
    {
        try {
            inVoltages = ps.readVoltage();
            voltageException = null;
        }
        catch (DriverException e) {
            voltageException = e;
        }
    }


    /**
     *  Writes all the voltages to the hardware.
     *
     *  @throws  DriverException
     */
    @Override
    protected void writeVoltageGroup() throws DriverException
    {
        ps.setVoltage(outVoltages);
    }


    /**
     *  Reads all the currents from the hardware.
     */
    @Override
    protected void readCurrentGroup()
    {
        try {
            inCurrents = ps.readCurrent();
            currentException = null;
        }
        catch (DriverException e) {
            currentException = e;
        }
    }


    /**
     *  Writes all the currents to the hardware.
     *
     *  @throws  DriverException
     */
    @Override
    protected void writeCurrentGroup() throws DriverException
    {
        ps.setCurrent(outCurrents);
    }


    /**
     *  Reads all the output states from the hardware.
     */
    @Override
    protected void readOutputGroup()
    {
        try {
            inOutputs = ps.getOutput();
            outputException = null;
        }
        catch (DriverException e) {
            outputException = e;
        }
    }


    /**
     *  Writes all the output states to the hardware.
     *
     *  @throws  DriverException
     */
    @Override
    protected void writeOutputGroup() throws DriverException
    {
        ps.setOutput(outOutputs);
    }

}
