package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.ATSPowerState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the Auxiliary Telescope power control subsystem.
 *
 *  @author Owen Saxton
 */
public class ATSPower implements HasLifecycle {

    private static final String
        HV_BIAS_CHAN = "HVBias",
        DPHI_CHAN = "DPHI";
    private static final String[] pwrChannels = new String[ATSPowerState.NUM_CHANNELS];
    static {
        pwrChannels[ATSPowerState.CHAN_DIGITAL] = "Digital";
        pwrChannels[ATSPowerState.CHAN_ANALOG] = "Analog";
        pwrChannels[ATSPowerState.CHAN_CLK_HIGH] = "ClockHigh";
        pwrChannels[ATSPowerState.CHAN_CLK_LOW] = "ClockLow";
        pwrChannels[ATSPowerState.CHAN_OD] = "OD";
        pwrChannels[ATSPowerState.CHAN_HEATER] = "Heater";
        pwrChannels[ATSPowerState.CHAN_DPHI] = DPHI_CHAN;
        pwrChannels[ATSPowerState.CHAN_HV_BIAS] = HV_BIAS_CHAN;
    }
    private static final int POWER_TIMEOUT = 2000;  // Power on/off timeout

    private static final Logger LOG = Logger.getLogger(ATSPower.class.getName());
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
        
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, PowerDevice> pwrDevices = new LinkedHashMap<>();
    
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, PowerControl> pwrControls = new LinkedHashMap<>();
    
    private PowerControl dphiControl, hvBiasControl;
    private PowerDevice hvBiasDevice;
    private final Boolean[] powerOn = new Boolean[ATSPowerState.NUM_CHANNELS];


    /**
     *  Power subsystem initialization
     */
    @Override
    public void build() {
/*
        //Define the Runnable object to be invoked periodically
        Runnable myRunnable = new Runnable() {
            @Override
            public void run() {
                //The code to be executed
                periodicBroadcast();
            }
        };

        //Create an AgentPeriodicTask that will execute the above Runnable instance
        //every second at a fixed rate
        AgentPeriodicTask periodicTask = new AgentPeriodicTask(BROADCAST_TASK, myRunnable).withPeriod(Duration.ofMillis(broadcastMillis));

        //Schedule the periodic task with the AgentPeriodicTaskService
        periodicTaskService.scheduleAgentPeriodicTask(periodicTask);
*/        
    }


    /**
     *  Power subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a Power Supply controller.
        agentPropertiesService.setAgentProperty(PowerAgentProperties.ATS_POWER_AGENT, getClass().getCanonicalName());

        // Checks power devices and control channels
        boolean error = false;
        if (pwrDevices.isEmpty()) {
            LOG.error("No power devices have been defined");
            error = true;
        }
        else {
            for (String chan : pwrChannels) {
                if (pwrControls.get(chan) == null) {
                    LOG.error("Required power channel " + chan + " has not been defined");
                    error = true;
                }
            }
        }
        if (error) {
            throw new RuntimeException("Fatal initialization error");
        }

        dphiControl = pwrControls.get(DPHI_CHAN);
        hvBiasControl = pwrControls.get(HV_BIAS_CHAN);
    }


    /**
     *  Starts the subsystem.
     *
     */
    @Override
    public void postStart()
    {
        hvBiasDevice = hvBiasControl.getDevice();
        pwrDevices.remove(hvBiasDevice.getName());
        getPowerOnState();
        //publishState();        // For any GUIs
        LOG.info("ATS power subsystem started");
    }


    /**
     *  Performs periodic power data broadcast.
     */
    public void periodicBroadcast()
    {
        if (pwrDevices.isEmpty()) return;
        ArrayList<PowerChanState> pState = new ArrayList<>();
        for (PowerDevice pDevc : pwrDevices.values()) {
            try {
                pState.addAll(pDevc.getPowerState());
            }
            catch (Exception e) {
            }
        }
        KeyValueData kvd = new KeyValueData(PowerChanState.KEY, pState);
        subsys.publishSubsystemDataOnStatusBus(kvd);
    }


    /**
     *  Sets the update (tick) period.
     *
     *  @param  period  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)")
                                int period)
    {
        setTickPeriod(period);
        publishState();    // Must do this for GUI
    }


    /**
     *  Gets the power state.
     *
     *  @return  The power state
     *  @throws  PowerException
     */
    @Command(type=CommandType.QUERY, description="Get the power state")
    public List<PowerChanState> getPowerState() throws PowerException
    {
        List<PowerChanState> pState = new ArrayList<>();
        for (PowerDevice pDevc : pwrDevices.values()) {
            pState.addAll(pDevc.getPowerState());
        }

        return pState;
    }


    /**
     *  Turns on the power, except for HV bias.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the power")
    public void powerOn() throws PowerException
    {
        PowerException excp = null;
        //System.out.println("Power on start time = " + System.currentTimeMillis());
        if (!isPowerOn()) {
            for (PowerDevice pDevc : pwrDevices.values()) {
                try {
                    pDevc.powerOn();
                }
                catch (PowerException e) {
                    excp = e;
                }
            }
            long maxTime = System.currentTimeMillis() + POWER_TIMEOUT;
            for (PowerDevice pDevc : pwrDevices.values()) {
                try {
                    pDevc.waitPowerOn((int)(maxTime - System.currentTimeMillis()));
                    //System.out.println("Power on end time = " + System.currentTimeMillis());
                }
                catch (PowerException e) {
                    excp = e;
                }
            }
        }
        getPowerOnState();
        publishState();
        if (excp != null) {
            throw excp;
        }
    }


    /**
     *  Turns off all the power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the power")
    public void powerOff() throws PowerException
    {
        PowerException excp = null;
        //System.out.println("Power off start time = " + System.currentTimeMillis());
        try {
            hvBiasDevice.powerOff();
        }
        catch (PowerException e) {
            excp = e;
        }
        for (PowerDevice pDevc : pwrDevices.values()) {
            try {
                pDevc.powerOff();
            }
            catch (PowerException e) {
                excp = e;
            }
        }
        long maxTime = System.currentTimeMillis() + POWER_TIMEOUT;
        for (PowerDevice pDevc : pwrDevices.values()) {
            try {
                pDevc.waitPowerOff((int)(maxTime - System.currentTimeMillis()));
                //System.out.println("Power off end time = " + System.currentTimeMillis());
            }
            catch (PowerException e) {
                excp = e;
            }
        }
        getPowerOnState();
        publishState();
        if (excp != null) {
            throw excp;
        }
    }


    /**
     *  Turns on the HV bias.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the HV bias")
    public void hvBiasOn() throws PowerException
    {
        if (!isPowerOn()) return;
        try {
            Boolean hvOn = powerOn[ATSPowerState.CHAN_HV_BIAS];
            if (hvOn != null && !hvOn) {
                setHvBiasValue(0);
            }
            hvBiasDevice.powerOn();
        }
        finally {
            getPowerOnState();
            publishState();
        }
    }


    /**
     *  Turns off the HV bias.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the HV bias")
    public void hvBiasOff() throws PowerException
    {
        try {
            setHvBiasValue(0);
            hvBiasDevice.powerOff();
        }
        finally {
            getPowerOnState();
            publishState();
        }
    }


    /**
     *  Sets the DPHI voltage
     *
     *  @param  value  The voltage to set
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Set the DPHI voltage")
    public void setDphi(@Argument(description="Voltage value") double value) throws PowerException
    {
        try {
            setDphiValue(value);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets the HV bias voltage
     *
     *  @param  value  The voltage to set
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Set the HV bias voltage")
    public void setHvBias(@Argument(description="Voltage value") double value) throws PowerException
    {
        try {
            setHvBiasValue(value);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Gets the full power state.
     *
     *  @return  The full power state
     *  @throws  PowerException
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public ATSPowerState getFullState() throws PowerException
    {
        return new ATSPowerState(getTickPeriod(), powerOn, dphiControl.getVoltage(), hvBiasControl.getVoltage());
    }


    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the tick period
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }
    

    /**
     *  Publishes the state of the power device.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        ATSPowerState ps = new ATSPowerState(getTickPeriod(), powerOn, dphiControl.getVoltage(),
                                             hvBiasControl.getVoltage());
        KeyValueData kvd = new KeyValueData(ATSPowerState.KEY, ps);
        subsys.publishSubsystemDataOnStatusBus(kvd);
    }    


    /**
     *  Gets the power channel states.
     */
    private void getPowerOnState()
    {
        for (int chan = 0; chan < ATSPowerState.NUM_CHANNELS; chan++) {
            try {
                powerOn[chan] = pwrControls.get(pwrChannels[chan]).readOutput();
            }
            catch (Exception e) {
                powerOn[chan] = null;
            }
        }
    }


    /**
     *  Determines whether the non-HV power is on.
     */
    private boolean isPowerOn()
    {
        boolean isOn = true;
        for (int chan = 0; chan < ATSPowerState.NUM_CHANNELS; chan++) {
            if (chan == ATSPowerState.CHAN_HV_BIAS) continue;
            if (powerOn[chan] == null || !powerOn[chan]) {
                isOn = false;
                break;
            }
        }
        return isOn;
    }


    /**
     *  Sets the HV bias value.
     */
    private void setHvBiasValue(double value) throws PowerException
    {
        hvBiasControl.setVoltage(-Math.abs(value));
        hvBiasControl.writeVoltage();
        hvBiasControl.writeOutput(true);
    }


    /**
     *  Sets the DPHI value.
     */
    private void setDphiValue(double value) throws PowerException
    {
        dphiControl.setVoltage(value);
        dphiControl.writeVoltage();
        dphiControl.writeOutput(true);
    }

}
