package org.lsst.ccs.subsystem.power;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Interface to an ASCII-commanded power supply device.
 *
 *  @author Owen Saxton
 */
public abstract class PowerDevice extends Device {

    /**
     *  Constants.
     */
    protected static final Logger LOG = Logger.getLogger(PowerDevice.class.getName());

    protected static final int
        NETWORK_CONN_INVALID = 0x01,
        CONN_TYPE_NOT_CONFIG = 0x02,
        DEVC_ID_NOT_CONFIG   = 0x04,
        DEVC_PARM_NOT_CONFIG = 0x08;

    private static final int
        MON_TYPE_VOLTAGE = 0,
        MON_TYPE_CURRENT = 1;

    private static final String
        CONN_TYPE = "connType",
        DEVC_ID   = "devcId",
        DEVC_PARM = "devcParm";

    private static final Map<String, DriverConstants.ConnType> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put(Power.ConnType.Network.name().toLowerCase(), DriverConstants.ConnType.NET);
        cTypeMap.put(Power.ConnType.Ftdi.name().toLowerCase(), DriverConstants.ConnType.FTDI);
        cTypeMap.put(Power.ConnType.Serial.name().toLowerCase(), DriverConstants.ConnType.SERIAL);
    }

    private static final Map<DriverConstants.ConnType, Power.ConnType> cTypeEMap = new HashMap();
    static {
        cTypeEMap.put(DriverConstants.ConnType.NET, Power.ConnType.Network);
        cTypeEMap.put(DriverConstants.ConnType.FTDI, Power.ConnType.Ftdi);
        cTypeEMap.put(DriverConstants.ConnType.SERIAL, Power.ConnType.Serial);
    }

    private static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
    }

    /**
     *  Data fields.
     */
    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @ConfigurationParameter(name=DEVC_ID, category=Power.POWER, isFinal=true)
    protected String devcId;
    @ConfigurationParameter(name=DEVC_PARM, category=Power.POWER, isFinal=true)
    protected int devcParm;
    @ConfigurationParameter(name=CONN_TYPE, category=Power.POWER, isFinal=true)
    protected String connType;

    private final String devcName;
    protected PowerSupplyDriver psd;
    private DriverConstants.ConnType connTypeE;
    private final int options, minChan, maxChan;


    /**
     *  Constructor.
     *
     *  @param  name      The device name
     *  @param  psd       The power supply device driver object
     *  @param  options   The options word
     *  @param  minChan   The minimum channel number
     *  @param  maxChan   The maximum channel number
     */
    public PowerDevice(String name, PowerSupplyDriver psd, int options, int minChan, int maxChan)
    {
        super();
        this.devcName = name;
        this.psd = psd;
        this.options = options;
        this.minChan = minChan;
        this.maxChan = maxChan;
    }


    /**
     *  Constructor.
     *
     *  @param  name      The device name
     *  @param  psd       The power supply device driver object
     *  @param  connType  The connection type: network; ftdi; serial
     *  @param  devcId    The device ID: host name; USB serial no.; serial device name
     *  @param  devcParm  The device parameter: port number; baud rate; baud rate
     *  @param  options   The options word
     *  @param  minChan   The minimum channel number
     *  @param  maxChan   The maximum channel number
     */
    @Deprecated
    public PowerDevice(String name, PowerSupplyDriver psd, String connType, String devcId,
                       int devcParm, int options, int minChan, int maxChan)
    {
        super();
        this.devcName = name;
        this.psd = psd;
        this.connType = connType;
        this.devcId = devcId;
        this.devcParm = devcParm;
        this.options = options;
        this.minChan = minChan;
        this.maxChan = maxChan;
    }


    /**
     *  Gets the minimum channel number.
     *
     *  @return  The minimum channel number
     */
    public int getMinChannel()
    {
        return minChan;
    }


    /**
     *  Gets the maximum channel number.
     *
     *  @return  The maximum channel number
     */
    public int getMaxChannel()
    {
        return maxChan;
    }

   
    /**
     *  Gets the connection type.
     *
     *  @return  The connection type string
     */
    @Command(type=CommandType.QUERY, description="Gets the connection type")
    public String getConnType()
    {
        return connType;
    }


    /**
     *  Gets the device ID.
     *
     *  @return  The device identification string
     */
    @Command(type=CommandType.QUERY, description="Gets the device ID")
    public String getDevcId()
    {
        return devcId;
    }


    /**
     *  Gets the device parameter.
     *
     *  @return  The device parameter
     */
    @Command(type=CommandType.QUERY, description="Gets the device parameter")
    public int getDevcParm()
    {
        return devcParm;
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (connType == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "connType", "is missing");
        }
        DriverConstants.ConnType cType = cTypeMap.get(connType.toLowerCase());
        if (cType == null || (cType == DriverConstants.ConnType.NET
                              && (options & NETWORK_CONN_INVALID) != 0)) {
            MonitorLogUtils.reportConfigError(LOG, name, "connType", "is invalid");
        }
        connTypeE = cType;
        if (devcId == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        fullName = devcName + (devcId.isEmpty() ? "" : " (" + devcId + ")");
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            psd.open(connTypeE, devcId, devcParm);
            setOnline(true);
            initSensors();
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            psd.close();
        }
        catch (DriverException e) {
        }
    }


    /**
     *  Re-initializes the device.
     *
    protected void reInitialize()
    {
        if (inited) {
            close();
            inited = false;
            initialize();
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            MonitorLogUtils.reportError(LOG, name, "type", type);
        }
        if (hwChan < minChan || hwChan > maxChan) {
            MonitorLogUtils.reportError(LOG, name, "hw channel number", hwChan);
        }
        
        return new int[]{mType, 0};
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        if (online) {
            try {
                value = (type == MON_TYPE_VOLTAGE) ? psd.readVoltage(hwChan) : psd.readCurrent(hwChan);
            }
            catch (DriverException e) {
                LOG.error("Error reading from " + fullName + ": " + e);
                setOnline(false);
            }
        }

        return value;
    }


    /**
     *  Gets the configuration data.
     *
     *  @return  A power configuration object
     */
    public Power getPowerConfig()
    {
        Power power = new Power();
        power.setName(name);
        power.setConnType(cTypeEMap.get(connType));
        power.setDevcId(devcId);
        power.setDevcParm(devcParm);
        Power.Channel[] chans = new Power.Channel[ctlChans.size()];
        int j = 0;
        for (Control ctl : ctlChans.values()) {
            chans[j++] = ((PowerControl)ctl).getConfig();
        }
        power.setChannels(chans);

        return power;
    }


    /**
     *  Sets the configuration data.
     *
     *  @param  power  A power configuration object
     *  @throws  PowerException
     */
    public void setPowerConfig(Power power) throws PowerException
    {
        /*
        if ((options & CONN_TYPE_NOT_CONFIG) == 0) {
            sce.change(name, CONN_TYPE, power.getConnType().name());
        }
        if ((options & DEVC_ID_NOT_CONFIG) == 0) {
            sce.change(name, DEVC_ID, power.getDevcId());
        }
        if ((options & DEVC_PARM_NOT_CONFIG) == 0) {
            sce.change(name, DEVC_PARM, power.getDevcParm());
        }
        */
        for (Power.Channel chan : power.getChannels()) {
            String cName = name + "." + chan.getName();
            ((PowerControl)ctlChans.get(cName)).setConfig(chan);
        }
    }


    /**
     *  Gets the list of power state data.
     *
     *  @return  A list of power state data
     */
    protected List<PowerChanState> getPowerState()
    {
        readVoltageGroup();
        readCurrentGroup();
        readOutputGroup();
        List<PowerChanState> pState = new ArrayList<>();
        for (Control ctl : ctlChans.values()) {
            pState.add(((PowerControl)ctl).getState());
        }
        return pState;
    }


    /**
     *  Powers on all channels.
     *
     *  @throws  PowerException
     */
    protected void powerOn() throws PowerException
    {
        for (Control ctl : ctlChans.values()) {
            ((PowerControl)ctl).writeAll();
        }
        try {
            writeVoltageGroup();
            writeCurrentGroup();
            writeOnDelayGroup();
            writeOffDelayGroup();
        }
        catch (DriverException e) {
            handleException(e);
        }
        setOutput(true);
    }


    /**
     *  Waits for power-on to complete.
     *
     *  @param  timeout  The timeout period (ms)
     *  @throws  PowerException
     */
    protected void waitPowerOn(int timeout) throws PowerException
    {
        waitOutputSet(true, timeout);
    }


    /**
     *  Powers off all channels.
     *
     *  @throws  PowerException
     */
    protected void powerOff() throws PowerException
    {
        setOutput(false);
    }


    /**
     *  Waits for power-off to complete.
     *
     *  @param  timeout  The timeout period (ms)
     *  @throws  PowerException
     */
    protected void waitPowerOff(int timeout) throws PowerException
    {
        waitOutputSet(false, timeout);
    }


    /**
     *  Writes the voltage to the hardware.
     *
     *  @param  voltage  The voltage to set
     *  @param  hwChan   The hardware channel number
     *  @throws  PowerException
     */
    protected void writeVoltage(double voltage, int hwChan) throws PowerException
    {
        try {
            psd.setVoltage(voltage, hwChan);
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Reads the voltage from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The read voltage
     *  @throws  PowerException
     */
    protected double readVoltage(int hwChan) throws PowerException
    {
        try {
            return psd.readVoltage(hwChan);
        }
        catch (DriverException e) {
            handleException(e);
            return 0;
        }
    }


    /**
     *  Writes the current to the hardware.
     *
     *  @param  current  The current to set
     *  @param  hwChan   The hardware channel number
     *  @throws  PowerException
     */
    protected void writeCurrent(double current, int hwChan) throws PowerException
    {
        try {
            psd.setCurrent(current, hwChan);
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Reads the current from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The read current
     *  @throws  PowerException
     */
    protected double readCurrent(int hwChan) throws PowerException
    {
        try {
            return psd.readCurrent(hwChan);
        }
        catch (DriverException e) {
            handleException(e);
            return 0;
        }
    }


    /**
     *  Writes the power-on delay to the hardware.
     *
     *  @param  onDelay  The delay to set
     *  @param  hwChan   The hardware channel number
     *  @throws  PowerException
     */
    protected void writeOnDelay(double onDelay, int hwChan) throws PowerException
    {
        try {
            psd.setOnDelay(onDelay, hwChan);
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Writes the power-off delay to the hardware.
     *
     *  @param  offDelay  The delay to set
     *  @param  hwChan    The hardware channel number
     *  @throws  PowerException
     */
    protected void writeOffDelay(double offDelay, int hwChan) throws PowerException
    {
        try {
            psd.setOffDelay(offDelay, hwChan);
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Writes the power output state to the hardware.
     *
     *  @param  value   The output state to set
     *  @param  hwChan  The hardware channel number
     *  @throws  PowerException
     */
    protected void writeOutput(boolean value, int hwChan) throws PowerException
    {
        try {
            psd.setOutput(value, hwChan);
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Reads the power output state from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The power output state
     *  @throws  PowerException
     */
    protected boolean readOutput(int hwChan) throws PowerException
    {
        try {
            return psd.getOutput(hwChan);
        }
        catch (DriverException e) {
            handleException(e);
            return false;
        }
    }


    /**
     *  Sets the power output state for all channels.
     *
     *  @param  value  The output state to set
     *  @throws  PowerException
     */
    private void setOutput(boolean value) throws PowerException
    {
        try {
            for (Control ctl : ctlChans.values()) {
                psd.setOutput(value, ctl.getHwChan());
            }
            writeOutputGroup();
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Waits for the power output state to be set for all channels.
     *
     *  @param  value  The output state that was set
     *  @param  timeout  The maximum time to wait (ms)
     *  @throws  PowerException
     */
    private void waitOutputSet(boolean value, int timeout) throws PowerException
    {
        if (timeout <= 0) return;
        long maxTime = System.currentTimeMillis() + timeout;
        try {
            readOutputGroup();
            for (Control ctl : ctlChans.values()) {
                while (System.currentTimeMillis() <= maxTime) {
                    if (value == psd.getOutput(ctl.getHwChan())) return;
                    try {
                        Thread.sleep(10);
                    }
                    catch (InterruptedException e) {}
                }
            }
            throw new DriverTimeoutException("Output wait timeout");
        }
        catch (DriverException e) {
            handleException(e);
        }
    }


    /**
     *  Writes a group of voltages to the hardware.
     *
     *  May be overridden for devices with this capability
     *
     *  @throws  DriverException
     */
    protected void writeVoltageGroup() throws DriverException
    {
    }


    /**
     *  Reads a group of voltages from the hardware.
     *
     *  May be overridden for devices with this capability
     */
    protected void readVoltageGroup()
    {
    }


    /**
     *  Writes a group of currents to the hardware.
     *
     *  May be overridden for devices with this capability
     *
     *  @throws  DriverException
     */
    protected void writeCurrentGroup() throws DriverException
    {
    }


    /**
     *  Reads a group of currents from the hardware.
     *
     *  May be overridden for devices with this capability
     */
    protected void readCurrentGroup()
    {
    }


    /**
     *  Writes a group of power-on delays to the hardware.
     *
     *  May be overridden for devices with this capability
     *
     *  @throws  DriverException
     */
    protected void writeOnDelayGroup() throws DriverException
    {
    }


    /**
     *  Writes a group of power-off delays to the hardware.
     *
     *  May be overridden for devices with this capability
     *
     *  @throws  DriverException
     */
    protected void writeOffDelayGroup() throws DriverException
    {
    }


    /**
     *  Writes a group of power output states to the hardware.
     *
     *  May be overridden for devices with this capability
     *
     *  @throws  DriverException
     */
    protected void writeOutputGroup() throws DriverException
    {
    }


    /**
     *  Reads a group of power output states from the hardware.
     *
     *  May be overridden for devices with this capability
     */
    protected void readOutputGroup()
    {
    }


    /**
     *  Handle driver exceptions.
     * 
     * @param e The exception
     * @throws PowerException 
     */
    protected void handleException(DriverException e) throws PowerException
    {
        if (e instanceof DriverTimeoutException) {
            LOG.error("Timeout while accessing the power supply");
            setOnline(false);
        }
        throw new PowerException(e.toString());
    }

}
