package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.auxelex.RebPS;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 *  Interface to a REB power supply board.
 *
 *  @author Owen Saxton
 */
public class RebPsDevice extends Device {

    /**
     *  Inner class for holding power supply parameters.
     */
    static class SeqParams {

        final int    psNum;
        final int    rebOff;
        final double minGood;
        final double maxGood;

        SeqParams(int psNum, double minGood, double maxGood)
        {
            this(psNum, 0, minGood, maxGood);
        }

        SeqParams(int psNum, int rebOff, double minGood, double maxGood)
        {
            this.psNum = psNum;
            this.rebOff = rebOff;
            this.minGood = minGood;
            this.maxGood = maxGood;
        }

    }

    /**
     *  Interface for open/close event listener.
     */
    public interface Event {

        public void opened();

        public void closed();

    }

    /**
     *  Public constants.
     */
    public enum PsName {
        DIGITAL(RebPS.PS_DIGITAL),
        ANALOG(RebPS.PS_ANALOG),
        CLOCKHI(RebPS.PS_CLK_HIGH),
        CLOCKLO(RebPS.PS_CLK_LOW),
        OD(RebPS.PS_OD),
        HEATER(RebPS.PS_HEATER),
        DPHI(RebPS.PS_DPHI);

        int value;

        PsName(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }
    }

    public enum ChanName {
        VBEFLDO(RebPS.CHAN_VOLT_BEF_LDO),
        VAFTLDO(RebPS.CHAN_VOLT_AFT_LDO),
        IBEFLDO(RebPS.CHAN_CURR_BEF_LDO);

        int value;

        ChanName(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }
    }

    /**
     *  Private constants.
     */
    private static final String
        UNKN_PSID = "PS-XX";

    private static final int
        PS_POWER = 255;

    private static final double
        LIM_LOW_DIGITAL = 4.5,
        LIM_HIGH_DIGITAL = 6.0,
        LIM_LOW_ANALOG = 6.5,
        LIM_HIGH_ANALOG = 8.5,
        LIM_LOW_CLKL_SR = 12.0,
        LIM_HIGH_CLKL_SR = 17.0,
        LIM_LOW_CLKL_CR = 10.0,
        LIM_HIGH_CLKL_CR = 12.0,
        LIM_LOW_CLKH_SR = 15.0,
        LIM_HIGH_CLKH_SR = 17.0,
        LIM_LOW_CLKH_CR = 10.0,
        LIM_HIGH_CLKH_CR = 12.0,
        LIM_LOW_OD_SR = 35.0,
        LIM_HIGH_OD_SR = 42.0,
        LIM_LOW_OD_CR = 35.0,
        LIM_HIGH_OD_CR = 37.0,
        LIM_LOW_HEATER_SR = 7.0,
        LIM_HIGH_HEATER_SR = 13.0,
        LIM_LOW_HEATER_CR = 5.0,
        LIM_HIGH_HEATER_CR = 7.0,
        LIM_LOW_DPHI = 4.0,
        LIM_HIGH_DPHI = 13.0;

    private static final Map<String, Integer> psMap = new HashMap<>();
    static {
        psMap.put("DIGITAL", RebPS.PS_DIGITAL);
        psMap.put("ANALOG", RebPS.PS_ANALOG);
        psMap.put("CLOCKHI", RebPS.PS_CLK_HIGH);
        psMap.put("CLOCKLO", RebPS.PS_CLK_LOW);
        psMap.put("DPHI", RebPS.PS_DPHI);
        psMap.put("HEATER", RebPS.PS_HEATER);
        psMap.put("HVBIAS", RebPS.PS_HV_BIAS);
        psMap.put("OD", RebPS.PS_OD);
        psMap.put("POWER", PS_POWER);
    }

    private static final SeqParams[]
        onSeqSR02 = {new SeqParams(RebPS.PS_DIGITAL, LIM_LOW_DIGITAL, LIM_HIGH_DIGITAL),
                     new SeqParams(RebPS.PS_ANALOG, LIM_LOW_ANALOG, LIM_HIGH_ANALOG),
                     new SeqParams(RebPS.PS_CLK_LOW, LIM_LOW_CLKL_SR, LIM_HIGH_CLKL_SR),
                     new SeqParams(RebPS.PS_CLK_HIGH, LIM_LOW_CLKH_SR, LIM_HIGH_CLKH_SR),
                     new SeqParams(RebPS.PS_OD, LIM_LOW_OD_SR, LIM_HIGH_OD_SR),
                     new SeqParams(RebPS.PS_HEATER, LIM_LOW_HEATER_SR, LIM_HIGH_HEATER_SR)};

    private static final SeqParams[]
        onSeqSR1 = {new SeqParams(RebPS.PS_DIGITAL, LIM_LOW_DIGITAL, LIM_HIGH_DIGITAL),
                    new SeqParams(RebPS.PS_ANALOG, LIM_LOW_ANALOG, LIM_HIGH_ANALOG),
                    new SeqParams(RebPS.PS_CLK_LOW, LIM_LOW_CLKL_SR, LIM_HIGH_CLKL_SR),
                    new SeqParams(RebPS.PS_CLK_HIGH, LIM_LOW_CLKH_SR, LIM_HIGH_CLKH_SR),
                    new SeqParams(RebPS.PS_OD, LIM_LOW_OD_SR, LIM_HIGH_OD_SR)};

    private static final SeqParams[]
        onSeqCR0 = {new SeqParams(RebPS.PS_DIGITAL, LIM_LOW_DIGITAL, LIM_HIGH_DIGITAL),
                    new SeqParams(RebPS.PS_ANALOG, LIM_LOW_ANALOG, LIM_HIGH_ANALOG),
                    new SeqParams(RebPS.PS_CLK_LOW, LIM_LOW_CLKL_CR, LIM_HIGH_CLKL_CR),
                    new SeqParams(RebPS.PS_CLK_HIGH, LIM_LOW_CLKH_CR, LIM_HIGH_CLKH_CR),
                    new SeqParams(RebPS.PS_OD, LIM_LOW_OD_CR, LIM_HIGH_OD_CR),
                    new SeqParams(RebPS.PS_DIGITAL, 2, LIM_LOW_HEATER_CR, LIM_HIGH_HEATER_CR)};

    private static final SeqParams[]
        onSeqCR1 = {new SeqParams(RebPS.PS_DIGITAL, LIM_LOW_DIGITAL, LIM_HIGH_DIGITAL),
                    new SeqParams(RebPS.PS_ANALOG, LIM_LOW_ANALOG, LIM_HIGH_ANALOG),
                    new SeqParams(RebPS.PS_CLK_LOW, LIM_LOW_CLKL_CR, LIM_HIGH_CLKL_CR),
                    new SeqParams(RebPS.PS_CLK_HIGH, LIM_LOW_CLKH_CR, LIM_HIGH_CLKH_CR),
                    new SeqParams(RebPS.PS_OD, LIM_LOW_OD_CR, LIM_HIGH_OD_CR)};
/*
    private static final SeqParams[]
        onSeqCRL = {new SeqParams(RebPs.PS_DIGITAL, 5.0, 7.0),
                    new SeqParams(RebPs.PS_ANALOG, 5.0, 7.0)};

    private static final int[]
        offSeqSR = {RebPs.PS_HV_BIAS, RebPs.PS_HEATER, RebPs.PS_OD, RebPs.PS_CLK_HIGH,
                    RebPs.PS_CLK_LOW, RebPs.PS_ANALOG, RebPs.PS_DIGITAL};

    private static final int[]
        offSeqCR = {RebPs.PS_HV_BIAS, RebPs.PS_DPHI, RebPs.PS_OD, RebPs.PS_CLK_HIGH,
                    RebPs.PS_CLK_LOW, RebPs.PS_ANALOG, RebPs.PS_DIGITAL};

    private static final int[]
        offSeqCRL = {RebPs.PS_ANALOG, RebPs.PS_DIGITAL};
*/

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name="ipAddr", category=Power.POWER, isFinal=true)
    private String ipAddr;

    private int psType = RebPS.TYPE_UNKNOWN;
    private final RebPS ps = new RebPS();
    private int psVersion = RebPS.VERSION_PROD;
    private Event listener;
    private String psId = UNKN_PSID;
    private Properties props;
    private boolean isSimulated;
    private double[] power = new double[RebPS.MAX_REBS];
    private int lastFail = 0;
    private int nTimeout = 0;


    /**
     *  Performs configuration checking.
     */
    @Override
    protected void initDevice()
    {
        isSimulated = BootstrapResourceUtils.getBootstrapSystemProperties().
                        getProperty("org.lsst.ccs.run.mode", "").equals("simulation");
        String fileName = "psid.properties";
        if (BootstrapResourceUtils.getBootstrapResource(fileName) == null) {
            log.warn("PS ID properties file (" + fileName + ") not found");
        }
        props = BootstrapResourceUtils.getBootstrapProperties(fileName);
        if (ipAddr == null && !isSimulated) {
            MonitorLogUtils.reportConfigError(log, name, "ipAddr", "is missing");
        }
        fullName = "REB PS board (" + (isSimulated ? "simulated" : ipAddr) + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            ps.open(isSimulated ? "" : ipAddr);
            ps.configTemperature();
            psVersion = ps.getVersion();
            psType = ps.getType();
            generatePsId();
            setOnline(true);
            initSensors();
            if (listener != null) {
                listener.opened();
            }
            log.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        if (listener != null) {
            listener.closed();
        }
        psId = UNKN_PSID;
        try {
            ps.close();
        }
        catch (DriverException e) {
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containg the encoded type [0] and subtype [1]
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer rebNum = null, psNum = null;
        String[] flds = type.split(":");
        if (flds.length == 1) {
            if (flds[0].toUpperCase().equals("TEMP")) {
                psNum = -1;
                rebNum = 0;
            }
        }
        else if (flds.length == 2) {
            try {
                rebNum = Integer.valueOf(flds[0]);
                psNum = psMap.get(flds[1].toUpperCase());
            }
            catch (NumberFormatException e) {
            }
        }
        if (psNum == null) {
            String text = rebNum == null ? "type (REB number)" : "type (PS name)";
            MonitorLogUtils.reportError(log, name, text, type);
        }
        if (psNum >= 0 && psNum != PS_POWER && !RebPS.testChannelNumber(psNum, hwChan)) {
            MonitorLogUtils.reportError(log, name, "hwChan", hwChan);
        }
        
        return new int[]{(rebNum << 8) | psNum, 0};
    }


    /**
     *  Initializes a monitoring channel.
     *
     *  @param  name     The channel name
     *  @param  id       The channel id
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     */
    @Override
    protected void initChannel(String name, int id, int hwChan, int type, int subtype)
    {
        try {
            if (type < 0) {
                if (hwChan >= ps.getNumTemperatures()) {
                    MonitorLogUtils.reportError(log, name, "hwChan", hwChan);
                }
            }
            else {
                int rebNum = type >> 8, psNum = type & 0xff;
                if (psNum != PS_POWER && !ps.testRebNumber(rebNum)) {
                    MonitorLogUtils.reportError(log, name, "REB number", rebNum);
                }
            }
        }
        catch (Exception e) {
            dropChannel(id);
        }
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        if (online) {
            try {
                if (type < 0) {
                    value = ps.readTemperature(hwChan);
                }
                else {
                    int rebNum = type >> 8, psNum = type & 0xff;
                    if (psNum == PS_POWER) {
                        if (rebNum >= RebPS.MAX_REBS) {
                            value = 0.0;
                            for (int j = 0; j < ps.getNumRebs(); j++) {
                                value += power[j];
                            }
                        }
                        else {
                            value = 0.0;
                            for (psNum = 0; psNum < RebPS.NUM_PS; psNum++) {
                                if (psNum == RebPS.PS_HV_BIAS || !ps.isPowerOn(rebNum, psNum)) continue;
                                double volts = ps.readChannel(rebNum, psNum, RebPS.CHAN_VOLT_AFT_LDO);
                                if (psNum == RebPS.PS_CLK_LOW) {
                                    volts = ps.readChannel(rebNum, psNum, RebPS.CHAN_VOLT_BEF_LDO) - volts;
                                }
                                value += volts * ps.readChannel(rebNum, psNum, RebPS.CHAN_CURR_BEF_LDO);
                            }
                            power[rebNum] = value;
                        }
                    }
                    else if (psNum != RebPS.PS_CLK_LOW || hwChan != RebPS.CHAN_VOLT_AFT_SW
                               || ps.isPowerOn(rebNum, psNum)) {   // Clock low V after switch is meaningless if powered off
                        value = ps.readChannel(rebNum, psNum, hwChan);
                    }
                }
                nTimeout = 0;
            }
            catch (DriverTimeoutException e) {
                log.error("Error reading from " + fullName + ": " + e);
                if (++nTimeout >= 2) {
                    setOnline(false);  // No power to the supply
                }
            }
            catch (DriverException e) {
                // Channel is not powered
            }
        }

        return value;
    }


    /**
     *  Checks whether a power supply has tripped.
     *
     *  @return  Whether a trip has occurred
     */
    public boolean checkPsTripped()
    {
        try {
            int nReb = ps.getNumRebs();
            int fail = ps.getFailureSummary() & ((1 << nReb) - 1);
            if (fail == lastFail) return false;
            lastFail = fail;
            if (fail == 0) return false;
            for (int reb = 0; reb < nReb; reb++) {
                if ((fail & (1 << reb)) != 0) {
                    log.error(String.format("REB PS %s tripped: failure details = 0x%08x",
                                            reb, ps.getFailureDetail(reb)));
                }
            }
            return true;
        }
        catch (DriverException e) {
            return false;  // Ignore it
        }
    }


    /**
     *  Gets the power state data.
     *
     *  @return  A list of power state data
     */
    public int[] getState()
    {
        int[] state = new int[ps.getNumRebs()];
        for (int reb = 0; reb < state.length; reb++) {
            state[reb] = 0x80000000;
            try {
                if (isOnline()) {
                    state[reb] = ps.getPower(reb);
                }
            }
            catch (DriverException e) {
            }
        }
        return state;
    }


    /**
     *  Gets a set of DPHI or HV bias DAC values.
     *
     *  @param  getHV  If true, get HV Bias DACs; otherwise get DPHI DACs 
     *  @return  An array of DAC values
     */
    public double[] getDacs(boolean getHV)
    {
        double[] dacs = new double[ps.getNumRebs()];
        int psNum = getHV ? RebPS.PS_HV_BIAS : RebPS.PS_DPHI;
        for (int reb = 0; reb < dacs.length; reb++) {
            dacs[reb] = Double.NaN;
            try {
                if (isOnline()) {
                    dacs[reb] = ps.readChannel(reb, psNum, RebPS.CHAN_VOLT_DAC);
                }
            }
            catch (DriverException e) {
            }
        }
        return dacs;
    }


   /**
     *  Sets a HV bias DAC value.
     *
     *  @param  rebNum  The REB number
     *  @param  value   The value to set
     *  @throws  PowerException
     */
    public void setBiasDac(int rebNum, double value) throws PowerException
    {
        try {
            ps.writeDac(rebNum, RebPS.PS_HV_BIAS, value);
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Sets a DPHI DAC value.
     *
     *  @param  rebNum  The REB number
     *  @param  value   The value to set
     *  @throws  PowerException
     */
    public void setDphiDac(int rebNum, double value) throws PowerException
    {
        try {
            ps.writeDac(rebNum, RebPS.PS_DPHI, value);
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Turns a power supply on or off.
     *
     *  Currently the only allowed operations are turning the master on or off,
     *  the bias HV supply on, or the CR DPHI supply on; all others are ignored.
     *
     *  @param  reb    The REB number
     *  @param  psNum  The power supply number, or -1 for master switch
     *  @param  on     Turn on if true, off if false
     *  @throws  PowerException
     */
    public void setPowerOn(int reb, int psNum, boolean on) throws PowerException
    {
        if (psNum >= 0 && psNum != RebPS.PS_HV_BIAS
              && !(psType == RebPS.TYPE_CORNER && psNum == RebPS.PS_DPHI)) return;
        try {
            if (psNum < 0) {
                ps.setPower(reb, on ? 1 : 0);
            }
            else {  // HV bias or DPHI
                System.out.println("Setting power on: " + psNum + " " + on);
                if (psType == RebPS.TYPE_CORNER && (reb % RebPS.REB_QUANTUM) == RebPS.REB_QUANTUM - 1) return;
                int value = ps.getPower(reb);
                if ((value & 1) == 0) return;  // Not powered
                int mask = 1 << (psNum + 1);
                int newValue = on ? value | mask : value & ~mask;
                if ((mask & (value ^ newValue)) == 0) return;  // No change
                ps.setPower(reb, newValue);
                //ps.writeDac(reb, psNum, psNum == RebPs.PS_HV_BIAS ? 0 : RebPs.DAC_MAXIMUM);
            }
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Toggles the on/off state of a power supply.
     *
     *  @param  reb    The REB number
     *  @param  psNum  The power supply number, or -1 for master switch
     *  @throws  PowerException
     */
    public void togglePower(int reb, int psNum) throws PowerException
    {
        try {
            int psn = (psNum < 0) ? 0 : psNum + 1;
            setPowerOn(reb, psNum, ((ps.getPower(reb) >> psn) & 1) == 0);
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Sequences a power supply section on or off.
     *
     *  @param  reb  The REB number
     *  @param  on   Turns on if true, off if false
     *  @throws  PowerException
     */
    public void sequencePower(int reb, boolean on) throws PowerException
    {
        if (psVersion == RebPS.VERSION_PROD) {
            setPowerOn(reb, -1, on);  // PS does the sequencing
            return;
        }
        try {
            SeqParams[] onSeq;
            if (psType == RebPS.TYPE_CORNER) {
                if (reb == RebPS.REB_QUANTUM - 1) return;
                onSeq = reb == 0 ? onSeqCR0 : onSeqCR1;
            }
            else {
                onSeq = reb == 1 ? onSeqSR1 : onSeqSR02;
            }
            if (on) {
                ps.writeDac(reb, RebPS.PS_HV_BIAS, 0);
                if (psType == RebPS.TYPE_CORNER) {
                    ps.writeDac(reb, RebPS.PS_DPHI, RebPS.DAC_MAXIMUM);
                }
                for (SeqParams parm : onSeq) {
                    int effReb = reb + parm.rebOff;
                    int value = ps.getPower(effReb);
                    if ((value & 1) == 0) {
                        value |= 1;
                        ps.setPower(effReb, value);
                    }
                    value |= 1 << (parm.psNum + 1);
                    ps.setPower(effReb, value);
                    if (!checkPower(effReb, parm)) {
                        value &= ~(1 << (parm.psNum + 1));
                        ps.setPower(effReb, value);
                        throw new PowerException("REB " + effReb + " power supply "
                                                 + parm.psNum + " failed to stabilize");
                    }
                }
            }
            else {
                ps.setPower(reb, ps.getPower(reb) & ~(1 << (RebPS.PS_HV_BIAS + 1)) & ~(1 << (RebPS.PS_DPHI + 1)));
                for (int j = onSeq.length - 1; j >= 0; j--) {
                    SeqParams parm = onSeq[j];
                    int effReb = reb + parm.rebOff;
                    int value = ps.getPower(effReb) & ~(1 << (parm.psNum + 1));
                    ps.setPower(effReb, value);
                    if (value == 1) {
                        ps.setPower(effReb, 0);
                    }
                }
            }
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Sequences a whole power supply on or off.
     *
     *  @param  on   Turns on if true, off if false
     *  @throws  PowerException
     */
    public void sequencePower(boolean on) throws PowerException
    {
        for (int reb = 0; reb < ps.getNumRebs(); reb++) {
            sequencePower(reb, on);
        }
    }


    /**
     *  Reads extrema for a power supply channel
     *
     *  @param  reb     The REB number
     *  @param  psName  The power supply enum
     *  @param  chan    The channel enum 
     *  @return  A two-element array containing the maximum (0) and minimum (1) values
     *  @throws  PowerException
     */
    public double[] readChanExtrema(int reb, PsName psName, ChanName chan) throws PowerException
    {
        try {
            double[] value = ps.readChanExtended(reb, psName.getValue(), chan.getValue());
            return new double[]{value[RebPS.EXT_VALUE_MAX], value[RebPS.EXT_VALUE_MIN]};
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Resets extrema for a power supply channel
     *
     *  @param  reb     The REB number
     *  @param  psName  The power supply enum
     *  @param  chan    The channel enum
     *  @throws  PowerException
     */
    public void resetChanExtrema(int reb, PsName psName, ChanName chan) throws PowerException
    {
        try {
            ps.resetChanExtrema(reb, psName.getValue(), chan.getValue());
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


    /**
     *  Sets an event listener.
     *
     *  @param  listen  The listener object
     */
    public void setListener(Event listen)
    {
        listener = listen;
    }


    /**
     *  Clears the event listener.
     */
    public void clearListener()
    {
        listener = null;
    }


    /**
     *  Gets the error counters.
     *
     *  @return  A two-element array containing the number of network timeouts
     *           and the number of sequence errors
     */
    public int[] getErrors()
    {
        return new int[]{ps.getNumTimeout(), ps.getNumSeqErr()};
    }


    /**
     *  Gets the power supply driver.
     *
     *  @return  The driver
     */
    public RebPS getPs()
    {
        return ps;
    }


    /**
     *  Gets the power supply type.
     *
     *  @return  The type
     */
    public int getPsType()
    {
        return psType;
    }


    /**
     *  Gets the power supply version.
     *
     *  @return  The version
     */
    public int getPsVersion()
    {
        return psVersion;
    }


    /**
     *  Gets the power supply ID.
     *
     *  @return  The ID
     */
    public String getPsId()
    {
        return psId;
    }


    /**
     *  Checks that a PS voltage has reached the correct value.
     *
     *  @param  parms  The power supply parameters
     *  @return  Whether the correct value was reached
     */
    private boolean checkPower(int reb, SeqParams parms) throws PowerException
    {
        boolean okay = false;
        int nGood = 0;
        for (int count = 0; count < 20 && !okay; count++) {
            try {
                Thread.sleep(50);
            }
            catch (InterruptedException e) {
            }
            try {
                double voltage = ps.readChannel(reb, parms.psNum, RebPS.CHAN_VOLT_AFT_SW);
                nGood = (voltage >= parms.minGood && voltage <= parms.maxGood) ? nGood + 1 : 0;
                okay = nGood >= 3;
            }
            catch (DriverException e) {
                throw new PowerException(e.getMessage());
            }
        }

        return okay;
    }


    /**
     *  Generates the power supply ID and type.
     */
    private void generatePsId() throws DriverException
    {
        if (isSimulated){
            psId = "PS-SIM";
            psType = RebPS.TYPE_SCIENCE;
            return;
        }
        String serial = String.format("%016x", ps.getSerialNo());
        psId = props.getProperty("org.lsst.ccs.power.psid." + serial);
        if (psId == null) {
            if (ps.getNumRebs() == 3) {
                log.warn("Unknown PS serial number: " + serial);
                psId = serial.substring(5, 10);
            }
            else {
                psId = (psType == RebPS.TYPE_CORNER ? "CR-" : "SR-");
                try {
                    int id = Integer.valueOf(ipAddr.split("\\.")[3]);
                    psId += String.format("%03d", id);
                }
                catch (IndexOutOfBoundsException | NumberFormatException e) {
                    psId += "XXX";
                }
            }
        }
        psType = psType == RebPS.TYPE_UNKNOWN ? psId.substring(0, 2).equals("CR")
                   ? RebPS.TYPE_CORNER : RebPS.TYPE_SCIENCE : psType;
    }

}
