package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.auxelex.PduCommon;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.power.constants.PowerConfig;

/**
 *  Interface to a SLAC PDU.
 *
 *  This class is designed only to be extended, not instantiated itself
 *
 *  @author Owen Saxton
 */
public class PduCommonDevice extends Device implements SwitchControl {

    /**
     *  Constants.
     */
    protected static final int
        MON_TYPE_VOLTAGE = 0,
        MON_TYPE_CURRENT = 1;

    private static final String
        NODE   = "node";

    private static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
    }

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name=NODE, category=PowerConfig.CATEGORY_QUADBOX, isFinal=true)
    protected Integer node;

    private final PduCommon pdu;
    private final String pduType;
    private final int switchDevice;
    private int validChans;


    /**
     *  Constructor.
     *
     *  @param  pdu      The PDU driver object
     *  @param  pduType  The PDU type description
     *  @param  swDevc   The switch device ID
     */
    public PduCommonDevice(PduCommon pdu, String pduType, int swDevc)
    {
        this.pdu = pdu;
        this.pduType = pduType;
        switchDevice = swDevc;
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (node == null) {
            MonitorLogUtils.reportConfigError(log, name, NODE, "is missing");
        }
        fullName = pduType + " PDU (" + node + ")";
        validChans = pdu.getValidChannels();
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            pdu.open(node);
            setOnline(true);
            initSensors();
            log.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            pdu.close();
        }
        catch (DriverException e) {
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        }
        if ((mType == MON_TYPE_VOLTAGE || mType == MON_TYPE_CURRENT)
              && (hwChan < 0 || ((1 << hwChan) & validChans) == 0)) {
            MonitorLogUtils.reportError(log, name, "hwChan", hwChan);
        }
        return new int[]{mType, 0};
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        String item = null;
        if (online) {
            try {
                switch (type) {
                case MON_TYPE_VOLTAGE:
                    item = "voltage";
                    value = pdu.readVoltage(hwChan);
                    break;

                case MON_TYPE_CURRENT:
                    item = "current";
                    value = pdu.readCurrent(hwChan);
                    break;
                }
            }
            catch (DriverException e) {
                log.error("Error reading " + fullName + " " + item + ": " + e);
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Gets the switch device type
     * 
     *  @return  The switch device type
     */
    @Override
    public int getSwitchDevice()
    {
        return switchDevice;
    }


    /**
     *  Turns a PDU channel on.
     *
     *  @param  chan  The channel number
     */
    @Override
    public void switchOn(int chan)
    {
        if (chan < 0) {
            try {
                pdu.setMainOn();
            }
            catch (DriverException e) {
                log.error("Error setting main power on " + fullName + ": " + e);
            }
        }
        else {
            try {
                pdu.powerOn(chan);
            }
            catch (DriverException e) {
                log.error("Error setting channel " + chan + " on " + fullName + ": " + e);
            }
        }
    }


    /**
     *  Turns a PDU channel off.
     *
     *  @param  chan  The channel number
     */
    @Override
    public void switchOff(int chan)
    {
        if (chan < 0) {
            try {
                pdu.setMainOff();
            }
            catch (DriverException e) {
                log.error("Error setting main power on " + fullName + ": " + e);
            }
        }
        else {
            try {
                pdu.powerOff(chan);
            }
            catch (DriverException e) {
                log.error("Error setting channel " + chan + " on " + fullName + ": " + e);
            }
        }
    }


    /**
     *  Gets the state of a PDU channel.
     *
     *  @param  chan  The channel number
     *  @return  Whether the channel is on or off
     */
    @Override
    public Boolean isSwitchOn(int chan)
    {
        if (!online) return null;
        if (chan < 0) {
            try {
                return pdu.isMainOn();
            }
            catch (DriverException e) {
                log.error("Error getting state for main power on " + fullName + ": " + e);
                return null;
            }
        }
        else {
            if ((validChans & (1 << chan)) == 0) return null;
            try {
                return pdu.isPowered(chan);
            }
            catch (DriverException e) {
                log.error("Error getting state for channel " + chan + " on " + fullName + ": " + e);
                return null;
            }
        }
    }

}
