/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystem.power;

import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.power.PowerControl;
import org.lsst.ccs.subsystem.power.PowerDevice;
import org.lsst.ccs.subsystem.power.data.ATSPowerState;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.utilities.logging.Logger;

public class ATSPower
implements HasLifecycle {
    private static final String HV_BIAS_CHAN = "HVBias";
    private static final String DPHI_CHAN = "DPHI";
    private static final String[] pwrChannels = new String[8];
    private static final int POWER_TIMEOUT = 2000;
    private final Logger log = Logger.getLogger((String)"org.lsst.ccs.subsystem.power");
    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy=LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private final Map<String, PowerDevice> pwrDevices = new LinkedHashMap<String, PowerDevice>();
    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private final Map<String, PowerControl> pwrControls = new LinkedHashMap<String, PowerControl>();
    private PowerControl dphiControl;
    private PowerControl hvBiasControl;
    private PowerDevice hvBiasDevice;
    private final Boolean[] powerOn = new Boolean[8];

    public void build() {
    }

    public void postInit() {
        this.subsys.setAgentProperty("atsPowerAgent", this.getClass().getCanonicalName());
        boolean error = false;
        if (this.pwrDevices.isEmpty()) {
            this.log.error((Object)"No power devices have been defined");
            error = true;
        } else {
            for (String chan : pwrChannels) {
                if (this.pwrControls.get(chan) != null) continue;
                this.log.error((Object)("Required power channel " + chan + " has not been defined"));
                error = true;
            }
        }
        if (error) {
            throw new RuntimeException("Fatal initialization error");
        }
        this.dphiControl = this.pwrControls.get(DPHI_CHAN);
        this.hvBiasControl = this.pwrControls.get(HV_BIAS_CHAN);
    }

    public void postStart() {
        this.hvBiasDevice = this.hvBiasControl.getDevice();
        this.pwrDevices.remove(this.hvBiasDevice.getName());
        this.getPowerOnState();
        this.log.info((Object)"ATS power subsystem started");
    }

    public void periodicBroadcast() {
        if (this.pwrDevices.isEmpty()) {
            return;
        }
        ArrayList<PowerChanState> pState = new ArrayList<PowerChanState>();
        for (PowerDevice pDevc : this.pwrDevices.values()) {
            try {
                pState.addAll(pDevc.getPowerState());
            }
            catch (Exception exception) {}
        }
        KeyValueData kvd = new KeyValueData("PowerChanState", pState);
        this.subsys.publishSubsystemDataOnStatusBus(kvd);
    }

    @Command(type=Command.CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)") int period) {
        this.setTickPeriod(period);
        this.publishState();
    }

    @Command(type=Command.CommandType.QUERY, description="Get the power state")
    public List<PowerChanState> getPowerState() throws PowerException {
        ArrayList<PowerChanState> pState = new ArrayList<PowerChanState>();
        for (PowerDevice pDevc : this.pwrDevices.values()) {
            pState.addAll(pDevc.getPowerState());
        }
        return pState;
    }

    @Command(type=Command.CommandType.ACTION, description="Turn on the power")
    public void powerOn() throws PowerException {
        PowerException excp = null;
        if (!this.isPowerOn()) {
            for (PowerDevice pDevc : this.pwrDevices.values()) {
                try {
                    pDevc.powerOn();
                }
                catch (PowerException e) {
                    excp = e;
                }
            }
            long maxTime = System.currentTimeMillis() + 2000L;
            for (PowerDevice pDevc : this.pwrDevices.values()) {
                try {
                    pDevc.waitPowerOn((int)(maxTime - System.currentTimeMillis()));
                }
                catch (PowerException e) {
                    excp = e;
                }
            }
        }
        this.getPowerOnState();
        this.publishState();
        if (excp != null) {
            throw excp;
        }
    }

    @Command(type=Command.CommandType.ACTION, description="Turn off the power")
    public void powerOff() throws PowerException {
        PowerException excp = null;
        try {
            this.hvBiasDevice.powerOff();
        }
        catch (PowerException e) {
            excp = e;
        }
        for (PowerDevice pDevc : this.pwrDevices.values()) {
            try {
                pDevc.powerOff();
            }
            catch (PowerException e) {
                excp = e;
            }
        }
        long maxTime = System.currentTimeMillis() + 2000L;
        for (PowerDevice pDevc : this.pwrDevices.values()) {
            try {
                pDevc.waitPowerOff((int)(maxTime - System.currentTimeMillis()));
            }
            catch (PowerException e) {
                excp = e;
            }
        }
        this.getPowerOnState();
        this.publishState();
        if (excp != null) {
            throw excp;
        }
    }

    @Command(type=Command.CommandType.ACTION, description="Turn on the HV bias")
    public void hvBiasOn() throws PowerException {
        if (!this.isPowerOn()) {
            return;
        }
        try {
            Boolean hvOn = this.powerOn[7];
            if (hvOn != null && !hvOn.booleanValue()) {
                this.setHvBiasValue(0.0);
            }
            this.hvBiasDevice.powerOn();
        }
        finally {
            this.getPowerOnState();
            this.publishState();
        }
    }

    @Command(type=Command.CommandType.ACTION, description="Turn off the HV bias")
    public void hvBiasOff() throws PowerException {
        try {
            this.setHvBiasValue(0.0);
            this.hvBiasDevice.powerOff();
        }
        finally {
            this.getPowerOnState();
            this.publishState();
        }
    }

    @Command(type=Command.CommandType.ACTION, description="Set the DPHI voltage")
    public void setDphi(@Argument(description="Voltage value") double value) throws PowerException {
        try {
            this.setDphiValue(value);
        }
        finally {
            this.publishState();
        }
    }

    @Command(type=Command.CommandType.ACTION, description="Set the HV bias voltage")
    public void setHvBias(@Argument(description="Voltage value") double value) throws PowerException {
        try {
            this.setHvBiasValue(value);
        }
        finally {
            this.publishState();
        }
    }

    @Command(type=Command.CommandType.QUERY, description="Get the full state")
    public ATSPowerState getFullState() throws PowerException {
        return new ATSPowerState(this.getTickPeriod(), this.powerOn, this.dphiControl.getVoltage(), this.hvBiasControl.getVoltage());
    }

    private void setTickPeriod(long period) {
        this.pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }

    private int getTickPeriod() {
        return (int)this.pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

    private void publishState() {
        ATSPowerState ps = new ATSPowerState(this.getTickPeriod(), this.powerOn, this.dphiControl.getVoltage(), this.hvBiasControl.getVoltage());
        KeyValueData kvd = new KeyValueData("ATSPowerState", (Serializable)ps);
        this.subsys.publishSubsystemDataOnStatusBus(kvd);
    }

    private void getPowerOnState() {
        for (int chan = 0; chan < 8; ++chan) {
            try {
                this.powerOn[chan] = this.pwrControls.get(pwrChannels[chan]).readOutput();
                continue;
            }
            catch (Exception e) {
                this.powerOn[chan] = null;
            }
        }
    }

    private boolean isPowerOn() {
        boolean isOn = true;
        for (int chan = 0; chan < 8; ++chan) {
            if (chan == 7 || this.powerOn[chan] != null && this.powerOn[chan].booleanValue()) continue;
            isOn = false;
            break;
        }
        return isOn;
    }

    private void setHvBiasValue(double value) throws PowerException {
        this.hvBiasControl.setVoltage(-Math.abs(value));
        this.hvBiasControl.writeVoltage();
        this.hvBiasControl.writeOutput(true);
    }

    private void setDphiValue(double value) throws PowerException {
        this.dphiControl.setVoltage(value);
        this.dphiControl.writeVoltage();
        this.dphiControl.writeOutput(true);
    }

    static {
        ATSPower.pwrChannels[0] = "Digital";
        ATSPower.pwrChannels[1] = "Analog";
        ATSPower.pwrChannels[2] = "ClockHigh";
        ATSPower.pwrChannels[3] = "ClockLow";
        ATSPower.pwrChannels[4] = "OD";
        ATSPower.pwrChannels[5] = "Heater";
        ATSPower.pwrChannels[6] = DPHI_CHAN;
        ATSPower.pwrChannels[7] = HV_BIAS_CHAN;
    }
}

