package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.lambda.Cpfe1000;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a Lambda CPFE1000 power supply system.
 *
 *  @author Owen Saxton
 */
public class CpfeDevice extends PowerDevice {

   /**
    *  Constants.
    */
    private static final int
        TYPE_STATUS = 0,
        TYPE_TEMPERATURE = 1,
        TYPE_VOLTAGE = 2,
        TYPE_CURRENT = 3;

    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("STATUS", TYPE_STATUS);
        typeMap.put("TEMP", TYPE_TEMPERATURE);
        typeMap.put("VOLTAGE", TYPE_VOLTAGE);
        typeMap.put("CURRENT", TYPE_CURRENT);
    }

    /**
    *  Data fields.
    */
    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private final Cpfe1000 cpfe;  // Associated CPFE object


    /**
     *  Constructor
     */
    public CpfeDevice()
    {
        super("CPFE1000F", new Cpfe1000(), NETWORK_CONN_INVALID, 0, 0);
        cpfe = (Cpfe1000)psd;
        devcParm = 0xa0;
    }


   /**
    *  Performs basic initialization.
    */
    @Override
    protected void initDevice()
    {
        super.initDevice();
        fullName = "CPFE1000 (" + devcId + ":" + devcParm + ")";
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(sLog, name, "type", type);
        }
        return new int[]{iType, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        try {
            switch (type) {

            case TYPE_STATUS:
                value = cpfe.readStatus(); break;

            case TYPE_TEMPERATURE:
                value = cpfe.readTemperature(); break;

            case TYPE_VOLTAGE:
                value = cpfe.readVoltage(); break;

            case TYPE_CURRENT:
                value = cpfe.readCurrent(); break;
            }
        }
        catch (DriverException e) {
            sLog.error("Error reading CPFE1000 data: " + e);
            setOnline(false);
        }

        return value;
    }

}
