package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.rebps.RebPs;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.power.config.RebPsEnum;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.data.PowerSupplyAgentProperties;
import org.lsst.ccs.subsystem.power.data.RebPsFullState;
import org.lsst.ccs.subsystem.power.data.RebPsState;
import org.lsst.ccs.subsystem.power.states.RebPowerState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the REB PS control subsystem.
 *
 *  @author Owen Saxton
 */
public class RebPower implements HasLifecycle, RebPsDevice.Event {

    private final Logger sLog = Logger.getLogger("org.lsst.ccs.subsystem.power");
    @LookupField( strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField( strategy=LookupField.Strategy.TREE)
    private Monitor mon;    
    @LookupField( strategy=LookupField.Strategy.CHILDREN)
    private PowerDevice mainPs;
    @LookupField( strategy=LookupField.Strategy.CHILDREN)    
    private RebPsDevice psDevice;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    //The map of all the Channels
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String,Channel> channels = new HashMap<>();
    
    //The list of the Channels that measure the REB power
    private final List<Channel> rebPowerChannels = new ArrayList<>();

    //Power usage per REB. These values define the RebsPowerState state limits.
    @ConfigurationParameter
    private double[] powerStateTransitionLimits = new double[] {0.2, 8, 15};

    private long updateStateMillis = 1000;
    private long checkTrippedMillis = 1000;
    
    //Hysteresis percentage to manage state transitions.
    @ConfigurationParameter
    private double powerStateTransitionHysteresisPercentage = 0.01;
    
    private Map<String,Double> rebPower = new HashMap<>();
    private Map<String,Double> rebSimulatedPower = new HashMap<>();
    private boolean isSimulated;
    private boolean running;


    /**
     * Override build method of HasLifecycle.
     */
    @Override
    public void build() {

        AgentPeriodicTask periodicTask;

        //Create and schedule an AgentPeriodicTask to update the RebPowerState
        periodicTask = new AgentPeriodicTask("reb-power-state",
                                             () -> updateRebPowerState()).withPeriod(Duration.ofMillis(updateStateMillis));
        periodicTaskService.scheduleAgentPeriodicTask(periodicTask);

        //Create and schedule an AgentPeriodicTask to check whether the power supply has tripped
        periodicTask = new AgentPeriodicTask("reb-power-check",
                                             () -> checkPsTripped()).withPeriod(Duration.ofMillis(checkTrippedMillis));
        periodicTaskService.scheduleAgentPeriodicTask(periodicTask);
    }        

    /**
     * Override init method of HasLifecycle.
     */
    @Override
    public void init() {
        isSimulated = BootstrapResourceUtils.getBootstrapSystemProperties().
                        getProperty("org.lsst.ccs.run.mode","").equals("simulation");
    }
    
    /**
     *  Initializes the power subsystem.
     */
    @Override
    public void postInit()
    {
        //Set a property to define that this Agent is a Power Supply controller.
        subsys.setAgentProperty(PowerSupplyAgentProperties.POWER_SUPPLY_TYPE_AGENT_PROPERTY,
                                RebPower.class.getCanonicalName());
        
        if (mainPs == null) {
            sLog.warning("No main power supply device specified");
        }

        if (psDevice == null) {
            sLog.error("No REB power devices specified");
        }
        else {
            psDevice.setListener(this);
        }
        
        for ( Entry<String,Channel> entry : channels.entrySet() ) {
            String channelName = entry.getKey();
            if ( channelName.endsWith(".Power") ) {
                sLog.fine("Adding power channel "+channelName);
                rebPowerChannels.add(entry.getValue());
            }
        }
        StateBundle sb = new StateBundle(RebPowerState.OFF);
        for (Channel ch : rebPowerChannels) {
            String rebName = ch.getName();
            sb.setComponentState(rebName, RebPowerState.OFF);
            if ( isSimulated ) {
                rebSimulatedPower.put(rebName,0.0);
            }
            rebPower.put(rebName, 0.0);
        }
        subsys.updateAgentState(sb);                
    }

 
    /**
     *  Starts the subsystem.
     *
     */
    @Override
    public void postStart() 
    {
        sLog.info("REB PS subsystem started");
        running = true;
        publishState();        // For any GUIs
    }


    /**
     *  Sets the simulated REB power
     * 
     *  @param rebName
     *  @param rebPower 
     */
    @Command(type=CommandType.ACTION, description="Set simulated reb power")
    public void setSimulatedRebPower(@Argument(name="rebName") String rebName,
                                     @Argument(name="rebPower", description="The simulated Reb Total power") double rebPower) {
        rebSimulatedPower.put(rebName, rebPower);
    }


    /**
     *  Updates the REB power state periodically.
     */
    private void updateRebPowerState() {
        StateBundle sb = new StateBundle();
        double totalPower = 0;
        double oldTotalPower = 0;
        for (Channel ch : rebPowerChannels) {
            String rebName = ch.getName();

            double rebPowerValue = isSimulated ? rebSimulatedPower.get(rebName) : ch.getValue();
            totalPower += rebPowerValue;
            oldTotalPower += rebPower.get(rebName);

            if (isChangeSignificant(rebPowerValue, rebPower.get(rebName))) {
                rebPower.put(rebName, rebPowerValue);
                sb.setComponentState(rebName, getRebPowerState(rebPowerValue));
            }
        }
        
        if (isChangeSignificant(totalPower, oldTotalPower)) {
            sb.setState(getRebPowerState(totalPower/3.));
        }
        
        if (!subsys.isInState(sb)) {
            sLog.fine("Changing Rebs Power State to " + sb);
            subsys.updateAgentState(sb);
        }
    }


    /**
     *  Checks whether a power supply has tripped.
     *
     *  Is called in a periodic task
     */
    private void checkPsTripped()
    {
        if (psDevice != null && psDevice.checkPsTripped()) {
            publishState();
        }
    }


    /**
     *  Generates the REB power state.
     * 
     *  @param value The power value
     *  @return The generated power state
     */
    private RebPowerState getRebPowerState(double value) {
        if (value < powerStateTransitionLimits[0]) {
            return RebPowerState.OFF;
        } else if (value < powerStateTransitionLimits[1]) {
            return RebPowerState.ON;
        } else if (value < powerStateTransitionLimits[2]) {
            return RebPowerState.LOW_POWER;
        } else {
            return RebPowerState.OPERATIONAL;
        }
    }


    /**
     *  Checks whether power change is significant
     * 
     *  @param newValue
     *  @param oldValue
     *  @return Whether significant
     */
    private boolean isChangeSignificant(double newValue, double oldValue) {
        return Math.abs(newValue - oldValue) > powerStateTransitionHysteresisPercentage * oldValue;
    }
    

    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the tick period
     */
    private long getTickPeriod()
    {
        return periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }
    

    /**
     *  Sets the update (tick) period.
     *
     *  @param  period  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)")
                                int period)
    {
        setTickPeriod(period);
        publishState();    // Must do this for GUI
    }


    /**
     *  Turns the main power supply on or off.
     *
     *  @param  on  Turns the power on if true, off if false
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn the main power supply on or off")
    public void setMainPower(@Argument(name="on", description="Whether on or off")
                             boolean on) throws Exception
    {
        try {
            if (mainPs != null) {
                if (on) {
                    mainPs.powerOn();
                    if (psDevice != null) {
                        psDevice.enable();
                    }
                }
                else {
                    if (psDevice != null) {
                        try {
                            psDevice.sequencePower(false);
                        }
                        finally {
                            psDevice.disable();
                            mainPs.powerOff();
                        }
                    }
                }
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


    /**
     *  Toggles the main power supply state.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Toggle the main power supply state")
    public void toggleMainPower() throws Exception
    {
        setMainPower(getMainState() == PowerChanState.PWR_STATE_OFF);
    }


    /**
     *  Turns a power supply on or off.
     *
     *  @param  reb  The REB number
     *  @param  ps   The power supply number, or -1 for master switch
     *  @param  on   Turns the power on if true, off if false
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turns a power supply on or off")
    public void setPowerOn(@Argument(name="reb", description="REB number")
                           int reb,
                           @Argument(name="ps", description="Power supply number")
                           int ps,
                           @Argument(name="on", description="Whether on or off")
                           boolean on) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.setPowerOn(reb, ps, on);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


    /**
     *  Turns a power supply on or off.
     *
     *  @param  reb     The REB number
     *  @param  psEnum  The power supply enumeration
     *  @param  on      Turns the power on if true, off if false
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turns a power supply on or off")
    public void setNamedPowerOn(@Argument(name="reb", description="REB number")
                                int reb,
                                @Argument(name="name", description="Power supply name")
                                RebPsEnum psEnum,
                                @Argument(name="on", description="Whether on or off")
                                boolean on) throws PowerException
    {
        setPowerOn(reb, psEnum.getNumber(), on);
    }


    /**
     *  Toggles the on/off state of a power supply.
     *
     *  @param  reb  The REB number
     *  @param  ps   The power supply number
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Toggle a power supply")
    public void togglePower(@Argument(name="reb", description="REB number")
                            int reb,
                            @Argument(name="ps", description="Power supply number")
                            int ps) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.togglePower(reb, ps);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


    /**
     *  Toggles the on/off state of a power supply.
     *
     *  @param  reb     The REB number
     *  @param  psEnum  The power supply enumeration
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Toggle a power supply")
    public void toggleNamedPower(@Argument(name="reb", description="REB number")
                                 int reb,
                                 @Argument(name="name", description="Power supply name")
                                 RebPsEnum psEnum) throws PowerException
    {
        togglePower(reb, psEnum.getNumber());
    }


    /**
     *  Sequences a power supply on or off.
     *
     *  @param  reb  The REB number
     *  @param  on   Turns on if true, off if false
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Sequence a power supply")
    public void sequencePower(@Argument(name="reb", description="REB number")
                              int reb,
                              @Argument(name="on", description="Whether to turn on")
                              boolean on) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.sequencePower(reb, on);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


    /**
     *  Sets a HV bias DAC value.
     *
     *  @param  reb    The REB number
     *  @param  value  The value to set
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Set a HV bias DAC")
    public void setBiasDac(@Argument(name="reb", description="REB number")
                           int reb,
                           @Argument(name="value", description="DAC value")
                           double value) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.setBiasDac(reb, value);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


    /**
     *  Sets a DPHI DAC value.
     *
     *  @param  reb    The REB number
     *  @param  value  The value to set
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Set a DPHI DAC")
    public void setDphiDac(@Argument(name="reb", description="REB number")
                           int reb,
                           @Argument(name="value", description="DAC value")
                           double value) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.setDphiDac(reb, value);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


    /**
     *  Gets the full REB power supply state.
     *
     *  @return  The full power supply state
     *  @throws  Exception
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public RebPsFullState getFullState() throws Exception
    {
        return new RebPsFullState(getState(), mon.getFullState());
    }


    /**
     *  Gets the power supply error counters.
     *
     *  @return  The two-element array of error counters
     */
    @Command(type=CommandType.QUERY, description="Get the error counters")
    public int[] getErrors()
    {
        if (psDevice != null) {
            return psDevice.getErrors();
        }
        else {
            return new int[]{0, 0};
        }
    }


    /**
     *  Terminates the program.
     */
    @Command(type=CommandType.ACTION, description="Terminate the program")
    public void quit()
    {
        System.exit(0);
    }


    /**
     *  Publishes the state of the REB power supply.
     *
     *  This is intended to be called whenever any element of the state is changed.
     */
    void publishState()
    {
        if (!running) return;
        KeyValueData kvd = new KeyValueData(RebPsState.KEY, getState());
        subsys.publishSubsystemDataOnStatusBus(kvd);
    }    


    /**
     *  Gets the state of the REB power supply.
     */
    RebPsState getState()
    {
        int[] psState;
        double[] biasDacs, dphiDacs;
        int psType = RebPs.TYPE_SCIENCE, psVersion = RebPs.VERSION_PROD;
        String psId = "PS-X";
        psState = new int[0];
        biasDacs = new double[0];
        dphiDacs = new double[0];
        if (psDevice != null) {
            psType = psDevice.getPsType();
            psVersion = psDevice.getPsVersion();
            psId = psDevice.getPsId();
            psState = psDevice.getState();
            biasDacs = psDevice.getDacs(true);
            dphiDacs = psDevice.getDacs(false);
        }
        return new RebPsState((int)getTickPeriod(), getMainState(), psState, biasDacs, dphiDacs, psType, psVersion, psId);
    }


    /**
     *  Gets the state of the main power supply.
     */
    int getMainState()
    {
        int state = PowerChanState.PWR_STATE_OFFLINE;
        if (mainPs != null && mainPs.isOnline()) {
            try {
                List<PowerChanState> pcState = mainPs.getPowerState();
                if (!pcState.isEmpty()) {
                    state = pcState.get(0).getState();
                }
            }
            catch (Exception e) {
            }
        }

        return state;
    }


    /**
     *  Called when a power supply device is opened.
     *
     *  Publishes the current state on the status bus.
     */
    @Override
    public void opened()
    {
        publishState();
    }


    /**
     *  Called when a power supply device is closed.
     *
     *  Publishes the current state on the status bus.
     */
    @Override
    public void closed()
    {
        publishState();
    }

}
