package org.lsst.ccs.subsystem.power;

import org.lsst.ccs.drivers.bk.Model9130;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Interface to a B&K model 9130 power supply device.
 *
 *  @author Owen Saxton
 */
public class BK9130Device extends PowerDevice {

    /**
     *  Data fields.
     */
    private final Model9130 ps;
    private final double[] outVoltages = new double[Model9130.NUM_CHANNELS];
    private final double[] outCurrents = new double[Model9130.NUM_CHANNELS];
    private final boolean[] outOutputs = new boolean[Model9130.NUM_CHANNELS];
    private double[] inVoltages, inCurrents;
    private boolean[] inOutputs;


    /**
     *  Constructor.
     */
    public BK9130Device()
    {
        super("B&K 9130 PS", new Model9130(), NETWORK_CONN_INVALID, 0, Model9130.MAX_CHANNEL);
        ps = (Model9130)psd;
    }


    /**
     *  Writes the voltage to the hardware.
     *
     *  @param  voltage  The voltage to write
     *  @param  hwChan   The hardware channel number
     *  @throws  DriverException
     */
    @Override
    protected void writeVoltage(double voltage, int hwChan) throws DriverException
    {
        //ps.setVoltage(voltage, hwChan);
        outVoltages[hwChan - Model9130.MIN_CHANNEL] = voltage;
    }


    /**
     *  Reads the voltage from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The voltage
     *  @throws  DriverException
     */
    @Override
    protected double readVoltage(int hwChan) throws DriverException
    {
        //return ps.readVoltage(hwChan);
        return inVoltages[hwChan - Model9130.MIN_CHANNEL];
    }


    /**
     *  Writes the current to the hardware.
     *
     *  @param  current  The current to write
     *  @param  hwChan   The hardware channel number
     *  @throws  DriverException
     */
    @Override
    protected void writeCurrent(double current, int hwChan) throws DriverException
    {
        //ps.setCurrent(current, hwChan);
        outCurrents[hwChan - Model9130.MIN_CHANNEL] = current;
    }


    /**
     *  Reads the current from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The current
     *  @throws  DriverException
     */
    @Override
    protected double readCurrent(int hwChan) throws DriverException
    {
        //return ps.readCurrent(hwChan);
        return inCurrents[hwChan - Model9130.MIN_CHANNEL];
    }


    /**
     *  Writes the power output state to the hardware.
     *
     *  @param  onState  The state to set: on if true; off if false
     *  @param  hwChan   The hardware channel number
     *  @throws  DriverException
     */
    @Override
    protected void writeOutput(boolean onState, int hwChan) throws DriverException
    {
        //ps.setOutput(onState, hwChan);
        outOutputs[hwChan - Model9130.MIN_CHANNEL] = onState;
    }


    /**
     *  Reads the power output state from the hardware.
     *
     *  @param  hwChan   The hardware channel number
     *  @return  The power output state
     *  @throws  DriverException
     */
    @Override
    protected boolean readOutput(int hwChan) throws DriverException
    {
        //return ps.getOutput(hwChan);
        return inOutputs[hwChan - Model9130.MIN_CHANNEL];
    }


    /**
     *  Reads all the voltages from the hardware.
     *
     *  @throws  DriverException
     */
    @Override
    protected void readVoltageGroup() throws DriverException
    {
        inVoltages = ps.readVoltage();
    }


    /**
     *  Writes all the voltages to the hardware.
     *
     *  @throws  DriverException
     */
    @Override
    protected void writeVoltageGroup() throws DriverException
    {
        ps.setVoltage(outVoltages);
    }


    /**
     *  Reads all the currents from the hardware.
     *
     *  @throws  DriverException
     */
    @Override
    protected void readCurrentGroup() throws DriverException
    {
        inCurrents = ps.readCurrent();
    }


    /**
     *  Writes all the currents to the hardware.
     *
     *  @throws  DriverException
     */
    @Override
    protected void writeCurrentGroup() throws DriverException
    {
        ps.setCurrent(outCurrents);
    }


    /**
     *  Reads all the output states from the hardware.
     *
     *  @throws  DriverException
     */
    @Override
    protected void readOutputGroup() throws DriverException
    {
        inOutputs = ps.getOutput();
    }


    /**
     *  Writes all the output states to the hardware.
     *
     *  @throws  DriverException
     */
    @Override
    protected void writeOutputGroup() throws DriverException
    {
        ps.setOutput(outOutputs);
    }

}
