package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.drivers.rebps.RebPs;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 ******************************************
 *
 *  Interface to a REB power supply board.
 *
 *  @author Owen Saxton
 *
 ******************************************
 */
public class RebPsDevice extends Device {

   /**
    *  Inner class for holding power supply parameters.
    */
    static class SeqParams {

        int    psNum;
        double minGood;
        double maxGood;

        SeqParams(int psNum, double minGood, double maxGood)
        {
            this.psNum = psNum;
            this.minGood = minGood;
            this.maxGood = maxGood;
        }

    }

   /**
    *  Interface for open/close event listener.
    */
    public interface Event {

        public void opened();

        public void closed();

    }

   /**
    *  Constants.
    */
    private static final String
        DEVC_ID   = "devcId",
        DEVC_PARM = "devcParm",
        UNKN_PSID = "PS-XX";

    private static final int
        MAX_DAC = 4095,
        PS_POWER = 255;

    private static final Map<String, Integer> psMap = new HashMap<>();
    static {
        psMap.put("DIGITAL", RebPs.PS_DIGITAL);
        psMap.put("ANALOG", RebPs.PS_ANALOG);
        psMap.put("CLOCKHI", RebPs.PS_CLK_HIGH);
        psMap.put("CLOCKLO", RebPs.PS_CLK_LOW);
        psMap.put("DPHI", RebPs.PS_DPHI);
        psMap.put("HEATER", RebPs.PS_HEATER);
        psMap.put("HVBIAS", RebPs.PS_HV_BIAS);
        psMap.put("OD", RebPs.PS_OD);
        psMap.put("POWER", PS_POWER);
    }

    private static final SeqParams[]
        onSeqSR = {new SeqParams(RebPs.PS_DIGITAL, 4.5, 6.0),
                   new SeqParams(RebPs.PS_ANALOG, 6.5, 8.0),
                   new SeqParams(RebPs.PS_OD, 39.0, 42.0),
                   new SeqParams(RebPs.PS_CLK_LOW, 15.0, 17.0),
                   new SeqParams(RebPs.PS_CLK_HIGH, 15.0, 17.0),
                   new SeqParams(RebPs.PS_HEATER, 7.0, 9.0)};

    private static final SeqParams[]
        onSeqCR = {new SeqParams(RebPs.PS_DIGITAL, 4.5, 6.0),
                   new SeqParams(RebPs.PS_ANALOG, 6.5, 8.0),
                   new SeqParams(RebPs.PS_OD, 35.0, 37.0),
                   new SeqParams(RebPs.PS_CLK_LOW, 10.0, 12.0),
                   new SeqParams(RebPs.PS_CLK_HIGH, 10.0, 12.0),
                   new SeqParams(RebPs.PS_DPHI, 4.0, 13.0)};

    private static final SeqParams[]
        onSeqCRL = {new SeqParams(RebPs.PS_DIGITAL, 5.0, 7.0),
                    new SeqParams(RebPs.PS_ANALOG, 5.0, 7.0)};

    private static final int[]
        offSeqSR = {RebPs.PS_HV_BIAS, RebPs.PS_HEATER, RebPs.PS_CLK_HIGH,
                    RebPs.PS_CLK_LOW, RebPs.PS_OD, RebPs.PS_ANALOG,
                    RebPs.PS_DIGITAL};

    private static final int[]
        offSeqCR = {RebPs.PS_HV_BIAS, RebPs.PS_DPHI, RebPs.PS_CLK_HIGH,
                    RebPs.PS_CLK_LOW, RebPs.PS_OD, RebPs.PS_ANALOG,
                    RebPs.PS_DIGITAL};

    private static final int[]
        offSeqCRL = {RebPs.PS_ANALOG, RebPs.PS_DIGITAL};

   /**
    *  Data fields.
    */
    private String devcId;
    private int devcParm;

    private final RebPs ps = new RebPs();
    private Event listener;
    private String psId = UNKN_PSID;
    private Properties props;
    private boolean gotCR;


   /**
    *  Performs configuration.
    *
    *  @param  mon  The monitor object
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        String fileName = "psid.properties";
        if (BootstrapResourceUtils.getBootstrapResource(fileName) == null) {
            log.warn("PS ID properties file (" + fileName + ") not found");
        }
        props = BootstrapResourceUtils.getBootstrapProperties(fileName);
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize()
    {
        if (!inited) {
            fullName = "REB PS board (" + devcId + ")";
        }
        try {
            if (devcParm == 0) {
                ps.open(devcId);
            }
            else {
                ps.open(devcId, devcParm);
            }
            ps.setTemperatureRes(3);
            generatePsId();
            setOnline(true);
            if (listener != null) {
                listener.opened();
            }
            log.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close()
    {
        try {
            ps.close();
        }
        catch (DriverException e) {
        }
        finally {
            if (listener != null) {
                listener.closed();
            }
            psId = UNKN_PSID;
        }
    }


   /**
    *  Checks a monitoring channel's parameters for validity.
    *
    *  @param  name     The channel name
    *
    *  @param  hwChan   The hardware channel
    *
    *  @param  type     The channel type string
    *
    *  @param  subtype  The channel subtype string
    *
    *  @return  Two-element array containg the encoded type [0] and
    *           subtype [1] 
    *
    *  @throws  Exception  If parameters are invalid
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        int rebNum = 0;
        Integer psNum = null;
        String[] flds = type.split(":");
        if (flds.length == 1) {
            if (flds[0].toUpperCase().equals("TEMP")) {
                psNum = -1;
            }
        }
        else if (flds.length == 2) {
            try {
                rebNum = Integer.valueOf(flds[0]);
                if (RebPs.testRebNumber(rebNum)) {
                    psNum = psMap.get(flds[1].toUpperCase());
                }
            }
            catch (NumberFormatException e) {
            }
        }
        if (psNum == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        }
        if (psNum >= 0 && psNum != PS_POWER && !RebPs.testChannelNumber(psNum, hwChan)) {
            MonitorLogUtils.reportError(log, name, "hw channel number", hwChan);
        }
        
        return new int[]{(rebNum << 8) | psNum, 0};
    }


   /**
    *  Reads a monitoring channel.
    *
    *  @param  hwChan  The hardware channel number
    *
    *  @param  type    The encoded channel type
    *
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        if (online) {
            try {
                if (type < 0) {
                    value = ps.readTemperature();
                }
                else {   // Clock low V after switch is meaningless if powered off
                    int rebNum = type >> 8, psNum = type & 0xff;
                    if (psNum == PS_POWER) {
                        value = 0.0;
                        for (psNum = 0; psNum < RebPs.NUM_PS; psNum++) {
                            if (psNum == RebPs.PS_HV_BIAS) continue;
                            if ((ps.getPower(rebNum) & (1 << (psNum + 1))) == 0) continue;
                            value += ps.readChannel(rebNum, psNum, RebPs.CHAN_VOLT_AFT_SW)
                                       * ps.readChannel(rebNum, psNum, RebPs.CHAN_CURR_BEF_LDO);
                        }
                    }
                    else if (psNum != RebPs.PS_CLK_LOW || hwChan != RebPs.CHAN_VOLT_AFT_SW
                               || (ps.getPower(rebNum) & (1 << (psNum + 1))) != 0) {
                        value = ps.readChannel(rebNum, psNum, hwChan);
                    }
                }
            }
            catch (DriverTimeoutException e) {
                log.error("Error reading from " + fullName + ": " + e);
                if (type < 0) {
                    setOnline(false);  // No power to the supply
                }
            }
            catch (DriverException e) {
                // Channel is not powered
            }
        }

        return value;
    }


   /**
    *  Gets the power state data.
    *
    *  @return  A list of power state data
    *
    *  @throws  PowerException
    */
    protected int[] getState() throws PowerException
    {
        try {
            int[] state = new int[RebPs.NUM_REBS];
            for (int reb = 0; reb < state.length; reb++) {
                state[reb] = ps.getPower(reb);
            }
            return state;
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    *  Gets the HV bias DAC values.
    *
    *  @return  An array of bias DAC values
    *
    *  @throws  PowerException
    */
    protected double[] getBiasDacs() throws PowerException
    {
        try {
            double[] dacs = new double[RebPs.NUM_REBS];
            for (int reb = 0; reb < dacs.length; reb++) {
                dacs[reb] = ps.readChannel(reb, RebPs.PS_HV_BIAS, RebPs.CHAN_VOLT_DAC);
            }
            return dacs;
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    *  Gets the DPHI DAC values.
    *
    *  @return  An array of DPHI DAC values
    *
    *  @throws  PowerException
    */
    protected double[] getDphiDacs() throws PowerException
    {
        try {
            double[] dacs = new double[RebPs.NUM_REBS - 1];
            for (int reb = 0; reb < dacs.length; reb++) {
                dacs[reb] = ps.readChannel(reb, RebPs.PS_DPHI, RebPs.CHAN_VOLT_DAC);
            }
            return dacs;
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    *  Sets a HV bias DAC value.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  value   The value to set
    *
    *  @throws  PowerException
    */
    protected void setBiasDac(int rebNum, double value) throws PowerException
    {
        try {
            ps.writeDac(rebNum, RebPs.PS_HV_BIAS, value);
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    *  Sets a DPHI DAC value.
    *
    *  @param  rebNum  The REB number
    *
    *  @param  value   The value to set
    *
    *  @throws  PowerException
    */
    protected void setDphiDac(int rebNum, double value) throws PowerException
    {
        try {
            ps.writeDac(rebNum, RebPs.PS_DPHI, value);
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    *  Turns a power supply on or off.
    *
    *  @param  reb    The REB number
    *
    *  @param  psNum  The power supply number, or -1 for master switch
    *
    *  @param  on     Turn on if true, off if false
    *
    *  @throws  PowerException
    */
    protected void setPowerOn(int reb, int psNum, boolean on) throws PowerException
    {
        try {
            boolean resetBiasDac = false, resetDphiDac = false;
            int mask = ps.getPower(reb);
            if (psNum < 0) {   // Master - affects all
                int[] offSeq = gotCR ? reb < RebPs.NUM_REBS ? offSeqCR : offSeqCRL : offSeqSR;
                for (int psn : offSeq) {
                    mask &= ~(1 << (psn + 1));
                    ps.setPower(reb, mask);
                }
                ps.setPower(reb, on ? 1 : 0);
                resetBiasDac = true;
                resetDphiDac = on;
            }
            else if ((mask & 1) != 0) {    // Master is on - can proceed
                if (on) {
                    mask |= (1 << (psNum + 1));
                }
                else {
                    mask &= ~(1 << (psNum + 1));
                    resetBiasDac = (psNum == RebPs.PS_HV_BIAS);
                    resetDphiDac = (psNum == RebPs.PS_DPHI);
                }
                ps.setPower(reb, mask);
            }
            if (resetBiasDac) {
                ps.writeDac(reb, RebPs.PS_HV_BIAS, 0);
            }
            if (resetDphiDac & gotCR && reb < RebPs.NUM_REBS - 1) {
                ps.writeDac(reb, RebPs.PS_DPHI, MAX_DAC);
            }
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    *  Toggles the on/off state of a power supply.
    *
    *  @param  reb    The REB number
    *
    *  @param  psNum  The power supply number, or -1 for master switch
    *
    *  @throws  PowerException
    */
    protected void togglePower(int reb, int psNum) throws PowerException
    {
        try {
            int psn = (psNum < 0) ? 0 : psNum + 1;
            setPowerOn(reb, psNum, ((ps.getPower(reb) >> psn) & 1) == 0);
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    *  Sequences a power supply section on or off.
    *
    *  @param  reb  The REB number
    *
    *  @param  on   Turns on if true, off if false
    *
    *  @throws  PowerException
    */
    protected void sequencePower(int reb, boolean on) throws PowerException
    {
        try {
            int value = ps.getPower(reb);
            if (on) {
                SeqParams[] onSeq = gotCR ? reb < RebPs.NUM_REBS - 1 ? onSeqCR : onSeqCRL : onSeqSR;
                value |= 1;
                ps.setPower(reb, value);
                if (gotCR && reb < RebPs.NUM_REBS) {
                    ps.writeDac(reb, RebPs.PS_DPHI, MAX_DAC);
                }
                for (SeqParams parm : onSeq) {
                    value |= 1 << (parm.psNum + 1);
                    ps.setPower(reb, value);
                    if (!checkPower(reb, parm)) {
                        value &= ~(1 << (parm.psNum + 1));
                        ps.setPower(reb, value);
                        throw new PowerException("REB " + reb + " power supply "
                                                 + parm.psNum + " failed to stabilize");
                    }
                }
            }
            else {
                int[] offSeq = gotCR ? reb < RebPs.NUM_REBS - 1 ? offSeqCR : offSeqCRL : offSeqSR;
                for (int psNum : offSeq) {
                    value &= ~(1 << (psNum + 1));
                    ps.setPower(reb, value);
                }
                value = 0;
                ps.setPower(reb, value);
                ps.writeDac(reb, RebPs.PS_HV_BIAS, 0);
            }
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    *  Sequences a whole power supply on or off.
    *
    *  @param  on   Turns on if true, off if false
    *
    *  @throws  PowerException
    */
    protected void sequencePower(boolean on) throws PowerException
    {
        for (int reb = 0; reb < RebPs.NUM_REBS; reb++) {
            sequencePower(reb, on);
        }
    }


   /**
    *  Sets an event listener.
    *
    *  @param  listen  The listener object
    */
    protected void setListener(Event listen)
    {
        listener = listen;
    }


   /**
    *  Clears the event listener.
    */
    protected void clearListener()
    {
        listener = null;
    }


   /**
    *  Gets the error counters.
    *
    *  @return  A two-element array containing the number of network timeouts
    *           and the number of sequence errors
    */
    protected int[] getErrors()
    {
        return new int[]{ps.getNumTimeout(), ps.getNumSeqErr()};
    }


   /**
    *  Gets the power supply ID.
    *
    *  @return  The ID
    */
    protected String getPsId()
    {
        return psId;
    }


   /**
    *  Checks that a PS voltage has reached the correct value.
    *
    *  @param  parms  The power supply parameters
    *
    *  @return  Whether the correct value was reached
    */
    private boolean checkPower(int reb, SeqParams parms) throws PowerException
    {
        boolean okay = false;
        int nGood = 0;
        for (int count = 0; count < 20 && !okay; count++) {
            try {
                Thread.sleep(50);
            }
            catch (InterruptedException e) {
            }
            try {
                double voltage = ps.readChannel(reb, parms.psNum, RebPs.CHAN_VOLT_AFT_SW);
                nGood = (voltage >= parms.minGood && voltage <= parms.maxGood) ? nGood + 1 : 0;
                okay = nGood >= 3;
            }
            catch (DriverException e) {
                throw new PowerException(e.getMessage());
            }
        }

        return okay;
    }


   /**
    *  Generates the power supply ID.
    */
    private void generatePsId() throws DriverException
    {
        String serial = String.format("%016x", ps.getSerialNo());
        psId = props.getProperty("org.lsst.ccs.power.psid." + serial);
        if (psId == null) {
            log.warn("Unknown PS serial number: " + serial);
            psId = serial.substring(5, 10);
        }
        gotCR = psId.substring(0, 2).equals("CR");
    }

}
