package org.lsst.ccs.subsystem.power;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerFullState;
import org.lsst.ccs.subsystem.power.data.PowerState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ********************************************
 *
 *  Implements the power control subsystem.
 *
 *  @author Owen Saxton
 *
 ********************************************
 */
public class MainPower implements HasLifecycle {

    
    private final Logger sLog = Logger.getLogger("org.lsst.ccs.subsystem.power");

    
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private Map<String, PowerDevice> pwrDevices;
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
        
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    public static final String BROADCAST_TASK = "publish-data";
    
    private long broadcastMillis = 10000;

    @Override
    public void init() {
        //Define the Runnable object to be invoked periodically
        Runnable myRunnable = new Runnable() {
            @Override
            public void run() {
                //The code to be executed
                periodicBroadcast();
            }
        };

        //Create an AgentPeriodicTask that will execute the above Runnable instance
        //every second at a fixed rate
        AgentPeriodicTask periodicTask = new AgentPeriodicTask(BROADCAST_TASK, myRunnable).withPeriod(Duration.ofMillis(broadcastMillis));

        //Schedule the periodic task with the AgentPeriodicTaskService
        periodicTaskService.scheduleAgentPeriodicTask(periodicTask);
        
    }

    /**
    *  Initializes the power subsystem.
    */
    @Override
    public void postInit()
    {
        /*
        *  Get all the power devices
        */
        if (pwrDevices.isEmpty()) {
            sLog.error("No power devices specified");
        }
    }


   /**
    *  Starts the subsystem.
    *
    *  @throws  HardwareException
    */
    @Override
    public void postStart() throws HardwareException
    {
        sLog.info("Power subsystem started");
        publishState();        // For any GUIs
    }


   /**
    *  Performs periodic power data broadcast.
    */
    public void periodicBroadcast()
    {
        if (pwrDevices.isEmpty()) return;
        ArrayList<PowerChanState> pState = new ArrayList<>();
        for (PowerDevice pDevc : pwrDevices.values()) {
            try {
                pState.addAll(pDevc.getPowerState());
            }
            catch (Exception e) {
            }
        }
        KeyValueData kvd = new KeyValueData(PowerChanState.KEY, pState);
        subsys.publishSubsystemDataOnStatusBus(kvd);
    }

   /**
    *  Sets the tick period
    */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod(BROADCAST_TASK, Duration.ofMillis(period));
    }
    
    private long getTickPeriod()
    {
        return periodicTaskService.getPeriodicTaskPeriod(BROADCAST_TASK).toMillis();
    }
    
   /**
    *  Sets the update (tick) period.
    *
    *  @param  period  The update period (milliseconds) to set.
    */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(int period)
    {
        
        setTickPeriod(period);
        publishState();    // Must do this for GUI
    }


   /**
    *  Saves the configuration data.
    *
    *  @param  name  The configuration name
    *
    *  @throws IOException
    */
    @Command(type=CommandType.ACTION, description="Save the current configuration")
    public void saveNamedConfig(String name) throws IOException
    {
        subsys.getConfigurationService().saveChangesForCategoriesAs(Power.POWER + ":" + name);
    }


   /**
    *  Gets the configuration data.
    *
    *  @return  The array of power configuration objects
    */
    @Command(type=CommandType.QUERY, description="Get the configuration data")
    public Power[] getPowerConfig()
    {
        Power[] power = new Power[pwrDevices.size()];
        int j = 0;
        for (PowerDevice pDevc : pwrDevices.values()) {
            power[j++] = pDevc.getPowerConfig();
        }

        return power;
    }


   /**
    *  Sets the configuration data.
    *
    *  @param  power  The array of power configuration objects
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.ACTION, description="Set the configuration data")
    public void setPowerConfig(Power[] power) throws Exception
    {
        for (Power pConf : power) {
            PowerDevice pDevc = pwrDevices.get(pConf.getName());
            pDevc.setPowerConfig(pConf);
        }
    }


   /**
    *  Gets the power state.
    *
    *  @return  The power state
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.QUERY, description="Get the power state")
    public List<PowerChanState> getPowerState() throws Exception
    {
        List<PowerChanState> pState = new ArrayList<>();
        for (PowerDevice pDevc : pwrDevices.values()) {
            pState.addAll(pDevc.getPowerState());
        }

        return pState;
    }


   /**
    *  Turns on the power.
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.ACTION, description="Turn on the power")
    public void powerOn() throws Exception
    {
        for (PowerDevice pDevc : pwrDevices.values()) {
            pDevc.powerOn();
        }
    }


   /**
    *  Turns off the power.
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.ACTION, description="Turn off the power")
    public void powerOff() throws Exception
    {
        for (PowerDevice pDevc : pwrDevices.values()) {
            pDevc.powerOff();
        }
    }


   /**
    *  Gets the full power state.
    *
    *  @return  The full power state
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public PowerFullState getFullState() throws Exception
    {
        return new PowerFullState(new PowerState((int)getTickPeriod()),
                                  subsys.getMonitor().getFullState());
    }


   /**
    *  Terminates the program.
    */
    @Command(type=CommandType.ACTION, description="Terminate the program")
    public void quit()
    {
        System.exit(0);
    }


   /**
    *  Publishes the state of the power device.
    *
    *  This is intended to be called whenever any element of the state is
    *  changed.
    */
    void publishState()
    {
        KeyValueData kvd = new KeyValueData(PowerState.KEY,
                                            new PowerState((int)getTickPeriod()));
        subsys.publishSubsystemDataOnStatusBus(kvd);
    }    

}
