package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.power.config.RebPsEnum;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.data.PowerSupplyAgentProperties;
import org.lsst.ccs.subsystem.power.data.RebPsFullState;
import org.lsst.ccs.subsystem.power.data.RebPsState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 **********************************************
 *
 *  Implements the REB PS control subsystem.
 *
 *  @author Owen Saxton
 *
 **********************************************
 */
public class RebPower implements HasLifecycle, RebPsDevice.Event {

    private final Logger sLog = Logger.getLogger("org.lsst.ccs.subsystem.power");
    @LookupField( strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField( strategy=LookupField.Strategy.TREE)
    private Monitor mon;    
    @LookupField( strategy=LookupField.Strategy.CHILDREN)
    private PowerDevice mainPs;
    @LookupField( strategy=LookupField.Strategy.CHILDREN)    
    private RebPsDevice psDevice;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

   /**
    *  Initializes the power subsystem.
    */
    @Override
    public void postInit()
    {
        //Set a property to define that this Agent is a Power Supply controller.
        subsys.setAgentProperty(PowerSupplyAgentProperties.POWER_SUPPLY_TYPE_AGENT_PROPERTY,
                                RebPower.class.getCanonicalName());
        
        if (mainPs == null) {
            sLog.warning("No main power supply device specified");
        }

        if (psDevice == null) {
            sLog.error("No REB power devices specified");
        }
        else {
            psDevice.setListener(this);
        }

    }

   /**
    *  Sets the tick period
    */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    
    private long getTickPeriod()
    {
        return periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }
    
   /**
    *  Starts the subsystem.
    *
    *  @throws  HardwareException
    */
    @Override
    public void postStart() throws HardwareException
    {
        sLog.info("REB PS subsystem started");
        publishState();        // For any GUIs
    }

   /**
    *  Sets the update (tick) period.
    *
    *  @param  period  The update period (milliseconds) to set.
    */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)")
                                int period)
    {
        setTickPeriod(period);
        publishState();    // Must do this for GUI
    }


   /**
    *  Turns the main power supply on or off.
    *
    *  @param  on  Turns the power on if true, off if false
    *
    *  @throws  PowerException
    */
    @Command(type=CommandType.ACTION, description="Turn the main power supply on or off")
    public void setMainPower(@Argument(name="on", description="Whether on or off")
                             boolean on) throws Exception
    {
        try {
            if (on) {
                mainPs.powerOn();
                psDevice.checkOnline();
            }
            else {
                psDevice.sequencePower(false);
                mainPs.powerOff();
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


   /**
    *  Toggles the main power supply state.
    *
    *  @throws  PowerException
    */
    @Command(type=CommandType.ACTION, description="Toggle the main power supply state")
    public void toggleMainPower() throws Exception
    {
        setMainPower(getMainState() == PowerChanState.PWR_STATE_OFF);
    }


   /**
    *  Turns a power supply on or off.
    *
    *  @param  reb  The REB number
    *
    *  @param  ps   The power supply number, or -1 for master switch
    *
    *  @param  on   Turns the power on if true, off if false
    *
    *  @throws  PowerException
    */
    @Command(type=CommandType.ACTION, description="Turns a power supply on or off")
    public void setPowerOn(@Argument(name="reb", description="REB number")
                           int reb,
                           @Argument(name="ps", description="Power supply number")
                           int ps,
                           @Argument(name="on", description="Whether on or off")
                           boolean on) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.setPowerOn(reb, ps, on);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


   /**
    *  Turns a power supply on or off.
    *
    *  @param  reb     The REB number
    *
    *  @param  psEnum  The power supply enumeration
    *
    *  @param  on      Turns the power on if true, off if false
    *
    *  @throws  PowerException
    */
    @Command(type=CommandType.ACTION, description="Turns a power supply on or off")
    public void setNamedPowerOn(@Argument(name="reb", description="REB number")
                                int reb,
                                @Argument(name="name", description="Power supply name")
                                RebPsEnum psEnum,
                                @Argument(name="on", description="Whether on or off")
                                boolean on) throws PowerException
    {
        setPowerOn(reb, psEnum.getNumber(), on);
    }


   /**
    *  Toggles the on/off state of a power supply.
    *
    *  @param  reb  The REB number
    *
    *  @param  ps   The power supply number
    *
    *  @throws  PowerException
    */
    @Command(type=CommandType.ACTION, description="Toggle a power supply")
    public void togglePower(@Argument(name="reb", description="REB number")
                            int reb,
                            @Argument(name="ps", description="Power supply number")
                            int ps) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.togglePower(reb, ps);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


   /**
    *  Toggles the on/off state of a power supply.
    *
    *  @param  reb     The REB number
    *
    *  @param  psEnum  The power supply enumeration
    *
    *  @throws  PowerException
    */
    @Command(type=CommandType.ACTION, description="Toggle a power supply")
    public void toggleNamedPower(@Argument(name="reb", description="REB number")
                                 int reb,
                                 @Argument(name="name", description="Power supply name")
                                 RebPsEnum psEnum) throws PowerException
    {
        togglePower(reb, psEnum.getNumber());
    }


   /**
    *  Sequences a power supply on or off.
    *
    *  @param  reb  The REB number
    *
    *  @param  on   Turns on if true, off if false
    *
    *  @throws  PowerException
    */
    @Command(type=CommandType.ACTION, description="Sequence a power supply")
    public void sequencePower(@Argument(name="reb", description="REB number")
                              int reb,
                              @Argument(name="on", description="Whether to turn on")
                              boolean on) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.sequencePower(reb, on);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


   /**
    *  Sets a HV bias DAC value.
    *
    *  @param  reb    The REB number
    *
    *  @param  value  The value to set
    *
    *  @throws  PowerException
    */
    @Command(type=CommandType.ACTION, description="Set a HV bias DAC")
    public void setBiasDac(@Argument(name="reb", description="REB number")
                           int reb,
                           @Argument(name="value", description="DAC value")
                           double value) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.setBiasDac(reb, value);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


   /**
    *  Sets a DPHI DAC value.
    *
    *  @param  reb    The REB number
    *
    *  @param  value  The value to set
    *
    *  @throws  PowerException
    */
    @Command(type=CommandType.ACTION, description="Set a DPHI DAC")
    public void setDphiDac(@Argument(name="reb", description="REB number")
                           int reb,
                           @Argument(name="value", description="DAC value")
                           double value) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.setDphiDac(reb, value);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


   /**
    *  Gets the full REB power supply state.
    *
    *  @return  The full power supply state
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public RebPsFullState getFullState() throws Exception
    {
        return new RebPsFullState(getState(), mon.getFullState());
    }


   /**
    *  Gets the power supply error counters.
    *
    *  @return  The two-element array of error counters
    */
    @Command(type=CommandType.QUERY, description="Get the error counters")
    public int[] getErrors()
    {
        if (psDevice != null) {
            return psDevice.getErrors();
        }
        else {
            return new int[]{0, 0};
        }
    }


   /**
    *  Terminates the program.
    */
    @Command(type=CommandType.ACTION, description="Terminate the program")
    public void quit()
    {
        System.exit(0);
    }


   /**
    *  Publishes the state of the REB power supply.
    *
    *  This is intended to be called whenever any element of the state is
    *  changed.
    */
    void publishState()
    {
        KeyValueData kvd = new KeyValueData(RebPsState.KEY, getState());
        subsys.publishSubsystemDataOnStatusBus(kvd);
    }    


   /**
    *  Gets the state of the REB power supply.
    */
    RebPsState getState()
    {
        int[] psState;
        double[] biasDacs, dphiDacs;
        String psId = "PS-X";
        psState = new int[0];
        biasDacs = new double[0];
        dphiDacs = new double[0];
        if (psDevice != null) {
            psId = psDevice.getPsId();
            try {
                psState = psDevice.getState();
            }
            catch (PowerException e) {
            }
            try {
                biasDacs = psDevice.getBiasDacs();
            }
            catch (PowerException e) {
            }
            try {
                dphiDacs = psDevice.getDphiDacs();
            }
            catch (PowerException e) {
            }
        }

        return new RebPsState((int)getTickPeriod(), getMainState(), psState, biasDacs, dphiDacs, psId);
    }


   /**
    *  Gets the state of the main power supply.
    */
    int getMainState()
    {
        int state = PowerChanState.PWR_STATE_OFFLINE;
        if (mainPs != null && mainPs.isOnline()) {
            try {
                List<PowerChanState> pcState = mainPs.getPowerState();
                if (!pcState.isEmpty()) {
                    state = pcState.get(0).getState();
                }
            }
            catch (Exception e) {
            }
        }

        return state;
    }


   /**
    *  Called when a power supply device is opened.
    *
    *  Publishes the current state on the status bus.
    */
    @Override
    public void opened()
    {
        publishState();
    }


   /**
    *  Called when a power supply device is closed.
    *
    *  Publishes the current state on the status bus.
    */
    @Override
    public void closed()
    {
        publishState();
    }

}
