package org.lsst.ccs.subsystem.power;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.SubsystemConfigurationEnvironment;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 ********************************************************
 *
 *  Interface to an ASCII commanded power supply device.
 *
 *  @author Owen Saxton
 *
 ********************************************************
 */
public abstract class PowerDevice extends Device {

   /**
    *  Constants.
    */
    protected static final int
        NETWORK_CONN_INVALID = 0x01,
        CONN_TYPE_NOT_CONFIG = 0x02,
        DEVC_ID_NOT_CONFIG   = 0x04,
        DEVC_PARM_NOT_CONFIG = 0x08;

    private static final int
        MON_TYPE_VOLTAGE = 0,
        MON_TYPE_CURRENT = 1;

    private static final String
        CONN_TYPE = "connType",
        DEVC_ID   = "devcId",
        DEVC_PARM = "devcParm";

    private static final Map<String, DriverConstants.ConnType> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put(Power.ConnType.Network.name().toLowerCase(), DriverConstants.ConnType.NET);
        cTypeMap.put(Power.ConnType.Ftdi.name().toLowerCase(), DriverConstants.ConnType.FTDI);
        cTypeMap.put(Power.ConnType.Serial.name().toLowerCase(), DriverConstants.ConnType.SERIAL);
    }

    private static final Map<DriverConstants.ConnType, Power.ConnType> cTypeEMap = new HashMap();
    static {
        cTypeEMap.put(DriverConstants.ConnType.NET, Power.ConnType.Network);
        cTypeEMap.put(DriverConstants.ConnType.FTDI, Power.ConnType.Ftdi);
        cTypeEMap.put(DriverConstants.ConnType.SERIAL, Power.ConnType.Serial);
    }

    private static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
    }

   /**
    *  Data fields.
    */
    @ConfigurationParameter(name=DEVC_ID, category=Power.POWER)
    protected String devcId = "";
    @ConfigurationParameter(name=DEVC_PARM, category=Power.POWER)
    protected int devcParm;
    @ConfigurationParameter(name=CONN_TYPE, category=Power.POWER)
    protected String connType = "";

    private final String devcName;
    protected PowerSupplyDriver psd;
    private DriverConstants.ConnType connTypeE;
    private final int options, minChan, maxChan;


   /**
    *  Constructor.
    *
    *  @param  name      The device name
    *
    *  @param  psd       The power supply device driver object
    *
    *  @param  connType  The connection type: network; ftdi; serial
    *
    *  @param  devcId    The device ID: host name; USB serial no.; serial
    *                    device name
    *
    *  @param  devcParm  The device parameter: port number; baud rate;
    *                    baud rate
    *
    *  @param  options   The options word
    *
    *  @param  minChan   The minimum channel number
    *
    *  @param  maxChan   The maximum channel number
    */
    public PowerDevice(String name, PowerSupplyDriver psd, String connType, String devcId,
                       int devcParm, int options, int minChan, int maxChan)
    {
        super();
        this.devcName = name;
        this.psd = psd;
        this.connType = connType;
        this.devcId = devcId;
        this.devcParm = devcParm;
        this.options = options;
        this.minChan = minChan;
        this.maxChan = maxChan;
    }


   /**
    *  Gets the minimum channel number.
    *
    *  @return  The minimum channel number
    */
    public int getMinChannel()
    {
        return minChan;
    }


   /**
    *  Gets the maximum channel number.
    *
    *  @return  The maximum channel number
    */
    public int getMaxChannel()
    {
        return maxChan;
    }

   
   /**
    *  Performs configuration.
    *
    */
    @Override
    protected void initDevice() {
        DriverConstants.ConnType cType = cTypeMap.get(connType.toLowerCase());
        if (cType == null || (cType == DriverConstants.ConnType.NET
                              && (options & NETWORK_CONN_INVALID) != 0)) {
            MonitorLogUtils.reportConfigError(log, getName(), "hdwType", "is invalid");
        }
        connTypeE = cType;
    }


   /**
    *  Sets the connection type.
    *
    *  @param  connType  The connection type string
    *
    *  @throws  PowerException
    */
    @ConfigurationParameterChanger
    public void setConnType(String connType) throws PowerException
    {
        if (connType.equalsIgnoreCase(this.connType)) return;
        DriverConstants.ConnType cType = cTypeMap.get(connType);
        if (cType == null
              || (cType == DriverConstants.ConnType.NET && (options & NETWORK_CONN_INVALID) != 0)) {
            throw new PowerException("Invalid connection type");
        }
        this.connType = connType;
        this.connTypeE = cType;
        reInitialize();
    }


   /**
    *  Gets the connection type.
    *
    *  @return  The connection type string
    */
    @Command(type=CommandType.QUERY, description="Gets the connection type")
    public String getConnType()
    {
        return connType;
    }


   /**
    *  Sets the device ID.
    *
    *  @param  devcId  The device identification string
    */
    @ConfigurationParameterChanger
    public void setDevcId(String devcId)
    {
        if (devcId.equals(this.devcId)) return;
        this.devcId = devcId;
        reInitialize();
    }


   /**
    *  Gets the device ID.
    *
    *  @return  The device identification string
    */
    @Command(type=CommandType.QUERY, description="Gets the device ID")
    public String getDevcId()
    {
        return devcId;
    }


   /**
    *  Sets the device parameter.
    *
    *  @param  devcParm  The device parameter
    */
    @ConfigurationParameterChanger
    public void setDevcParm(int devcParm)
    {
        if (devcParm == this.devcParm) return;
        this.devcParm = devcParm;
        reInitialize();
    }


   /**
    *  Gets the device parameter.
    *
    *  @return  The device parameter
    */
    @Command(type=CommandType.QUERY, description="Gets the device parameter")
    public int getDevcParm()
    {
        return devcParm;
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize()
    {
        if (!inited) {
            fullName = devcName + (devcId.isEmpty() ? "" : " (" + devcId + ")");
        }
        try {
            psd.open(connTypeE, devcId, devcParm);
            setOnline(true);
            log.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close()
    {
        try {
            psd.close();
        }
        catch (DriverException e) {
        }
    }


   /**
    *  Re-initializes the device.
    */
    protected void reInitialize()
    {
        if (inited) {
            close();
            inited = false;
            initialize();
        }
    }


   /**
    *  Checks a monitoring channel's parameters for validity.
    *
    *  @param  name     The channel name
    *
    *  @param  hwChan   The hardware channel
    *
    *  @param  type     The channel type string
    *
    *  @param  subtype  The channel subtype string
    *
    *  @return  Two-element array containg the encoded type [0] and
    *           subtype [1] 
    *
    *  @throws  Exception  If parameters are invalid
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        }
        if (hwChan < minChan || hwChan > maxChan) {
            MonitorLogUtils.reportError(log, getName(), "hw channel number", hwChan);
        }
        
        return new int[]{mType, 0};
    }


   /**
    *  Reads a monitoring channel.
    *
    *  @param  hwChan  The hardware channel number
    *
    *  @param  type    The encoded channel type
    *
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        if (online) {
            try {
                value = (type == MON_TYPE_VOLTAGE) ? psd.readVoltage(hwChan) : psd.readCurrent(hwChan);
            }
            catch (DriverException e) {
                log.error("Error reading from " + fullName + ": " + e);
                setOnline(false);
            }
        }

        return value;
    }


   /**
    *  Gets the configuration data.
    *
    *  @return  A power configuration object
    */
    public Power getPowerConfig()
    {
        Power power = new Power();
        power.setName(getName());
        power.setConnType(cTypeEMap.get(connType));
        power.setDevcId(devcId);
        power.setDevcParm(devcParm);
        Power.Channel[] chans = new Power.Channel[ctlChans.size()];
        int j = 0;
        for (Control ctl : ctlChans.values()) {
            chans[j++] = ((PowerControl)ctl).getConfig();
        }
        power.setChannels(chans);

        return power;
    }


   /**
    *  Sets the configuration data.
    *
    *  @param  power  A power configuration object
    *
    *  @throws  Exception
    */
    public void setPowerConfig(Power power) throws Exception
    {
        SubsystemConfigurationEnvironment sce = getSubsystem().getSubsystemConfigurationEnvironment();
        String dName = getName();
        
        if ((options & CONN_TYPE_NOT_CONFIG) == 0) {
            sce.change(dName, CONN_TYPE, power.getConnType().name());
        }
        if ((options & DEVC_ID_NOT_CONFIG) == 0) {
            sce.change(dName, DEVC_ID, power.getDevcId());
        }
        if ((options & DEVC_PARM_NOT_CONFIG) == 0) {
            sce.change(dName, DEVC_PARM, power.getDevcParm());
        }
        for (Power.Channel chan : power.getChannels()) {
            String cName = dName + "." + chan.getName();
            ((PowerControl)ctlChans.get(cName)).setConfig(chan);
        }
    }


   /**
    *  Gets the list of power state data.
    *
    *  @return  A list of power state data
    *
    *  @throws  Exception
    */
    protected List<PowerChanState> getPowerState() throws Exception
    {
        try {
            readVoltageGroup();
            readCurrentGroup();
            readOutputGroup();
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
        List<PowerChanState> pState = new ArrayList<>();
        for (Control ctl : ctlChans.values()) {
            pState.add(((PowerControl)ctl).getState());
        }
   
        return pState;
    }


   /**
    *  Powers on all channels.
    *
    *  @throws  Exception
    */
    protected void powerOn() throws Exception
    {
        for (Control ctl : ctlChans.values()) {
            ((PowerControl)ctl).writeAll();
        }
        try {
            writeVoltageGroup();
            writeCurrentGroup();
            writeOnDelayGroup();
            writeOffDelayGroup();
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
        setOutput(true);
    }


   /**
    *  Powers off all channels.
    *
    *  @throws  Exception
    */
    protected void powerOff() throws Exception
    {
        setOutput(false);
    }


   /**
    *  Writes the voltage to the hardware.
    *
    *  @param  voltage  The voltage to set
    *
    *  @param  hwChan   The hardware channel number
    *
    *  @throws  Exception
    */
    protected void writeVoltage(double voltage, int hwChan) throws Exception
    {
        try {
            psd.setVoltage(voltage, hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    *  Reads the voltage from the hardware.
    *
    *  @param  hwChan   The hardware channel number
    *
    *  @return  The read voltage
    *
    *  @throws  Exception
    */
    protected double readVoltage(int hwChan) throws Exception
    {
        try {
            return psd.readVoltage(hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
            return 0;
        }
    }


   /**
    *  Writes the current to the hardware.
    *
    *  @param  current  The current to set
    *
    *  @param  hwChan   The hardware channel number
    *
    *  @throws  Exception
    */
    protected void writeCurrent(double current, int hwChan) throws Exception
    {
        try {
            psd.setCurrent(current, hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    *  Reads the current from the hardware.
    *
    *  @param  hwChan   The hardware channel number
    *
    *  @return  The read current
    *
    *  @throws  Exception
    */
    protected double readCurrent(int hwChan) throws Exception
    {
        try {
            return psd.readCurrent(hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
            return 0;
        }
    }


   /**
    *  Writes the power-on delay to the hardware.
    *
    *  @param  onDelay  The delay to set
    *
    *  @param  hwChan   The hardware channel number
    *
    *  @throws  Exception
    */
    protected void writeOnDelay(double onDelay, int hwChan) throws Exception
    {
        try {
            psd.setOnDelay(onDelay, hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    *  Writes the power-off delay to the hardware.
    *
    *  @param  offDelay  The delay to set
    *
    *  @param  hwChan    The hardware channel number
    *
    *  @throws  Exception
    */
    protected void writeOffDelay(double offDelay, int hwChan) throws Exception
    {
        try {
            psd.setOffDelay(offDelay, hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    *  Writes the power output state to the hardware.
    *
    *  @param  value   The output state to set
    *
    *  @param  hwChan  The hardware channel number
    *
    *  @throws  Exception
    */
    protected void writeOutput(boolean value, int hwChan) throws Exception
    {
        try {
            psd.setOutput(value, hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    *  Reads the power output state from the hardware.
    *
    *  @param  hwChan   The hardware channel number
    *
    *  @return  The power output state
    *
    *  @throws  Exception
    */
    protected boolean readOutput(int hwChan) throws Exception
    {
        try {
            return psd.getOutput(hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
            return false;
        }
    }


   /**
    *  Sets the power output state for all channels.
    *
    *  @param  value  The output state to set
    *
    *  @throws  Exception
    */
    private void setOutput(boolean value) throws Exception
    {
        try {
            for (Control ctl : ctlChans.values()) {
                psd.setOutput(value, ctl.getHwChan());
            }
            writeOutputGroup();
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    *  Writes a group of voltages to the hardware.
    *
    *  May be overridden for devices with this capability
    *
    *  @throws  DriverException
    */
    protected void writeVoltageGroup() throws DriverException
    {
    }


   /**
    *  Reads a group of voltages from the hardware.
    *
    *  May be overridden for devices with this capability
    *
    *  @throws  DriverException
    */
    protected void readVoltageGroup() throws DriverException
    {
    }


   /**
    *  Writes a group of currents to the hardware.
    *
    *  May be overridden for devices with this capability
    *
    *  @throws  DriverException
    */
    protected void writeCurrentGroup() throws DriverException
    {
    }


   /**
    *  Reads a group of currents from the hardware.
    *
    *  May be overridden for devices with this capability
    *
    *  @throws  DriverException
    */
    protected void readCurrentGroup() throws DriverException
    {
    }


   /**
    *  Writes a group of power-on delays to the hardware.
    *
    *  May be overridden for devices with this capability
    *
    *  @throws  DriverException
    */
    protected void writeOnDelayGroup() throws DriverException
    {
    }


   /**
    *  Writes a group of power-off delays to the hardware.
    *
    *  May be overridden for devices with this capability
    *
    *  @throws  DriverException
    */
    protected void writeOffDelayGroup() throws DriverException
    {
    }


   /**
    *  Writes a group of power output states to the hardware.
    *
    *  May be overridden for devices with this capability
    *
    *  @throws  DriverException
    */
    protected void writeOutputGroup() throws DriverException
    {
    }


   /**
    *  Reads a group of power output states from the hardware.
    *
    *  May be overridden for devices with this capability
    *
    *  @throws  DriverException
    */
    protected void readOutputGroup() throws DriverException
    {
    }

}
