package org.lsst.ccs.subsystem.power;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.SubsystemConfigurationEnvironment;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.MonitorCommands;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerFullState;
import org.lsst.ccs.subsystem.power.data.PowerState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ********************************************
 *
 *  Implements the power control subsystem.
 *
 *  @author Owen Saxton
 *
 ********************************************
 */
public class MainPower extends Module implements HardwareController {

    /*
     *  Constants
     */
    private static final int
        TICK_INTERVAL = 10000;

   /**
    *  Private fields.
    */
    private int monMillis = 1000;

    private Map<String, PowerDevice> pwrDevices;
    private final Logger sLog = Logger.getLogger("org.lsst.ccs.subsystem.power");
    private Subsystem subsys;
    private SubsystemConfigurationEnvironment sce;
    private Monitor mon;


   /**
    *  Initializes the power subsystem.
    */
    @Override
    public void initModule()
    {
        /*
        *  General initialization
        */
        subsys = getSubsystem();
        sce = subsys.getSubsystemConfigurationEnvironment();

        /*
        *  Initialize devices
        */
        mon = new Monitor(this, null, sLog);
        mon.initConfiguration();

        /*
        *  Make monitor commands available
        */
        subsys.addCommandsFromObject(new MonitorCommands(mon), "");

        /*
        *  Get all the power devices
        */
        pwrDevices = subsys.getComponentLookup().getChildren(getName(), PowerDevice.class);
        if (pwrDevices.isEmpty()) {
            sLog.error("No power devices specified");
        }
    }


   /**
    *  Initializes the hardware.
    *
    *  @return  GO
    */
    @Override
    public TreeWalkerDiag checkHardware()
    {
        mon.initSensors();

        return TreeWalkerDiag.GO;
    }


   /**
    *  Checks whether hardware is started.
    */
    @Override
    public void checkStarted()
    {
    }


   /**
    *  Checks whether hardware is stopped.
    */
    @Override
    public void checkStopped()
    {
    }


   /**
    *  Starts the subsystem.
    *
    *  @throws  HardwareException
    */
    @Override
    public void postStart() throws HardwareException
    {
        if (getTickMillis() <= 0) {
            sce.change(getName(), "tickMillis", TICK_INTERVAL);
        }
        mon.start(monMillis);
        sLog.info("Power subsystem started");
        publishState();        // For any GUIs
        mon.publishState();    // For any GUIs
        mon.publishLimits();   // For GUIs and the trending database
    }


   /**
    *  Performs periodic power data broadcast.
    */
    @Override
    public void tick()
    {
        if (pwrDevices.isEmpty()) return;
        ArrayList<PowerChanState> pState = new ArrayList<>();
        for (PowerDevice pDevc : pwrDevices.values()) {
            try {
                pState.addAll(pDevc.getPowerState());
            }
            catch (Exception e) {
            }
        }
        KeyValueData kvd = new KeyValueData(PowerChanState.KEY, pState);
        subsys.publishSubsystemDataOnStatusBus(kvd);
        mon.publishData();
    }


   /**
    *  Sets the update (tick) period.
    *
    *  @param  period  The update period (milliseconds) to set.
    */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(int period)
    {
        sce.change(getName(), "tickMillis", period);
        publishState();    // Must do this for GUI
    }


   /**
    *  Saves the configuration data.
    *
    *  @param  name  The configuration name
    *
    *  @throws IOException
    */
    @Command(type=CommandType.ACTION, description="Save the current configuration")
    public void saveNamedConfig(String name) throws IOException
    {
        sce.saveChangesForCategoriesAs(Power.POWER + ":" + name);
    }


   /**
    *  Gets the configuration data.
    *
    *  @return  The array of power configuration objects
    */
    @Command(type=CommandType.QUERY, description="Get the configuration data")
    public Power[] getPowerConfig()
    {
        Power[] power = new Power[pwrDevices.size()];
        int j = 0;
        for (PowerDevice pDevc : pwrDevices.values()) {
            power[j++] = pDevc.getPowerConfig();
        }

        return power;
    }


   /**
    *  Sets the configuration data.
    *
    *  @param  power  The array of power configuration objects
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.ACTION, description="Set the configuration data")
    public void setPowerConfig(Power[] power) throws Exception
    {
        for (Power pConf : power) {
            PowerDevice pDevc = pwrDevices.get(pConf.getName());
            pDevc.setPowerConfig(pConf);
        }
    }


   /**
    *  Gets the power state.
    *
    *  @return  The power state
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.QUERY, description="Get the power state")
    public List<PowerChanState> getPowerState() throws Exception
    {
        List<PowerChanState> pState = new ArrayList<>();
        for (PowerDevice pDevc : pwrDevices.values()) {
            pState.addAll(pDevc.getPowerState());
        }

        return pState;
    }


   /**
    *  Turns on the power.
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.ACTION, description="Turn on the power")
    public void powerOn() throws Exception
    {
        for (PowerDevice pDevc : pwrDevices.values()) {
            pDevc.powerOn();
        }
    }


   /**
    *  Turns off the power.
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.ACTION, description="Turn off the power")
    public void powerOff() throws Exception
    {
        for (PowerDevice pDevc : pwrDevices.values()) {
            pDevc.powerOff();
        }
    }


   /**
    *  Gets the full power state.
    *
    *  @return  The full power state
    *
    *  @throws  Exception
    */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public PowerFullState getFullState() throws Exception
    {
        return new PowerFullState(new PowerState(getTickMillis()),
                                  mon.getFullState());
    }


   /**
    *  Terminates the program.
    */
    @Command(type=CommandType.ACTION, description="Terminate the program")
    public void quit()
    {
        System.exit(0);
    }


   /**
    *  Publishes the state of the power module.
    *
    *  This is intended to be called whenever any element of the state is
    *  changed.
    */
    void publishState()
    {
        KeyValueData kvd = new KeyValueData(PowerState.KEY,
                                            new PowerState(getTickMillis()));
        subsys.publishSubsystemDataOnStatusBus(kvd);
    }    

}
