package org.lsst.ccs.subsystem.power;

import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.subsystem.monitor.Control;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;

/**
 ***********************************************
 *
 *  Implements power supply control functions.
 *
 *  @author Owen Saxton
 *
 ***********************************************
 */
public class PowerControl extends Control {

   /**
    *  Constants.
    */
    private static final String
        VOLTAGE   = "voltage",
        CURRENT   = "current",
        ON_DELAY  = "onDelay",
        OFF_DELAY = "offDelay";

   /**
    *  Data fields.
    */
    // Supplied configurable fields
    @ConfigurationParameter(name=VOLTAGE, category=Power.POWER)
    private double      voltage;       // Voltage
    @ConfigurationParameter(name=CURRENT, category=Power.POWER)
    private double      current;       // Current
    @ConfigurationParameter(name=ON_DELAY, category=Power.POWER)
    private double      onDelay;       // Turn-on delay
    @ConfigurationParameter(name=OFF_DELAY, category=Power.POWER)
    private double      offDelay;      // Turn-off delay

    // Derived fields
    private String      shortName;     // Last part of dot-separated name
    private PowerDevice pDevc;         // Parent (power supply) device


   /**
    *  Constructor.
    *
    *  @param  desc      Description
    *
    *  @param  hwChan    Hardware channel number
    *
    *  @param  voltage   The set voltage
    *
    *  @param  current   The set current limit
    *
    *  @param  onDelay   The turn-on delay (sec)
    *
    *  @param  offDelay  The turn-off delay (sec)
    */
    public PowerControl(String desc, int hwChan,
                        double voltage, double current, double onDelay, double offDelay)
    {
        super(desc, hwChan);
        this.voltage = voltage;
        this.current = current;
        this.onDelay = onDelay;
        this.offDelay = offDelay;
    }


   /**
    *  Configures channel description.
    *
    *  @param  mon   The associated monitor
    *
    *  @param  devc  The parent Device
    */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        pDevc = (PowerDevice)devc;
        try {
            if (hwChan < pDevc.getMinChannel()
                  || hwChan > pDevc.getMaxChannel()) {
                mon.reportError(getName(), "hw channel number", hwChan);
            }
        }
        catch (Exception e) {
        }
        String[] parts = getName().split("\\.");
        shortName = parts[parts.length - 1];
    }


   /**
    *  Sets/gets the voltage.
    *
    *  @param  value  The value to set
    */
    @ConfigurationParameterChanger
    public void setVoltage(double value)
    {
        voltage = value;
    }

    @Command(type=Command.CommandType.QUERY, description="Gets the set voltage")
    public double getVoltage()
    {
        return voltage;
    }


   /**
    *  Sets/gets the current.
    *
    *  @param  value  The value to set
    */
    @ConfigurationParameterChanger
    public void setCurrent(double value)
    {
        current = value;
    }

    @Command(type=Command.CommandType.QUERY, description="Gets the set current")
    public double getCurrent()
    {
        return current;
    }


   /**
    *  Sets/gets the power-on delay.
    *
    *  @param  value  The value to set
    */
    @ConfigurationParameterChanger
    public void setOnDelay(double value)
    {
        onDelay = value;
    }

    @Command(type=Command.CommandType.QUERY,
             description="Gets the power-on delay")
    public double getOnDelay()
    {
        return onDelay;
    }


   /**
    *  Sets/gets the power-off delay.
    *
    *  @param  value  The value to set
    */
    @ConfigurationParameterChanger
    public void setOffDelay(double value)
    {
        offDelay = value;
    }

    @Command(type=Command.CommandType.QUERY,
             description="Gets the power-off delay")
    public double getOffDelay()
    {
        return offDelay;
    }


   /**
    *  Changes the voltage.
    *
    *  @param  value  The value to set
    *
    *  @throws  Exception
    */
    @Command(type=Command.CommandType.ACTION, description="Changes the voltage")
    public void changeVoltage(double value) throws Exception
    {
        pDevc.writeVoltage(value, hwChan);
    }


   /**
    *  Adds to a map of configured values.
    *
    *  @param  values  The map to add to
    */
    @Override
    public void addConfigValues(Map<String, Object> values)
    {
        String stem = getName() + ".";
        values.put(stem + VOLTAGE, voltage);
        values.put(stem + CURRENT, current);
        values.put(stem + ON_DELAY, onDelay);
        values.put(stem + OFF_DELAY, offDelay);
    }


   /**
    *  Sets all the configuration data.
    */
    void setConfig(Power.Channel chan) throws Exception
    {
        change(VOLTAGE, chan.getVoltage());
        change(CURRENT, chan.getCurrent());
        change(ON_DELAY, chan.getOnDelay());
        change(OFF_DELAY, chan.getOffDelay());
    }


   /**
    *  Gets all configuration data.
    */
    Power.Channel getConfig()
    {
        Power.Channel chan = new Power.Channel();
        chan.setName(shortName);
        chan.setVoltage(voltage);
        chan.setCurrent(current);
        chan.setOnDelay(onDelay);
        chan.setOffDelay(offDelay);

        return chan;
    }


   /**
    *  Gets state data from the hardware.
    */
    PowerChanState getState()
    {
        PowerChanState pwr = new PowerChanState();
        pwr.setName(shortName);
        try {
            pwr.setVoltage(readVoltage());
            pwr.setCurrent(readCurrent());
            pwr.setState(readOutput() ? PowerChanState.PWR_STATE_ON
                                      : PowerChanState.PWR_STATE_OFF);
        }
        catch (Exception e) {
            log.warning("Error reading power state: " + e);
            pwr.setVoltage(0);
            pwr.setCurrent(0);
            pwr.setState(PowerChanState.PWR_STATE_OFFLINE);
        }

        return pwr;
    }


   /**
    *  Writes all configured quantities to the hardware.
    */
    void writeAll() throws Exception
    {
        writeVoltage();
        writeCurrent();
        writeOnDelay();
        writeOffDelay();
    }


   /**
    *  Writes the voltage to the hardware.
    */
    void writeVoltage() throws Exception
    {
        pDevc.writeVoltage(voltage, hwChan);
    }


   /**
    *  Reads the voltage from the hardware.
    */
    double readVoltage() throws Exception
    {
        return pDevc.readVoltage(hwChan);
    }


   /**
    *  Writes the current to the hardware.
    */
    void writeCurrent() throws Exception
    {
        pDevc.writeCurrent(current, hwChan);
    }


   /**
    *  Reads the current from the hardware.
    */
    double readCurrent() throws Exception
    {
        return pDevc.readCurrent(hwChan);
    }


   /**
    *  Writes the power-on delay to the hardware.
    */
    void writeOnDelay() throws Exception
    {
        pDevc.writeOnDelay(onDelay, hwChan);
    }


   /**
    *  Writes the power-off delay to the hardware.
    */
    void writeOffDelay() throws Exception
    {
        pDevc.writeOffDelay(offDelay, hwChan);
    }


   /**
    *  Writes the power output state to the hardware.
    */
    void writeOutput(boolean value) throws Exception
    {
        pDevc.writeOutput(value, hwChan);
    }


   /**
    *  Reads the power output state from the hardware.
    */
    boolean readOutput() throws Exception
    {
        return pDevc.readOutput(hwChan);
    }

}
