package org.lsst.ccs.subsystem.power;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.drivers.rebps.RebPs;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.framework.Configurable;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 ******************************************************************************
 **
 **  Interface to a REB power supply board.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RebPsDevice extends Device {

   /**
    ***************************************************************************
    **
    **  Inner class for holding power supply sequencing parameters.
    **
    ***************************************************************************
    */
    static class SeqParams {

        int    id;
        double minGood;
        double maxGood;
        int    depId;

        SeqParams(int id, double minGood, double maxGood, int depId)
        {
            this.id = id;
            this.minGood = minGood;
            this.maxGood = maxGood;
            this.depId = depId;
        }

    }

   /**
    ***************************************************************************
    **
    **  Interface for open/close event listener.
    **
    ***************************************************************************
    */
    public interface Event {

        public void opened();

        public void closed();

    }

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private static final String
        DEVC_ID   = "devcId",
        DEVC_PARM = "devcParm",
        UNKN_PSID = "PS-XX";

    private static final Map<String, Integer> psMap = new HashMap<>();
    static {
        psMap.put("DIGITAL", RebPs.PS_DIGITAL);
        psMap.put("ANALOG", RebPs.PS_ANALOG);
        psMap.put("CLOCKHI", RebPs.PS_CLK_HIGH);
        psMap.put("CLOCKLO", RebPs.PS_CLK_LOW);
        psMap.put("DPHI", RebPs.PS_DPHI);
        psMap.put("HEATER", RebPs.PS_HEATER);
        psMap.put("HVBIAS", RebPs.PS_HV_BIAS);
        psMap.put("OD", RebPs.PS_OD);
    }

    private static final int[]
        offSeq = {RebPs.PS_HV_BIAS, RebPs.PS_HEATER, RebPs.PS_CLK_HIGH,
                  RebPs.PS_CLK_LOW, RebPs.PS_OD, RebPs.PS_ANALOG,
                  RebPs.PS_DIGITAL};

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private String devcId;
    private int devcParm;
    private RebPs ps;
    private final List<SeqParams> seqP = new ArrayList<>();
    private Event listener;
    private String psId = UNKN_PSID;
    private Properties props;


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  devcId    The host name
    **
    **  @param  devcParm  The port number
    **
    ***************************************************************************
    */
    public RebPsDevice(String devcId, int devcParm)
    {
        super();
        this.devcId = devcId;
        this.devcParm = devcParm;
        seqP.add(new SeqParams(RebPs.PS_DIGITAL, 4.9, 6.0, -1));
        seqP.add(new SeqParams(RebPs.PS_ANALOG, 6.9, 8.0, RebPs.PS_DIGITAL));
        seqP.add(new SeqParams(RebPs.PS_HEATER, 8.0, 9.0, RebPs.PS_ANALOG));
        seqP.add(new SeqParams(RebPs.PS_CLK_LOW, 14.5, 16.0, RebPs.PS_HEATER));
        seqP.add(new SeqParams(RebPs.PS_CLK_HIGH, 15.0, 16.0, RebPs.PS_CLK_LOW));
        seqP.add(new SeqParams(RebPs.PS_OD, 40.0, 41.0, RebPs.PS_CLK_HIGH));
    }


   /**
    ***************************************************************************
    **
    **  Performs configuration.
    **
    **  @param  mon  The monitor object
    **
    ***************************************************************************
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        Configurable.Environment env = getEnvironment();
        String param = null;
        try {
            param = DEVC_ID;
            env.getCheckedValueFromConfiguration(param, devcId);
            param = DEVC_PARM;
            env.getCheckedValueFromConfiguration(param, devcParm);
        }
        catch (Exception e) {
            mon.reportConfigError(getName(), param, "is missing");
        }

        String fileName = "psid.properties";
        if (BootstrapResourceUtils.getBootstrapResource(fileName) == null) {
            log.warn("PS ID properties file (" + fileName + ") not found");
        }
        props = BootstrapResourceUtils.getBootstrapProperties(fileName);
    }


   /**
    ***************************************************************************
    **
    **  Sets the device ID.
    **
    **  @param  devcId  The device identification string
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setDevcId(String devcId)
    {
        if (devcId.equals(this.devcId)) return;
        this.devcId = devcId;
        close();
        inited = false;
        initialize();
    }


   /**
    ***************************************************************************
    **
    **  Gets the device ID.
    **
    **  @return  The device identification string
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Gets the device ID")
    public String getDevcId()
    {
        return devcId;
    }


   /**
    ***************************************************************************
    **
    **  Sets the device parameter.
    **
    **  @param  devcParm  The device parameter
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setDevcParm(int devcParm)
    {
        if (devcParm == this.devcParm) return;
        this.devcParm = devcParm;
        close();
        inited = false;
        initialize();
    }


   /**
    ***************************************************************************
    **
    **  Gets the device parameter.
    **
    **  @return  The device parameter
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Gets the device parameter")
    public int getDevcParm()
    {
        return devcParm;
    }


   /**
    ***************************************************************************
    **
    **  Performs full initialization.
    **
    ***************************************************************************
    */
    @Override
    protected void initialize()
    {
        if (!inited) {
            fullName = "REB PS board (" + devcId + ")";
        }
        try {
            if (!inited) {
                ps = new RebPs();
            }
            if (devcParm == 0) {
                ps.open(devcId);
            }
            else {
                ps.open(devcId, devcParm);
            }
            ps.setTemperatureRes(3);
            generatePsId();
            setOnline(true);
            String message = "Connected to " + fullName;
            if (listener != null) {
                listener.opened();
            }
            if (!inited) {
                log.info(message);
            }
            else {
                log.error(message);
            }
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    protected void close()
    {
        try {
            ps.close();
        }
        catch (DriverException e) {
        }
        finally {
            if (listener != null) {
                listener.closed();
            }
            psId = UNKN_PSID;
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a monitoring channel's parameters for validity.
    **
    **  @param  name     The channel name
    **
    **  @param  hwChan   The hardware channel
    **
    **  @param  type     The channel type string
    **
    **  @param  subtype  The channel subtype string
    **
    **  @return  Two-element array containg the encoded type [0] and
    **           subtype [1] 
    **
    **  @throws  Exception  If parameters are invalid
    **
    ***************************************************************************
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        int rebNum = 0;
        Integer psNum = null;
        String[] flds = type.split(":");
        if (flds.length == 1) {
            if (flds[0].toUpperCase().equals("TEMP")) {
                psNum = -1;
            }
        }
        else if (flds.length == 2) {
            try {
                rebNum = Integer.valueOf(flds[0]);
                if (RebPs.testRebNumber(rebNum)) {
                    psNum = psMap.get(flds[1].toUpperCase());
                }
            }
            catch (NumberFormatException e) {
            }
        }
        if (psNum == null) {
            mon.reportError(name, "type", type);
        }
        if (psNum >= 0 && !RebPs.testChannelNumber(psNum, hwChan)) {
            mon.reportError(name, "hw channel number", hwChan);
        }
        
        return new int[]{(rebNum << 8) | psNum, 0};
    }


   /**
    ***************************************************************************
    **
    **  Reads a monitoring channel.
    **
    **  @param  hwChan  The hardware channel number
    **
    **  @param  type    The encoded channel type
    **
    **  @return  The read value
    **
    ***************************************************************************
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        if (online) {
            try {
                if (type < 0) {
                    value = ps.readTemperature();
                }
                else {
                    value = ps.readChannel(type >> 8, type & 0xff, hwChan);
                }
            }
            catch (DriverTimeoutException e) {
                log.error("Error reading from " + fullName + ": " + e);
                //setOnline(false);  //Annoying and not useful
            }
            catch (DriverException e) {
                // Channel is not powered
            }
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Adds to the map of configured values.
    **
    **  @param  values  The map of values to be added to
    **
    ***************************************************************************
    */
    @Override
    protected void addConfigValues(Map<String, Object> values)
    {
        String stem = getName() + ".";
        values.put(stem + DEVC_ID, devcId);
        values.put(stem + DEVC_PARM, devcParm);
    }


   /**
    ***************************************************************************
    **
    **  Gets the power state data.
    **
    **  @return  A list of power state data
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    protected int[] getState() throws PowerException
    {
        try {
            int[] state = new int[RebPs.NUM_REBS];
            for (int reb = 0; reb < state.length; reb++) {
                state[reb] = ps.getPower(reb);
            }
            return state;
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the HV bias DAC values.
    **
    **  @return  An array of bias DAC values
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    protected double[] getBiasDacs() throws PowerException
    {
        try {
            double[] dacs = new double[RebPs.NUM_REBS];
            for (int reb = 0; reb < dacs.length; reb++) {
                dacs[reb] = ps.readChannel(reb, RebPs.PS_HV_BIAS,
                                           RebPs.CHAN_VOLT_DAC);
            }
            return dacs;
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets a HV bias DAC value.
    **
    **  @param  rebNum  The REB number
    **
    **  @param  value   The value to set
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    protected void setBiasDac(int rebNum, double value) throws PowerException
    {
        try {
            ps.writeDac(rebNum, RebPs.PS_HV_BIAS, value);
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    ***************************************************************************
    **
    **  Toggles the on/off state of a power supply element.
    **
    **  @param  reb    The REB number
    **
    **  @param  psNum  The power supply number, or -1 for master switch
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    protected void togglePower(int reb, int psNum) throws PowerException
    {
        try {
            boolean hvDacOff = false;
            int mask = ps.getPower(reb);
            boolean mainOn = ((mask & 1) != 0);
            if (psNum < 0) {
                if (mainOn) {
                    for (int psn : offSeq) {
                        mask &= ~(1 << (psn + 1));
                        ps.setPower(reb, mask);
                    }
                    hvDacOff = true;
                }
                ps.setPower(reb, mainOn ? 0 : 1);
            }
            else if (mainOn) {
                ps.setPower(reb, mask ^ (1 << (psNum + 1)));
                hvDacOff = psNum == RebPs.PS_HV_BIAS
                             && (mask & (1 << (psNum + 1))) != 0;
            }
            if (hvDacOff) {
                ps.writeDac(reb, RebPs.PS_HV_BIAS, 0);
            }
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    ***************************************************************************
    **
    **  Sequences a power supply section on or off.
    **
    **  @param  reb  The REB number
    **
    **  @param  on   Turns on if true, off if false
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    protected void sequencePower(int reb, boolean on) throws PowerException
    {
        try {
            int value = ps.getPower(reb);
            if (on) {
                value |= 1;
                ps.setPower(reb, value);
                for (int j = 0; j < seqP.size(); j++) {
                    value |= 1 << (seqP.get(j).id + 1);
                    ps.setPower(reb, value);
                }
            }
            else {
                value &= ~(1 << (RebPs.PS_HV_BIAS + 1));
                ps.setPower(reb, value);
                for (int j = seqP.size() - 1; j >= 0; j--) {
                    value &= ~(1 << (seqP.get(j).id + 1));
                    ps.setPower(reb, value);
                }
                value = 0;
                ps.setPower(reb, value);
            }
        }
        catch (DriverException e) {
            throw new PowerException(e.getMessage());
        }
    }


   /**
    ***************************************************************************
    **
    **  Sequences a whole power supply on or off.
    **
    **  @param  on   Turns on if true, off if false
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    protected void sequencePower(boolean on) throws PowerException
    {
        for (int reb = 0; reb < RebPs.NUM_REBS; reb++) {
            sequencePower(reb, on);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets an event listener.
    **
    **  @param  listen  The listener object
    **
    ***************************************************************************
    */
    protected void setListener(Event listen)
    {
        listener = listen;
    }


   /**
    ***************************************************************************
    **
    **  Clears the event listener.
    **
    ***************************************************************************
    */
    protected void clearListener()
    {
        listener = null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the error counters.
    **
    **  @return  A two-element array containing the number of network timeouts
    **           and the number of sequence errors
    **
    ***************************************************************************
    */
    protected int[] getErrors()
    {
        return new int[]{ps.getNumTimeout(), ps.getNumSeqErr()};
    }


   /**
    ***************************************************************************
    **
    **  Gets the power supply ID.
    **
    **  @return  The ID
    **
    ***************************************************************************
    */
    protected String getPsId()
    {
        return psId;
    }


   /**
    ***************************************************************************
    **
    **  Generates the power supply ID.
    **
    ***************************************************************************
    */
    private void generatePsId() throws DriverException
    {
        String serial = String.format("%016x", ps.getSerialNo());
        psId = props.getProperty("org.lsst.ccs.power.psid." + serial);
        if (psId == null) {
            log.warn("Unknown PS serial number: " + serial);
            psId = serial.substring(5, 10);
        }
    }

}
