package org.lsst.ccs.subsystem.power;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.data.RebPsFullState;
import org.lsst.ccs.subsystem.power.data.RebPsState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements the REB PS control subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RebPower extends Module
                      implements HardwareController, RebPsDevice.Event {

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private final String configName;
    private final int monMillis;
    private final Logger sLog;
    private Monitor mon;
    private PowerDevice mainPs;
    private Map<String, RebPsDevice> psDevMap;
    private RebPsDevice psDevice;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    **  @param  name            The subsystem name.
    **
    **  @param  tickMillis      The tick period (ms).
    **
    **  @param  monMillis       The monitor period (ms).
    **
    **  @param  configName      The name of the configuration.
    **
    ***************************************************************************
    */
    public RebPower(String name, int tickMillis, int monMillis,
                    String configName)
    {
        super(name, tickMillis);
        this.monMillis = monMillis;
        this.configName = configName;
        sLog = Logger.getLogger("org.lsst.ccs.subsystem.power");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the power subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize devices
        */
        mon = new Monitor(this, null, sLog);
        mon.initConfiguration();

        /*
        **  Get the auxillary power supply device
        */
        Map<String, PowerDevice> mainPsMap = getChildren(PowerDevice.class);
        if (mainPsMap.isEmpty()) {
            sLog.warning("No main power supply device specified");
        }
        else {
            mainPs = (PowerDevice)mainPsMap.values().toArray()[0];
        }

        /*
        **  Get all the REB power supply devices
        */
        psDevMap = getChildren(RebPsDevice.class);
        if (psDevMap.isEmpty()) {
            sLog.error("No REB power devices specified");
        }
        else {
            psDevice = (RebPsDevice)psDevMap.values().toArray()[0];
            psDevice.setListener(this);
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware.
    **
    **  @return  GO
    **
    ***************************************************************************
    */
    @Override
    public TreeWalkerDiag checkHardware()
    {
        mon.initSensors();
        return TreeWalkerDiag.GO;
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is started.
    **
    ***************************************************************************
    */
    @Override
    public void checkStarted()
    {
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is stopped.
    **
    ***************************************************************************
    */
    @Override
    public void checkStopped()
    {
    }


   /**
    ***************************************************************************
    **
    **  Starts the subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void startTicking()
    {
        mon.start(monMillis);
        sLog.info("REB PS subsystem started");
        publishState();        // For any GUIs
        mon.publishState();    // For any GUIs
        mon.publishLimits();   // For GUIs and the trending database
        super.startTicking();
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic power data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        mon.publishData();
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        publishState();    // Must do this for GUI
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data.
    **
    **  @throws IOException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Save the current configuration")
    public void saveConfig() throws IOException
    {
        getEnvironment().saveAllChanges();
        mon.clearLimitChanges();
        mon.publishState();         // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Toggles the main power supply on or off.
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Control the main power supply")
    public void toggleMainPower() throws Exception
    {
        int state = getMainState();
        try {
            if (state == PowerChanState.PWR_STATE_OFF) {
                mainPs.powerOn();
                psDevice.checkOnline();
            }
            else if (state == PowerChanState.PWR_STATE_ON) {
                psDevice.sequencePower(false);
                mainPs.powerOff();
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


   /**
    ***************************************************************************
    **
    **  Toggles the on/off state of a power supply element.
    **
    **  @param  reb  The REB number
    **
    **  @param  ps   The power supply number
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Toggle a power supply element")
    public void togglePower(int reb, int ps) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.togglePower(reb, ps);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


   /**
    ***************************************************************************
    **
    **  Sequences a power supply on or off.
    **
    **  @param  reb  The REB number
    **
    **  @param  on   Turns on if true, off if false
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Sequence a power supply")
    public void sequencePower(int reb, boolean on) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.sequencePower(reb, on);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets a HV bias DAC value.
    **
    **  @param  reb    The REB number
    **
    **  @param  value  The value to set
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Set a HV bias DAC")
    public void setBiasDac(int reb, double value) throws PowerException
    {
        try {
            if (psDevice != null) {
                psDevice.setBiasDac(reb, value);
            }
        }
        finally {
            publishState();         // Must always do this
            mon.setPublishData();
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the full REB power supply state.
    **
    **  @return  The full power supply state
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public RebPsFullState getFullState() throws Exception
    {
        return new RebPsFullState(getState(), mon.getFullState());
    }


   /**
    ***************************************************************************
    **
    **  Gets the power supply error counters.
    **
    **  @return  The two-element array of error counters
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Get the error counters")
    public int[] getErrors()
    {
        if (psDevice != null) {
            return psDevice.getErrors();
        }
        else {
            return new int[]{0, 0};
        }
    }


   /**
    ***************************************************************************
    **
    **  Terminates the program.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Terminate the program")
    public void quit()
    {
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Publishes the state of the REB power supply.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    void publishState()
    {
        KeyValueData kvd = new KeyValueData(RebPsState.KEY, getState());
        getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }    


   /**
    ***************************************************************************
    **
    **  Gets the state of the REB power supply.
    **
    ***************************************************************************
    */
    RebPsState getState()
    {
        int[] psState;
        double[] biasDacs;
        String psId = "PS-X";
        if (psDevice != null) {
            try {
                psState = psDevice.getState();
                biasDacs = psDevice.getBiasDacs();
                psId = psDevice.getPsId();
            }
            catch (PowerException e) {
                psState = new int[0];
                biasDacs = new double[0];
            }
        }
        else {
            psState = new int[0];
            biasDacs = new double[0];
        }

        return new RebPsState(getTickMillis(), getMainState(), psState,
                              biasDacs, psId);
    }


   /**
    ***************************************************************************
    **
    **  Gets the state of the main power supply.
    **
    ***************************************************************************
    */
    int getMainState()
    {
        int state = PowerChanState.PWR_STATE_OFFLINE;
        if (mainPs != null && mainPs.isOnline()) {
            try {
                List<PowerChanState> pcState = mainPs.getPowerState();
                if (!pcState.isEmpty()) {
                    state = pcState.get(0).getState();
                }
            }
            catch (Exception e) {
            }
        }

        return state;
    }


    @Override
    public void opened()
    {
        publishState();
    }


    @Override
    public void closed()
    {
        publishState();
    }
}
