package org.lsst.ccs.subsystem.power;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.framework.Configurable;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystem.monitor.Control;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 ******************************************************************************
 **
 **  Interface to an ASCII commanded power supply device.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public abstract class PowerDevice extends Device {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    protected final static int
        NETWORK_CONN_INVALID = 0x01,
        CONN_TYPE_NOT_CONFIG = 0x02,
        DEVC_ID_NOT_CONFIG   = 0x04,
        DEVC_PARM_NOT_CONFIG = 0x08;

    private final static int
        MON_TYPE_VOLTAGE = 0,
        MON_TYPE_CURRENT = 1;

    private final static String
        CONN_TYPE = "connType",
        DEVC_ID   = "devcId",
        DEVC_PARM = "devcParm";

    private final static Map<String, Integer> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put(Power.ConnType.Network.name().toLowerCase(),
                     Ascii.CONN_TYPE_NETWORK);
        cTypeMap.put(Power.ConnType.Ftdi.name().toLowerCase(),
                     Ascii.CONN_TYPE_FTDI);
        cTypeMap.put(Power.ConnType.Serial.name().toLowerCase(),
                     Ascii.CONN_TYPE_SERIAL);
    }

    private final static Map<Integer, Power.ConnType> cTypeEMap = new HashMap();
    static {
        cTypeEMap.put(Ascii.CONN_TYPE_NETWORK, Power.ConnType.Network);
        cTypeEMap.put(Ascii.CONN_TYPE_FTDI, Power.ConnType.Ftdi);
        cTypeEMap.put(Ascii.CONN_TYPE_SERIAL, Power.ConnType.Serial);
    }

    private final static Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
    }

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    protected PowerSupplyDriver psd;
    protected String devcId;
    protected int devcParm;
    private String connTypeS;
    private int connType;
    private final int options, minChan, maxChan;


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  psd       The power supply device driver object
    **
    **  @param  connType  The connection type: network; ftdi; serial
    **
    **  @param  devcId    The device ID: host name; USB serial no.; serial
    **                    device name
    **
    **  @param  devcParm  The device parameter: port number; baud rate;
    **                    baud rate
    **
    **  @param  options   The options word
    **
    **  @param  minChan   The minimum channel number
    **
    **  @param  maxChan   The maximum channel number
    **
    ***************************************************************************
    */
    public PowerDevice(PowerSupplyDriver psd, String connType, String devcId,
                       int devcParm, int options, int minChan, int maxChan)
    {
        super();
        this.psd = psd;
        this.connTypeS = connType;
        this.devcId = devcId;
        this.devcParm = devcParm;
        this.options = options;
        this.minChan = minChan;
        this.maxChan = maxChan;
    }


   /**
    ***************************************************************************
    **
    **  Gets the minimum channel number.
    **
    **  @return  The minimum channel number
    **
    ***************************************************************************
    */
    public int getMinChannel()
    {
        return minChan;
    }


   /**
    ***************************************************************************
    **
    **  Gets the maximum channel number.
    **
    **  @return  The maximum channel number
    **
    ***************************************************************************
    */
    public int getMaxChannel()
    {
        return maxChan;
    }


   /**
    ***************************************************************************
    **
    **  Performs configuration.
    **
    **  @param  mon  The monitor object
    **
    ***************************************************************************
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        Integer cType = cTypeMap.get(connTypeS.toLowerCase());
        if (cType == null || (cType == Ascii.CONN_TYPE_NETWORK
                              && (options & NETWORK_CONN_INVALID) != 0)) {
            mon.reportConfigError(getName(), CONN_TYPE, "is invalid");
        }
        connType = cType;

        Configurable.Environment env = getEnvironment();
        String param = null;
        try {
            if ((options & CONN_TYPE_NOT_CONFIG) == 0) {
                param = CONN_TYPE;
                env.getCheckedValueFromConfiguration(param, connTypeS);
            }
            if ((options & DEVC_ID_NOT_CONFIG) == 0) {
                param = DEVC_ID;
                env.getCheckedValueFromConfiguration(param, devcId);
            }
            if ((options & DEVC_PARM_NOT_CONFIG) == 0) {
                param = DEVC_PARM;
                env.getCheckedValueFromConfiguration(param, devcParm);
            }
        }
        catch (Exception e) {
            mon.reportConfigError(getName(), param, "is missing");
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the connection type.
    **
    **  @param  connTypeS  The connection type string
    **
    **  @throws  PowerException
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setConnType(String connTypeS) throws PowerException
    {
        if (connTypeS.equalsIgnoreCase(this.connTypeS)) return;
        Integer cType = cTypeMap.get(connTypeS);
        if (cType == null) {
            throw new PowerException("Invalid connection type");
        }
        this.connTypeS = connTypeS;
        connType = cType;
        close();
        inited = false;
        initialize();
    }


   /**
    ***************************************************************************
    **
    **  Gets the connection type.
    **
    **  @return  The connection type string
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Gets the connection type")
    public String getConnType()
    {
        return connTypeS;
    }


   /**
    ***************************************************************************
    **
    **  Sets the device ID.
    **
    **  @param  devcId  The device identification string
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setDevcId(String devcId)
    {
        if (devcId.equals(this.devcId)) return;
        this.devcId = devcId;
        close();
        inited = false;
        initialize();
    }


   /**
    ***************************************************************************
    **
    **  Gets the device ID.
    **
    **  @return  The device identification string
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Gets the device ID")
    public String getDevcId()
    {
        return devcId;
    }


   /**
    ***************************************************************************
    **
    **  Sets the device parameter.
    **
    **  @param  devcParm  The device parameter
    **
    ***************************************************************************
    */
    @ConfigChanger
    public void setDevcParm(int devcParm)
    {
        if (devcParm == this.devcParm) return;
        this.devcParm = devcParm;
        close();
        inited = false;
        initialize();
    }


   /**
    ***************************************************************************
    **
    **  Gets the device parameter.
    **
    **  @return  The device parameter
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Gets the device parameter")
    public int getDevcParm()
    {
        return devcParm;
    }


   /**
    ***************************************************************************
    **
    **  Performs full initialization.
    **
    ***************************************************************************
    */
    @Override
    protected void initialize()
    {
        try {
            psd.open(connType, devcId, devcParm);
            setOnline(true);
            String message = "Connected to " + fullName;
            if (!inited) {
                log.info(message);
            }
            else {
                log.error(message);
            }
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    protected void close()
    {
        try {
            psd.close();
        }
        catch (DriverException e) {
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a monitoring channel's parameters for validity.
    **
    **  @param  name     The channel name
    **
    **  @param  hwChan   The hardware channel
    **
    **  @param  type     The channel type string
    **
    **  @param  subtype  The channel subtype string
    **
    **  @return  Two-element array containg the encoded type [0] and
    **           subtype [1] 
    **
    **  @throws  Exception  If parameters are invalid
    **
    ***************************************************************************
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            mon.reportError(name, "type", type);
        }
        if (hwChan < minChan || hwChan > maxChan) {
            mon.reportError(getName(), "hw channel number", hwChan);
        }
        
        return new int[]{mType, 0};
    }


   /**
    ***************************************************************************
    **
    **  Reads a monitoring channel.
    **
    **  @param  hwChan  The hardware channel number
    **
    **  @param  type    The encoded channel type
    **
    **  @return  The read value
    **
    ***************************************************************************
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        if (online) {
            try {
                value = (type == MON_TYPE_VOLTAGE) ? psd.readVoltage(hwChan)
                                                   : psd.readCurrent(hwChan);
            }
            catch (DriverException e) {
                log.error("Error reading from " + fullName + ": " + e);
                setOnline(false);
            }
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Gets the configuration data.
    **
    **  @return  A power configuration object
    **
    ***************************************************************************
    */
    public Power getPowerConfig()
    {
        Power power = new Power();
        power.setName(getName());
        power.setConnType(cTypeEMap.get(connType));
        power.setDevcId(devcId);
        power.setDevcParm(devcParm);
        Power.Channel[] chans = new Power.Channel[ctlChans.size()];
        int j = 0;
        for (Control ctl : ctlChans.values()) {
            chans[j++] = ((PowerControl)ctl).getConfig();
        }
        power.setChannels(chans);

        return power;
    }


   /**
    ***************************************************************************
    **
    **  Sets the configuration data.
    **
    **  @param  power  A power configuration object
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    public void setPowerConfig(Power power) throws Exception
    {
        if ((options & CONN_TYPE_NOT_CONFIG) == 0) {
            change(CONN_TYPE, power.getConnType().name());
        }
        if ((options & DEVC_ID_NOT_CONFIG) == 0) {
            change(DEVC_ID, power.getDevcId());
        }
        if ((options & DEVC_PARM_NOT_CONFIG) == 0) {
            change(DEVC_PARM, power.getDevcParm());
        }
        for (Power.Channel chan : power.getChannels()) {
            String name = getName() + "." + chan.getName();
            ((PowerControl)ctlChans.get(name)).setConfig(chan);
        }
    }


   /**
    ***************************************************************************
    **
    **  Adds to the map of configured values.
    **
    **  @param  values  The map of values to be added to
    **
    ***************************************************************************
    */
    @Override
    protected void addConfigValues(Map<String, Object> values)
    {
        String stem = getName() + ".";
        values.put(stem + CONN_TYPE, connTypeS);
        values.put(stem + DEVC_ID, devcId);
        values.put(stem + DEVC_PARM, devcParm);
    }


   /**
    ***************************************************************************
    **
    **  Gets the list of power state data.
    **
    **  @return  A list of power state data
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected List<PowerChanState> getPowerState() throws Exception
    {
        try {
            readVoltageGroup();
            readCurrentGroup();
            readOutputGroup();
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
        List<PowerChanState> pState = new ArrayList<>();
        for (Control ctl : ctlChans.values()) {
            pState.add(((PowerControl)ctl).getState());
        }
   
        return pState;
    }


   /**
    ***************************************************************************
    **
    **  Powers on all channels.
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected void powerOn() throws Exception
    {
        for (Control ctl : ctlChans.values()) {
            ((PowerControl)ctl).writeAll();
        }
        try {
            writeVoltageGroup();
            writeCurrentGroup();
            writeOnDelayGroup();
            writeOffDelayGroup();
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
        setOutput(true);
    }


   /**
    ***************************************************************************
    **
    **  Powers off all channels.
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected void powerOff() throws Exception
    {
        setOutput(false);
    }


   /**
    ***************************************************************************
    **
    **  Writes the voltage to the hardware.
    **
    **  @param  voltage  The voltage to set
    **
    **  @param  hwChan   The hardware channel number
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected void writeVoltage(double voltage, int hwChan) throws Exception
    {
        try {
            psd.setVoltage(voltage, hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the voltage from the hardware.
    **
    **  @param  hwChan   The hardware channel number
    **
    **  @return  The read voltage
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected double readVoltage(int hwChan) throws Exception
    {
        try {
            return psd.readVoltage(hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
            return 0;
        }
    }


   /**
    ***************************************************************************
    **
    **  Writes the current to the hardware.
    **
    **  @param  current  The current to set
    **
    **  @param  hwChan   The hardware channel number
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected void writeCurrent(double current, int hwChan) throws Exception
    {
        try {
            psd.setCurrent(current, hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the current from the hardware.
    **
    **  @param  hwChan   The hardware channel number
    **
    **  @return  The read current
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected double readCurrent(int hwChan) throws Exception
    {
        try {
            return psd.readCurrent(hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
            return 0;
        }
    }


   /**
    ***************************************************************************
    **
    **  Writes the power-on delay to the hardware.
    **
    **  @param  onDelay  The delay to set
    **
    **  @param  hwChan   The hardware channel number
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected void writeOnDelay(double onDelay, int hwChan) throws Exception
    {
        try {
            psd.setOnDelay(onDelay, hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    ***************************************************************************
    **
    **  Writes the power-off delay to the hardware.
    **
    **  @param  offDelay  The delay to set
    **
    **  @param  hwChan    The hardware channel number
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected void writeOffDelay(double offDelay, int hwChan) throws Exception
    {
        try {
            psd.setOffDelay(offDelay, hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    ***************************************************************************
    **
    **  Writes the power output state to the hardware.
    **
    **  @param  value   The output state to set
    **
    **  @param  hwChan  The hardware channel number
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected void writeOutput(boolean value, int hwChan) throws Exception
    {
        try {
            psd.setOutput(value, hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the power output state from the hardware.
    **
    **  @param  hwChan   The hardware channel number
    **
    **  @return  The power output state
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    protected boolean readOutput(int hwChan) throws Exception
    {
        try {
            return psd.getOutput(hwChan);
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
            return false;
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the power output state for all channels.
    **
    **  @param  value  The output state to set
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    private void setOutput(boolean value) throws Exception
    {
        try {
            for (Control ctl : ctlChans.values()) {
                psd.setOutput(value, ctl.getHwChan());
            }
            writeOutputGroup();
        }
        catch (DriverException e) {
            checkTimeout(e, PowerException.class);
        }
    }


   /**
    ***************************************************************************
    **
    **  Writes a group of voltages to the hardware.
    **
    **  May be overridden for devices with this capability
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    protected void writeVoltageGroup() throws DriverException
    {
    }


   /**
    ***************************************************************************
    **
    **  Reads a group of voltages from the hardware.
    **
    **  May be overridden for devices with this capability
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    protected void readVoltageGroup() throws DriverException
    {
    }


   /**
    ***************************************************************************
    **
    **  Writes a group of currents to the hardware.
    **
    **  May be overridden for devices with this capability
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    protected void writeCurrentGroup() throws DriverException
    {
    }


   /**
    ***************************************************************************
    **
    **  Reads a group of currents from the hardware.
    **
    **  May be overridden for devices with this capability
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    protected void readCurrentGroup() throws DriverException
    {
    }


   /**
    ***************************************************************************
    **
    **  Writes a group of power-on delays to the hardware.
    **
    **  May be overridden for devices with this capability
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    protected void writeOnDelayGroup() throws DriverException
    {
    }


   /**
    ***************************************************************************
    **
    **  Writes a group of power-off delays to the hardware.
    **
    **  May be overridden for devices with this capability
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    protected void writeOffDelayGroup() throws DriverException
    {
    }


   /**
    ***************************************************************************
    **
    **  Writes a group of power output states to the hardware.
    **
    **  May be overridden for devices with this capability
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    protected void writeOutputGroup() throws DriverException
    {
    }


   /**
    ***************************************************************************
    **
    **  Reads a group of power output states from the hardware.
    **
    **  May be overridden for devices with this capability
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    protected void readOutputGroup() throws DriverException
    {
    }

}
