package org.lsst.ccs.subsystem.power;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerFullState;
import org.lsst.ccs.subsystem.power.data.PowerState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements the VST power control subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class MainPower extends Module implements HardwareController {

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private final String configName;
    private Map<String, PowerDevice> pwrDevices;
    private final int monMillis;
    private final Logger sLog;
    private Monitor mon;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    **  @param  name            The subsystem name.
    **
    **  @param  tickMillis      The tick period (ms).
    **
    **  @param  monMillis       The monitor period (ms).
    **
    **  @param  configName      The name of the configuration.
    **
    ***************************************************************************
    */
    public MainPower(String name, int tickMillis, int monMillis,
                     String configName)
    {
        super(name, tickMillis);
        this.monMillis = monMillis;
        this.configName = configName;
        sLog = Logger.getLogger("org.lsst.ccs.subsystem.power");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the power subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize devices
        */
        mon = new Monitor(this, null, sLog);
        mon.initConfiguration();

        /*
        **  Get all the power devices
        */
        pwrDevices = getChildren(PowerDevice.class);
        if (pwrDevices.isEmpty()) {
            sLog.error("No power devices specified");
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the hardware.
    **
    **  @return  GO
    **
    ***************************************************************************
    */
    @Override
    public TreeWalkerDiag checkHardware()
    {
        mon.initSensors();

        return TreeWalkerDiag.GO;
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is started.
    **
    ***************************************************************************
    */
    @Override
    public void checkStarted()
    {
    }


   /**
    ***************************************************************************
    **
    **  Checks whether hardware is stopped.
    **
    ***************************************************************************
    */
    @Override
    public void checkStopped()
    {
    }


   /**
    ***************************************************************************
    **
    **  Starts the subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void startTicking()
    {
        mon.start(monMillis);
        sLog.info("Power subsystem started");
        publishState();        // For any GUIs
        mon.publishState();    // For any GUIs
        mon.publishLimits();   // For GUIs and the trending database
        super.startTicking();
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic power data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        if (pwrDevices.isEmpty()) return;
        ArrayList<PowerChanState> pState = new ArrayList<>();
        for (PowerDevice pDevc : pwrDevices.values()) {
            try {
                pState.addAll(pDevc.getPowerState());
            }
            catch (Exception e) {
            }
        }
        KeyValueData kvd = new KeyValueData(PowerChanState.KEY, pState);
        getSubsystem().publishSubsystemDataOnStatusBus(kvd);
        mon.publishData();
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        publishState();    // Must do this for GUI
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data.
    **
    **  @throws IOException
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Save the current configuration")
    public void saveConfig() throws IOException
    {
        getEnvironment().saveAllChanges();
        mon.clearLimitChanges();
        mon.publishState();         // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the configuration data.
    **
    **  @return  The array of power configuration objects
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Get the configuration data")
    public Power[] getPowerConfig()
    {
        Power[] power = new Power[pwrDevices.size()];
        int j = 0;
        for (PowerDevice pDevc : pwrDevices.values()) {
            power[j++] = pDevc.getPowerConfig();
        }

        return power;
    }


   /**
    ***************************************************************************
    **
    **  Sets the configuration data.
    **
    **  @param  power  The array of power configuration objects
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Set the configuration data")
    public void setPowerConfig(Power[] power) throws Exception
    {
        for (Power pConf : power) {
            PowerDevice pDevc = pwrDevices.get(pConf.getName());
            pDevc.setPowerConfig(pConf);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the power state.
    **
    **  @return  The power state
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Get the power state")
    public List<PowerChanState> getPowerState() throws Exception
    {
        List<PowerChanState> pState = new ArrayList<>();
        for (PowerDevice pDevc : pwrDevices.values()) {
            pState.addAll(pDevc.getPowerState());
        }

        return pState;
    }


   /**
    ***************************************************************************
    **
    **  Turns on the power.
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Turn on the power")
    public void powerOn() throws Exception
    {
        for (PowerDevice pDevc : pwrDevices.values()) {
            pDevc.powerOn();
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns off the power.
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Turn off the power")
    public void powerOff() throws Exception
    {
        for (PowerDevice pDevc : pwrDevices.values()) {
            pDevc.powerOff();
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the full power state.
    **
    **  @return  The full power state
    **
    **  @throws  Exception
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public PowerFullState getFullState() throws Exception
    {
        return new PowerFullState(new PowerState(getTickMillis()),
                                  mon.getFullState());
    }


   /**
    ***************************************************************************
    **
    **  Terminates the program.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Terminate the program")
    public void quit()
    {
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Publishes the state of the power module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    void publishState()
    {
        KeyValueData kvd = new KeyValueData(PowerState.KEY,
                                            new PowerState(getTickMillis()));
        getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }    

}
