package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.power.data.PowerChanState;

/**
 *  Implements the power status sub-panel.
 *
 *  @author Owen Saxton
 */
public class PowerStatusPanel extends JPanel {
    
    private static final Map<Integer, String> pStateMap = new HashMap<>();
    static {
        pStateMap.put(PowerChanState.PWR_STATE_OFFLINE, "N/A");
        pStateMap.put(PowerChanState.PWR_STATE_OFF, "Off");
        pStateMap.put(PowerChanState.PWR_STATE_ON, "On");
    }
    private static final Map<Integer, Color> pColorMap = new HashMap<>();
    static {
        pColorMap.put(PowerChanState.PWR_STATE_OFFLINE, UiConstants.RED);
        pColorMap.put(PowerChanState.PWR_STATE_OFF, Color.BLACK);
        pColorMap.put(PowerChanState.PWR_STATE_ON, UiConstants.GREEN);
    }

    private final JPanel pwrStPanel = new JPanel();
    private final JLabel lbPsName = new JLabel("Power Status: ");
    private final JLabel lbPsValue = new JLabel();
    private final JLabel lbPwrHdrState = new JLabel("State");
    private final JLabel lbPwrHdrVoltage = new JLabel("Voltage");
    private final JLabel lbPwrHdrCurrent = new JLabel("Current");
    private JLabel[] lbPwrName = new JLabel[0];
    private JLabel[] lbPwrState;
    private JLabel[] lbPwrVoltage;
    private JLabel[] lbPwrCurrent;

    private final Parent parent;
    private static final long serialVersionUID = 1L;

    /**
     *  Parent interface for providing services.
     */
    public interface Parent {

        void powerResized();

    }

    /**
     *  Constructor.
     *
     *  @param  parent  Parent object for providing services
     */
    public PowerStatusPanel(Parent parent) {
        this.parent = parent;
        initPowerPanel();
    }

    /**
     *  Updates the panel.
     *
     *  @param  power  The power data
     */
    public void updatePanel(List<PowerChanState> power) {
        lbPsValue.setText("RUNNING");
        lbPsValue.setForeground(UiConstants.GREEN);
        updatePower(power);
    }

    /**
     *  Disables the panel.
     */
    public void disablePanel() {
        lbPsValue.setText("STOPPED");
        lbPsValue.setForeground(UiConstants.RED);
    }

    /**
     *  Initializes the power sub-panel.
     */
    private void initPowerPanel() {
        setBorder(BorderFactory.createTitledBorder(""));
        setLayout(new GridBagLayout());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;

        gbc.gridwidth = 4;
        pwrStPanel.add(lbPsName);
        lbPsValue.setText("STOPPED");
        lbPsValue.setForeground(UiConstants.RED);
        pwrStPanel.add(lbPsValue);
        add(pwrStPanel, gbc);

        gbc.insets = new Insets(5, 10, 5, 10);
        gbc.anchor = GridBagConstraints.EAST;
        gbc.gridwidth = 1;
        gbc.gridx = 1;
        gbc.gridy++;
        add(lbPwrHdrState, gbc);
        gbc.gridx++;
        add(lbPwrHdrVoltage, gbc);
        gbc.gridx++;
        add(lbPwrHdrCurrent, gbc);
    }

    /**
     *  Updates the power state.
     */
    private void updatePower(List<PowerChanState> power) {
        if (power.size() == lbPwrName.length) {
            updatePwrPanel(power);
        }
        else {
            resizePwrPanel(power);
            parent.powerResized();
        }
    }

    /**
     *  Updates the power sub-panel in place.
     */
    private void updatePwrPanel(List<PowerChanState> power) {
        for (int j = 0; j < power.size(); j++) {
            PowerChanState pwr = power.get(j);
            lbPwrName[j].setText(pwr.getName() + ":");
            lbPwrState[j].setText(pStateMap.get(pwr.getState()));
            lbPwrState[j].setForeground(pColorMap.get(pwr.getState()));
            lbPwrVoltage[j].setText(String.format("%.3f", pwr.getVoltage()));
            lbPwrCurrent[j].setText(String.format("%.3f", pwr.getCurrent()));
        }
    }

    /**
     *  Resizes the power sub-panel.
     */
    private void resizePwrPanel(List<PowerChanState> power) {
        int nChan = power.size();
        for (int j = 0; j < lbPwrName.length; j++) {
            remove(lbPwrName[j]);
            remove(lbPwrState[j]);
            remove(lbPwrVoltage[j]);
            remove(lbPwrCurrent[j]);
        }

        lbPwrName = new JLabel[nChan];
        lbPwrState = new JLabel[nChan];
        lbPwrVoltage = new JLabel[nChan];
        lbPwrCurrent = new JLabel[nChan];

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(0, 10, 5, 10);
        gbc.anchor = GridBagConstraints.EAST;
        gbc.gridwidth = 1;
        gbc.gridx = 1;
        gbc.gridy = 2;

        for (int j = 0; j < nChan; j++) {
            PowerChanState pwr = power.get(j);
            gbc.gridy++;
            gbc.gridx = 0;
            lbPwrName[j] = new JLabel(pwr.getName() + ":");
            add(lbPwrName[j], gbc);
            gbc.gridx++;
            lbPwrState[j] = new JLabel(pStateMap.get(pwr.getState()));
            lbPwrState[j].setForeground(pColorMap.get(pwr.getState()));
            add(lbPwrState[j], gbc);
            gbc.gridx++;
            lbPwrVoltage[j]
              = new JLabel(String.format("%.3f", pwr.getVoltage()));
            add(lbPwrVoltage[j], gbc);
            gbc.gridx++;
            lbPwrCurrent[j]
              = new JLabel(String.format("%.3f", pwr.getCurrent()));
            add(lbPwrCurrent[j], gbc);
        }
    }

}
