package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import javax.swing.border.TitledBorder;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.power.constants.ComCamQuadBoxSwitches;
import org.lsst.ccs.subsystem.power.data.ComCamQuadBoxState;
import org.lsst.ccs.subsystem.power.constants.SwitchState;

/**
 *  Implements the REB power supply control & monitoring panel.
 *
 *  @author Owen Saxton
 */
public class ComCamQuadBoxControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final String CMND_GET_STATE = "getFullState";
    private static final String[] panelNames = new String[ComCamQuadBoxSwitches.NUM_DEVICES];
    static {
        panelNames[ComCamQuadBoxSwitches.DEVC_BFR] = "BFR";
        panelNames[ComCamQuadBoxSwitches.DEVC_PDU_5V] = "5V Clean PDU";
        panelNames[ComCamQuadBoxSwitches.DEVC_PDU_24VC] = "24V Clean PDU";
        panelNames[ComCamQuadBoxSwitches.DEVC_PDU_24VD] = "24V Dirty PDU";
        panelNames[ComCamQuadBoxSwitches.DEVC_PDU_48V] = "48V Dirty PDU";
	//        panelNames[ComCamQuadBoxSwitches.DEVC_REB_BULK] = "REB Bulk PS";
    }
    private static final Map<Integer, String> switchNames = new LinkedHashMap<>();
    static {
        switchNames.put(ComCamQuadBoxSwitches.SW_PDU_5V, "5V Clean PDU");
        switchNames.put(ComCamQuadBoxSwitches.SW_PDU_24VC, "24V Clean PDU");
        switchNames.put(ComCamQuadBoxSwitches.SW_PDU_24VD, "24V Dirty PDU");
        switchNames.put(ComCamQuadBoxSwitches.SW_PDU_48V, "48V Dirty PDU");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_REB_BULK_PS_0, "REB PS 0");

        switchNames.put(ComCamQuadBoxSwitches.SW_RELAY_AUX_CMC_DPP_01, "VAT valve");
        switchNames.put(ComCamQuadBoxSwitches.SW_RELAY_DPP_01,        "Scroll Pump");
        switchNames.put(ComCamQuadBoxSwitches.SW_RELAY_VCR_CMC_PCT_00, "Cryo Turbo Pump");
        switchNames.put(ComCamQuadBoxSwitches.SW_RELAY_SHT_SHT_PSW_00, "Shutter");
	//	switchNames.put(ComCamQuadBoxSwitches.SW_RELAY_PTH_CMC_SCR_00, "Scroll 1200W");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_RELAY_PTH_CMC_SCR_01, "Scroll 400W");

	//        switchNames.put(ComCamQuadBoxSwitches.SW_TRIM_HEATERS, "Trim Heaters");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_HEATER_1, "Heater 1");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_HEATER_2, "Heater 2");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_REB_BULK_PS_4, "REB PS spare");

        switchNames.put(ComCamQuadBoxSwitches.SW_OTM_3_A, "OTM-A");
        switchNames.put(ComCamQuadBoxSwitches.SW_OTM_3_B, "OTM-B");

	// ----------------------------------------------------------------

        switchNames.put(ComCamQuadBoxSwitches.SW_MAIN_24C, "Main");
        switchNames.put(ComCamQuadBoxSwitches.SW_ION_PUMP, "Ion Pump");
//        switchNames.put(ComCamQuadBoxSwitches.SW_HCU_PWR_CRY, "Power/Cryo HCUs");
        switchNames.put(ComCamQuadBoxSwitches.SW_HCU_FES_SHU, "FES/Shutter HCUs");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_UNDEF_24VC, "undefined24VC");
        switchNames.put(ComCamQuadBoxSwitches.SW_BODY_PURGE, "CryoRTDs/HEX MAQ20");
//        switchNames.put(ComCamQuadBoxSwitches.SW_BPU_MAQ20, "BPU MAQ20");
        switchNames.put(ComCamQuadBoxSwitches.SW_GAUGES, "Gauges");
        //switchNames.put(ComCamQuadBoxSwitches.SW_UNUSED_24C, "Unused");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_SHUTTER_1, "Shutter");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_SHUTTER_2, "Shutter 2");

	// ----------------------------------------------------------------

        switchNames.put(ComCamQuadBoxSwitches.SW_MAIN_24D, "Main");
//        switchNames.put(ComCamQuadBoxSwitches.SW_CRYO_TURBO, "Cryo turbo");
        switchNames.put(ComCamQuadBoxSwitches.SW_FES, "Filter Changer");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_HEX_TURBO, "Hex turbo");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_FES_CHANGER_D, "FES changer");
	//        switchNames.put(ComCamQuadBoxSwitches.SW_SHUTTER_BRAKE, "Shutter brake");
	switchNames.put(ComCamQuadBoxSwitches.SW_BOOSTER_PUMP, "Booster Pump");
	// ----------------------------------------------------------------

        switchNames.put(ComCamQuadBoxSwitches.SW_MAIN_48, "Main");
        switchNames.put(ComCamQuadBoxSwitches.SW_CRYOTEL_CTRL0, "CryoTel Ctrl 0");
        switchNames.put(ComCamQuadBoxSwitches.SW_CRYOTEL_CTRL1, "CryoTel Ctrl 1"); 
        switchNames.put(ComCamQuadBoxSwitches.SW_CRYOTEL_CTRL2, "CryoTel Ctrl 2");
//        switchNames.put(ComCamQuadBoxSwitches.SW_UTC , "UTC");
        switchNames.put(ComCamQuadBoxSwitches.SW_REB_BULK_0, "REB PS 0");
    }
    private static final int[] nameWidths = new int[ComCamQuadBoxSwitches.NUM_DEVICES];
    static {
        nameWidths[ComCamQuadBoxSwitches.DEVC_BFR] = 110;
        nameWidths[ComCamQuadBoxSwitches.DEVC_PDU_5V] = 70;
        nameWidths[ComCamQuadBoxSwitches.DEVC_PDU_24VC] = 130;
        nameWidths[ComCamQuadBoxSwitches.DEVC_PDU_24VD] = 105;
        nameWidths[ComCamQuadBoxSwitches.DEVC_PDU_48V] = 105;
    }
    private static final Map<Integer, Integer> bfrSwitchMap = new HashMap<>();
    static {
        bfrSwitchMap.put(ComCamQuadBoxSwitches.DEVC_PDU_5V, ComCamQuadBoxSwitches.SW_PDU_5V);
        bfrSwitchMap.put(ComCamQuadBoxSwitches.DEVC_PDU_24VC, ComCamQuadBoxSwitches.SW_PDU_24VC);
        bfrSwitchMap.put(ComCamQuadBoxSwitches.DEVC_PDU_24VD, ComCamQuadBoxSwitches.SW_PDU_24VD);
        bfrSwitchMap.put(ComCamQuadBoxSwitches.DEVC_PDU_48V, ComCamQuadBoxSwitches.SW_PDU_48V);
    }
    private static final Map<Integer, Integer> mainSwitchMap = new HashMap<>();
    static {
        mainSwitchMap.put(ComCamQuadBoxSwitches.DEVC_PDU_24VC, ComCamQuadBoxSwitches.SW_MAIN_24C);
        mainSwitchMap.put(ComCamQuadBoxSwitches.DEVC_PDU_24VD, ComCamQuadBoxSwitches.SW_MAIN_24D);
        mainSwitchMap.put(ComCamQuadBoxSwitches.DEVC_PDU_48V, ComCamQuadBoxSwitches.SW_MAIN_48);
    }

    private static final int statusLabelWidth = UiUtilities.maxEnumLabelWidth(SwitchState.class);

    private final CommandSender sender;
    private final UiUtilities uiUtils;
    private SystemStatusPanel statusPanel;
    private final JPanel topPanel = new JPanel();
    private final JPanel[] switchPanel = new JPanel[ComCamQuadBoxSwitches.NUM_DEVICES];
    private final JLabel[][] switchLabel = new JLabel[ComCamQuadBoxSwitches.NUM_DEVICES][];
    private final JLabel[][] switchStatus = new JLabel[ComCamQuadBoxSwitches.NUM_DEVICES][];
    private final ButtonGroup[][] switchBG = new ButtonGroup[ComCamQuadBoxSwitches.NUM_DEVICES][];
    private final JRadioButton[][] switchOffRB = new JRadioButton[ComCamQuadBoxSwitches.NUM_DEVICES][];
    private final JRadioButton[][] switchOnRB = new JRadioButton[ComCamQuadBoxSwitches.NUM_DEVICES][];
    private final Map<Integer, Integer> switchLines = new HashMap<>();

    public ComCamQuadBoxControlPanel(String agent) {
	sender = new CommandSender(agent, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    private void initComponents()
    {
        //** Create all low-level components

        // State indicator
        statusPanel = new SystemStatusPanel(sender);

        // Switch panel elements                                                                                                                                        
        for (int dvc = 0; dvc < ComCamQuadBoxSwitches.NUM_DEVICES; dvc++) {
            int nSwitch = ComCamQuadBoxSwitches.NUM_SWITCHES[dvc];
            switchLabel[dvc] = new JLabel[nSwitch + 1];
            switchStatus[dvc] = new JLabel[nSwitch + 1];
            switchBG[dvc] = new ButtonGroup[nSwitch + 1];
            switchOnRB[dvc] = new JRadioButton[nSwitch + 1];
            switchOffRB[dvc] = new JRadioButton[nSwitch + 1];
        }
        int[] lines = new int[ComCamQuadBoxSwitches.NUM_DEVICES];
	for (int swId : switchNames.keySet()) {
            int dvc = swId >> 8;
            int line = lines[dvc]++;
            switchLines.put(swId, line);
            switchLabel[dvc][line] = UiUtilities.newLabel(switchNames.get(swId) + ": ", 0);
            switchStatus[dvc][line] = UiUtilities.newLabel("X", statusLabelWidth);
            JRadioButton on = switchOnRB[dvc][line] = uiUtils.newRadioButton("On", "N" + Integer.toString(swId));
            JRadioButton off = switchOffRB[dvc][line] = uiUtils.newRadioButton("Off", "F" + Integer.toString(swId));
            ButtonGroup bg = switchBG[dvc][line] = new ButtonGroup();
            bg.add(on);
            bg.add(off);
        }


        //** Construct panels

        // Top panel
        GridBagConstraints gbt = new GridBagConstraints();
        topPanel.setLayout(new GridBagLayout());
        gbt.gridx = 0;
        gbt.gridy = 0;
        topPanel.add(statusPanel, gbt);
        
        // Switch panels
        GridBagConstraints gbs = new GridBagConstraints();
        for (int dvc = 0; dvc < ComCamQuadBoxSwitches.NUM_DEVICES; dvc++) {
            JPanel panel = switchPanel[dvc] = new JPanel();
            TitledBorder border = new TitledBorder(new LineBorder(Color.BLACK), panelNames[dvc]);
            border.setTitleColor(UiConstants.BLUE);
            panel.setBorder(border);
            panel.setLayout(new GridBagLayout());
            gbs.gridy = 0;
            gbs.insets.top = 0;
            for (int line = 0; line < switchLabel[dvc].length; line++) {
                if (switchLabel[dvc][line] == null) continue;
                gbs.gridx = 0;
                gbs.insets.left = 4;
                gbs.insets.right = 0;
                panel.add(switchLabel[dvc][line], gbs);
                gbs.gridx++;
                gbs.insets.left = 0;
                panel.add(switchStatus[dvc][line], gbs);
                gbs.gridx++;
                gbs.insets.right = 4;
                panel.add(switchOffRB[dvc][line], gbs);
                gbs.gridx++;
                panel.add(switchOnRB[dvc][line], gbs);
                gbs.gridy++;
                gbs.insets.top = -4;
            }
            panel.setMinimumSize(panel.getPreferredSize());
        }

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 3;
        add(topPanel, gbc);
        gbc.gridy++;
        gbc.gridwidth = 1;
        add(switchPanel[ComCamQuadBoxSwitches.DEVC_BFR], gbc);
        gbc.gridx++;
        add(switchPanel[ComCamQuadBoxSwitches.DEVC_PDU_48V], gbc);
        gbc.gridx++;
        add(switchPanel[ComCamQuadBoxSwitches.DEVC_PDU_24VC], gbc);
        gbc.gridy++;
        gbc.gridx = 0;
	//        add(switchPanel[ComCamQuadBoxSwitches.DEVC_REB_BULK], gbc);
	//        gbc.gridx++;
        add(switchPanel[ComCamQuadBoxSwitches.DEVC_PDU_24VD], gbc);
        gbc.gridx++;
        add(switchPanel[ComCamQuadBoxSwitches.DEVC_PDU_5V], gbc);
    }

    public void updateControlPanel(ComCamQuadBoxState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void initPanel() {
        sender.sendCommand(true, null, CMND_GET_STATE);
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((ComCamQuadBoxState)reply);
    }

    @Override
    public void onCommandReject(String path, String command, Object[] args) {
        if (!command.equals(CMND_GET_STATE)) {
            sender.sendCommand(true, null, CMND_GET_STATE);
        }
    }

    public void updatePanel(ComCamQuadBoxState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    @Override
    public void handleRadioButton(String name) {
        boolean on = name.charAt(0) == 'N';
        int swId = Integer.valueOf(name.substring(1));
        sender.sendCommand(null, "setNamedSwitchOn", ComCamQuadBoxSwitches.switchIdToName.get(swId), on);
    }


    class UpdatePowerStatus implements Runnable {

        private final ComCamQuadBoxState qs;

        UpdatePowerStatus(ComCamQuadBoxState qs) {
            this.qs = qs;
        }

        @Override
        public void run() {
            statusPanel.updatePanel(qs.getTickMillis());

            for (int swId : switchNames.keySet()) {
                int devc = swId >> 8;
                int line = switchLines.get(swId);
                SwitchState st = qs.getSwitchState(swId);
                st = st == null ? SwitchState.OFFLINE : st;
                JLabel label = switchStatus[devc][line];
                label.setText(st.toString());
                label.setForeground(st == SwitchState.ON ? UiConstants.GREEN :
                                    st == SwitchState.OFF ? Color.BLACK : UiConstants.BLUE);
                if (st != SwitchState.OFFLINE) {
                    JRadioButton selButton = st == SwitchState.ON ? switchOnRB[devc][line] : switchOffRB[devc][line];
                    selButton.setSelected(true);
                }
                switchStatus[devc][line].setEnabled(true);
                switchOffRB[devc][line].setEnabled(true);
                switchOnRB[devc][line].setEnabled(true);
            }
            for (int devc : bfrSwitchMap.keySet()) {
                SwitchState bfrSwSt = qs.getSwitchState(bfrSwitchMap.get(devc));
                Integer mainSwId = mainSwitchMap.get(devc);
                SwitchState mainSwSt = mainSwId == null ? SwitchState.ON : qs.getSwitchState(mainSwId);
                String suffix = bfrSwSt != SwitchState.ON ? " (BFR off)" : mainSwSt != SwitchState.ON ? " (Main off)" : "";
                Color color = bfrSwSt != SwitchState.ON ? UiConstants.RED :
                              mainSwSt != SwitchState.ON ? UiConstants.PURPLE : UiConstants.BLUE;
                JPanel panel = switchPanel[devc];
                TitledBorder border = (TitledBorder)panel.getBorder();
                border.setTitle(panelNames[devc] + suffix);
                border.setTitleColor(color);
            }
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
	    public void run() {
            statusPanel.disablePanel();
            for (int swId : switchNames.keySet()) {
                int dvc = swId >> 8;
                int line = switchLines.get(swId);
                switchStatus[dvc][line].setEnabled(false);
                switchOffRB[dvc][line].setEnabled(false);
                switchOnRB[dvc][line].setEnabled(false);
            }
            repaint();
        }
    }


    private static final long serialVersionUID = 1L;
}
