package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import java.util.HashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JRadioButton;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.power.constants.ATSChannels;
import org.lsst.ccs.subsystem.power.constants.MonitorControl;
import org.lsst.ccs.subsystem.power.data.ATSPowerState;

/**
 *  Implements the REB power supply control & monitoring panel.
 *
 *  @author Owen Saxton
 */
public class ATSPowerControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final String CMND_GET_STATE = "getFullState";
    private static final Map<Integer, String> chanNames = new HashMap<>();
    static {
        chanNames.put(ATSChannels.CHAN_DIGITAL, "Dig");
        chanNames.put(ATSChannels.CHAN_ANALOG, "Ana");
        chanNames.put(ATSChannels.CHAN_CLK_HIGH, "ClkH");
        chanNames.put(ATSChannels.CHAN_CLK_LOW, "ClkL");
        chanNames.put(ATSChannels.CHAN_OD, "OD");
        chanNames.put(ATSChannels.CHAN_DPHI, "DPHI");
        chanNames.put(ATSChannels.CHAN_HV_BIAS, "HVbias");
    }
    private static final String SOLID_BAR = "\u2588\u2588\u2588\u2588";

    private final CommandSender sender;
    private final UiUtilities uiUtils;
    private SystemStatusPanel sysStatusPnl;
    private JButton btnPowerOn, btnPowerOff, btnDphiOn, btnDphiOff, btnBiasOn, btnBiasOff;
    private JButton btnOtmOn, btnOtmOff, btnFanOn, btnFanOff, btnAuxOn, btnAuxOff;
    private JRadioButton rbTempOn, rbTempOff;
    private ButtonGroup bgTempCtrl;
    private TextFieldX tfDphiVal, tfBiasVal, tfTempLimit;
    private final JLabel[] lblChanStatus = new JLabel[ATSChannels.NUM_REB_CHANNELS];
    private JLabel lblOtmStatus, lblFanStatus, lblAuxStatus, lblTempChannel, lblTempValue;
    private String tempChannel = "";

    public ATSPowerControlPanel(String agent) {
        sender = new CommandSender(agent, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, CMND_GET_STATE);
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((ATSPowerState)reply);
    }

    @Override
    public void onCommandReject(String path, String command, Object[] args) {
        if (!command.equals(CMND_GET_STATE)) {
            sender.sendCommand(true, null, CMND_GET_STATE);
        }
    }

    public void updatePanel(ATSPowerState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {
        //** Create all low-level components

        // REB power control buttons and fields
        btnPowerOn = newButton("On", UiConstants.GREEN, "powerOn");
        btnPowerOff = newButton("Off", UiConstants.RED, "powerOff");
        btnDphiOn = newButton("On", UiConstants.GREEN, "dphiOn");
        btnDphiOff = newButton("Off", UiConstants.RED, "dphiOff");
        btnBiasOn = newButton("On", UiConstants.GREEN, "hvBiasOn");
        btnBiasOff = newButton("Off", UiConstants.RED, "hvBiasOff");
        btnOtmOn = newButton("On", UiConstants.GREEN, "otmOn");
        btnOtmOff = newButton("Off", UiConstants.RED, "otmOff");
        btnFanOn = newButton("On", UiConstants.GREEN, "fanOn");
        btnFanOff = newButton("Off", UiConstants.RED, "fanOff");
        btnAuxOn = newButton("On", UiConstants.GREEN, "auxOn");
        btnAuxOff = newButton("Off", UiConstants.RED, "auxOff");

        tfDphiVal = uiUtils.newTextFieldX("00000.0", "setDphi", TextFieldX.TYPE_DOUBLE, true);
        tfBiasVal = uiUtils.newTextFieldX("00000.0", "setHvBias", TextFieldX.TYPE_DOUBLE, true);

        // Channel status labels
        for (int cs = 0; cs < lblChanStatus.length; cs++) {
            lblChanStatus[cs] = UiUtilities.newLabel(SOLID_BAR, 0);
        }
        lblOtmStatus = UiUtilities.newLabel(SOLID_BAR, 0);
        lblFanStatus = UiUtilities.newLabel(SOLID_BAR, 0);
        lblAuxStatus = UiUtilities.newLabel(SOLID_BAR, 0);

        // Temperature control buttons and fields
        rbTempOn = uiUtils.newRadioButton("On", "N");
        rbTempOff = uiUtils.newRadioButton("Off", "F");
        bgTempCtrl = new ButtonGroup();
        bgTempCtrl.add(rbTempOn);
        bgTempCtrl.add(rbTempOff);
        tfTempLimit = uiUtils.newTextFieldX("0000.0", "setTempLimit", TextFieldX.TYPE_DOUBLE, true);
        lblTempChannel = UiUtilities.newLabel(tempChannel, 0);
        lblTempValue = UiUtilities.newLabel("99.99", 0);

        //** Construct panels

        // State indicator
        sysStatusPnl = new SystemStatusPanel(sender, MonitorControl.NODE_NAME, true);

        // Power supply control
        JPanel pnlPowerControl = new JPanel();
        pnlPowerControl.setLayout(new GridBagLayout());
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.insets = new Insets(4, 6, 4, 6);

        gbp.gridx = 0;
        gbp.gridy = 0;
        gbp.gridwidth = 2;
        pnlPowerControl.add(UiUtilities.newLabel("Seq. Power", 0), gbp);
        gbp.gridy = 1;
        gbp.gridwidth = 1;
        gbp.insets.right = 2;
        pnlPowerControl.add(btnPowerOn, gbp);
        gbp.gridx++;
        gbp.insets.left = 2;
        gbp.insets.right = 6;
        pnlPowerControl.add(btnPowerOff, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        gbp.gridwidth = 2;
        gbp.insets.left = 6;
        pnlPowerControl.add(UiUtilities.newLabel("DPHI", 0), gbp);
        gbp.gridy = 1;
        gbp.gridwidth = 1;
        gbp.insets.right = 2;
        pnlPowerControl.add(btnDphiOn, gbp);
        gbp.gridx++;
        gbp.insets.left = 2;
        gbp.insets.right = 6;
        pnlPowerControl.add(btnDphiOff, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        pnlPowerControl.add(UiUtilities.newLabel("DPHI Volts", 0), gbp);
        gbp.gridy = 1;
        pnlPowerControl.add(tfDphiVal, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        gbp.gridwidth = 2;
        gbp.insets.left = 6;
        pnlPowerControl.add(UiUtilities.newLabel("HV Bias", 0), gbp);
        gbp.gridy = 1;
        gbp.gridwidth = 1;
        gbp.insets.right = 2;
        pnlPowerControl.add(btnBiasOn, gbp);
        gbp.gridx++;
        gbp.insets.left = 2;
        gbp.insets.right = 6;
        pnlPowerControl.add(btnBiasOff, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        gbp.insets.left = 6;
        pnlPowerControl.add(UiUtilities.newLabel("Bias Volts", 0), gbp);
        gbp.gridy = 1;
        pnlPowerControl.add(tfBiasVal, gbp);

        // Channel status
        JPanel pnlChanStatus = new JPanel();
        pnlChanStatus.setLayout(new GridBagLayout());
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.insets = new Insets(2, 6, 2, 6);
        gbs.gridx = 0;
        for (int j = 0; j < lblChanStatus.length; j++) {
            gbs.gridy = 0;
            pnlChanStatus.add(UiUtilities.newLabel(chanNames.get(j), 0), gbs);
            gbs.gridy = 1;
            pnlChanStatus.add(lblChanStatus[j], gbs);
            gbs.gridx++;
        }

        // REB power control
        JPanel pnlREBPower = UiUtilities.newBorderedPanel("REB Power");
        pnlREBPower.setLayout(new GridBagLayout());
        GridBagConstraints gbrp = new GridBagConstraints();
        gbrp.insets = new Insets(0, 4, 4, 4);
        gbrp.gridx = 0;
        gbrp.gridy = 0;
        pnlREBPower.add(pnlPowerControl, gbrp);
        gbrp.gridy++;
        pnlREBPower.add(pnlChanStatus, gbrp);

        // Ancillary power control
        JPanel pnlAncPower = UiUtilities.newBorderedPanel("Ancillary Power");
        GridBagConstraints gbap = new GridBagConstraints();
        gbap.insets = new Insets(0, 6, 4, 6);

        gbap.gridx = 0;
        gbap.gridy = 0;
        gbap.gridwidth = 2;
        pnlAncPower.add(UiUtilities.newLabel("OTM", 0), gbap);
        gbap.gridy++;
        gbap.gridwidth = 1;
        gbap.insets.right = 2;
        pnlAncPower.add(btnOtmOn, gbap);
        gbap.gridx++;
        gbap.insets.left = 2;
        gbap.insets.right = 6;
        pnlAncPower.add(btnOtmOff, gbap);
        gbap.gridx--;
        gbap.gridy++;
        gbap.gridwidth = 2;
        gbap.insets.left = 6;
        pnlAncPower.add(lblOtmStatus, gbap);

        gbap.gridx = 2;
        gbap.gridy = 0;
        pnlAncPower.add(UiUtilities.newLabel("Fan", 0), gbap);
        gbap.gridy++;
        gbap.gridwidth = 1;
        gbap.insets.right = 2;
        pnlAncPower.add(btnFanOn, gbap);
        gbap.gridx++;
        gbap.insets.left = 2;
        gbap.insets.right = 6;
        pnlAncPower.add(btnFanOff, gbap);
        gbap.gridx--;
        gbap.gridy++;
        gbap.gridwidth = 2;
        gbap.insets.left = 6;
        pnlAncPower.add(lblFanStatus, gbap);

        gbap.gridx = 4;
        gbap.gridy = 0;
        gbap.gridwidth = 2;
        pnlAncPower.add(UiUtilities.newLabel("Aux", 0), gbap);
        gbap.gridy++;
        gbap.gridwidth = 1;
        gbap.insets.right = 2;
        pnlAncPower.add(btnAuxOn, gbap);
        gbap.gridx++;
        gbap.insets.left = 2;
        gbap.insets.right = 6;
        pnlAncPower.add(btnAuxOff, gbap);
        gbap.gridx--;
        gbap.gridy++;
        gbap.gridwidth = 2;
        gbap.insets.left = 6;
        pnlAncPower.add(lblAuxStatus, gbap);

        // Temperature control line
        JPanel pnlTempCtrlLine = new JPanel();
        pnlTempCtrlLine.setLayout(new GridBagLayout());
        GridBagConstraints gbtcl = new GridBagConstraints();
        gbtcl.insets = new Insets(0, 6, 0, 0);
        gbtcl.gridx = 0;
        gbtcl.gridy = 0;
        pnlTempCtrlLine.add(rbTempOn, gbtcl);
        gbtcl.gridx++;
        gbtcl.insets.right = 6;
        pnlTempCtrlLine.add(rbTempOff, gbtcl);
        gbtcl.gridx++;
        gbtcl.insets.right = 0;
        pnlTempCtrlLine.add(UiUtilities.newLabel("Limit:", 0), gbtcl);
        gbtcl.gridx++;
        pnlTempCtrlLine.add(tfTempLimit, gbtcl);
        gbtcl.gridx++;
        gbtcl.insets.right = 6;
        pnlTempCtrlLine.add(UiUtilities.newLabel("C", 0), gbtcl);
    
        // Temperature channel line
        JPanel pnlTempChannel = new JPanel();
        pnlTempChannel.setLayout(new GridBagLayout());
        GridBagConstraints gbtnl = new GridBagConstraints();
        gbtnl.insets = new Insets(0, 6, 0, 0);
        gbtnl.gridx = 0;
        gbtnl.gridy = 0;
        pnlTempChannel.add(UiUtilities.newLabel("Channel:", 0), gbtnl);
        gbtnl.gridx++;
        lblTempChannel = UiUtilities.newLabel(tempChannel, 0);
        pnlTempChannel.add(lblTempChannel, gbtnl);
        gbtnl.gridx++;
        gbtnl.insets.left = 20;
        pnlTempChannel.add(UiUtilities.newLabel("Value:", 0), gbtnl);
        gbtnl.gridx++;
        gbtnl.insets.left = 6;
        gbtnl.insets.right = 6;
        pnlTempChannel.add(lblTempValue, gbtnl);

        // Temperature control panel
        JPanel pnlTempCtrl = UiUtilities.newBorderedPanel("Temperature Control");
        pnlTempCtrl.setLayout(new GridBagLayout());
        GridBagConstraints gbtc = new GridBagConstraints();
        gbtc.insets = new Insets(4, 0, 4, 0);
        gbtc.gridx = 0;
        gbtc.gridy = 0;
        pnlTempCtrl.add(pnlTempCtrlLine, gbtc);
        gbtc.gridy++;
        gbtc.insets.top = 0;
        pnlTempCtrl.add(pnlTempChannel, gbtc);

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(4, 0, 4, 0);
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(sysStatusPnl, gbc);
        gbc.gridy++;
        add(pnlREBPower, gbc);
        gbc.gridy++;
        add(pnlAncPower, gbc);
        gbc.gridy++;
        add(pnlTempCtrl, gbc);
    }

    private void updateTempChannel(String name) {
        if (name.equals(tempChannel)) return;
        tempChannel = name;
        lblTempChannel.setText(tempChannel);
        Dimension d = UiUtilities.getLabelSize(tempChannel);
        lblTempChannel.setMinimumSize(d);
        lblTempChannel.setPreferredSize(d);
    }

    private JButton newButton(String text, Color color, String command) {
        JButton btn = uiUtils.newButton(text, command, true);
        btn.setForeground(color);
        return btn;
    }

    @Override
    public void handleButton(String command) {
        sender.sendCommand(null, command);
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        sender.sendCommand(null, name, (Double)value);
    }

    @Override
    public void handleRadioButton(String name) {
        boolean on = name.charAt(0) == 'N';
        sender.sendCommand(null, "setTempFanControl", on);
    }

    class UpdatePowerStatus implements Runnable {

        private final ATSPowerState rs;

        UpdatePowerStatus(ATSPowerState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            sysStatusPnl.updatePanel(rs.getTickMillis());

            btnPowerOn.setEnabled(true);
            btnPowerOff.setEnabled(true);
            btnDphiOn.setEnabled(true);
            btnDphiOff.setEnabled(true);
            btnBiasOn.setEnabled(true);
            btnBiasOff.setEnabled(true);

            tfBiasVal.update(rs.getHvBias(), true);
            tfDphiVal.update(rs.getDphi(), true);

            Boolean[] state = rs.getPowerState();
            for (int j = 0; j < lblChanStatus.length; j++) {
                lblChanStatus[j].setForeground(state[j] == null ? UiConstants.BLUE :
                                               state[j] ? UiConstants.GREEN : UiConstants.RED);
                lblChanStatus[j].setEnabled(true);
            }
            
            btnOtmOn.setEnabled(true);
            btnOtmOff.setEnabled(true);
            btnFanOn.setEnabled(true);
            btnFanOff.setEnabled(true);
            btnAuxOn.setEnabled(true);
            btnAuxOff.setEnabled(true);
            lblOtmStatus.setForeground(state[ATSChannels.CHAN_OTM] == null ? UiConstants.BLUE :
                                       state[ATSChannels.CHAN_OTM] ? UiConstants.GREEN : UiConstants.RED);
            lblOtmStatus.setEnabled(true);
            lblFanStatus.setForeground(state[ATSChannels.CHAN_FAN] == null ? UiConstants.BLUE :
                                       state[ATSChannels.CHAN_FAN] ? UiConstants.GREEN : UiConstants.RED);
            lblFanStatus.setEnabled(true);
            lblAuxStatus.setForeground(state[ATSChannels.CHAN_AUX] == null ? UiConstants.BLUE :
                                       state[ATSChannels.CHAN_AUX] ? UiConstants.GREEN : UiConstants.RED);
            lblAuxStatus.setEnabled(true);

            rbTempOn.setEnabled(true);
            rbTempOff.setEnabled(true);
            JRadioButton selButton = rs.getTempFanControl() ? rbTempOn : rbTempOff;
            selButton.setSelected(true);
            tfTempLimit.update(rs.getTempLimit(), true);
            updateTempChannel(rs.getTempChannel());
            lblTempValue.setText(String.format("%.1f", rs.getTempValue()));
            lblTempChannel.setEnabled(true);
            lblTempValue.setEnabled(true);

            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            sysStatusPnl.disablePanel();
            btnPowerOn.setEnabled(false);
            btnPowerOff.setEnabled(false);
            btnDphiOn.setEnabled(false);
            btnDphiOff.setEnabled(false);
            btnBiasOn.setEnabled(false);
            btnBiasOff.setEnabled(false);
            tfBiasVal.setDisabled();
            tfDphiVal.setDisabled();
            for (JLabel lbl : lblChanStatus) {
                lbl.setEnabled(false);
            }
            btnOtmOn.setEnabled(false);
            btnOtmOff.setEnabled(false);
            btnFanOn.setEnabled(false);
            btnFanOff.setEnabled(false);
            btnAuxOn.setEnabled(false);
            btnAuxOff.setEnabled(false);
            lblOtmStatus.setEnabled(false);
            lblFanStatus.setEnabled(false);
            lblAuxStatus.setEnabled(false);
            rbTempOn.setEnabled(false);
            rbTempOff.setEnabled(false);
            tfTempLimit.setDisabled();
            lblTempChannel.setEnabled(false);
            lblTempValue.setEnabled(false);
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
