package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import javax.swing.border.TitledBorder;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.power.constants.MonitorControl;
import org.lsst.ccs.subsystem.power.constants.QuadBoxSwitches;
import org.lsst.ccs.subsystem.power.data.QuadBoxState;
import org.lsst.ccs.subsystem.power.constants.SwitchState;

/**
 *  Implements the REB power supply control & monitoring panel.
 *
 *  @author Owen Saxton
 */
public class QuadBoxControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final String[] panelNames = new String[QuadBoxSwitches.NUM_DEVICES];
    static {
        panelNames[QuadBoxSwitches.DEVC_BFR] = "BFR";
        panelNames[QuadBoxSwitches.DEVC_PDU_5V] = "5V Clean PDU";
        panelNames[QuadBoxSwitches.DEVC_PDU_24VC] = "24V Clean PDU";
        panelNames[QuadBoxSwitches.DEVC_PDU_24VD] = "24V Dirty PDU";
        panelNames[QuadBoxSwitches.DEVC_PDU_48V] = "48V Dirty PDU";
        panelNames[QuadBoxSwitches.DEVC_REB_BULK] = "REB Bulk PS";
        panelNames[QuadBoxSwitches.DEVC_MAQ20] = "HCU/PLC Control";
    }
    private static final Map<Integer, String> switchNames = new LinkedHashMap<>();
    static {
        switchNames.put(QuadBoxSwitches.SW_PDU_5V, "5V Clean PDU");
        switchNames.put(QuadBoxSwitches.SW_PDU_24VC, "24V Clean PDU");
        switchNames.put(QuadBoxSwitches.SW_PDU_24VD, "24V Dirty PDU");
        switchNames.put(QuadBoxSwitches.SW_PDU_48V, "48V Dirty Power");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_PS_0, "REB PS 0-2");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_PS_1, "REB PS 3-5");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_PS_2, "REB PS 6-8");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_PS_3, "REB PS 9-12");
        switchNames.put(QuadBoxSwitches.SW_TRIM_HEATERS, "Trim Heaters");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_PS_4, "REB PS spare");
        switchNames.put(QuadBoxSwitches.SW_OTM_0_A, "OTM 0-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_0_B, "OTM 0-B");
        switchNames.put(QuadBoxSwitches.SW_OTM_1_A, "OTM 1-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_1_B, "OTM 1-B");
        switchNames.put(QuadBoxSwitches.SW_OTM_2_A, "OTM 2-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_2_B, "OTM 2-B");
        switchNames.put(QuadBoxSwitches.SW_OTM_3_A, "OTM 3-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_3_B, "OTM 3-B");
        switchNames.put(QuadBoxSwitches.SW_OTM_4_A, "OTM 4-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_4_B, "OTM 4-B");
        switchNames.put(QuadBoxSwitches.SW_OTM_5_A, "OTM 5-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_5_B, "OTM 5-B");
        switchNames.put(QuadBoxSwitches.SW_MAIN_24C, "Main");
        switchNames.put(QuadBoxSwitches.SW_CRYO_MAQ20, "Cryo MAQ20");
        switchNames.put(QuadBoxSwitches.SW_BODY_MAQ20, "Body MAQ20");
        switchNames.put(QuadBoxSwitches.SW_GAUGES, "Gauges");
        switchNames.put(QuadBoxSwitches.SW_ION_PUMPS, "Ion pumps");
        switchNames.put(QuadBoxSwitches.SW_FES_CAROUSEL, "FES carousel");
        switchNames.put(QuadBoxSwitches.SW_FES_CHANGER_C, "FES changer");
        switchNames.put(QuadBoxSwitches.SW_FES_LOADER_C, "FES loader");
        switchNames.put(QuadBoxSwitches.SW_SHTR_PLC_1, "Shtr PLC 1");
        switchNames.put(QuadBoxSwitches.SW_SHTR_PLC_2, "Shtr PLC 2");
        switchNames.put(QuadBoxSwitches.SW_MAIN_24D, "Main");
        switchNames.put(QuadBoxSwitches.SW_CRYO_TURBO, "Cryo turbo");
        switchNames.put(QuadBoxSwitches.SW_HEX_TURBO, "Hex turbo");
        switchNames.put(QuadBoxSwitches.SW_FES_CLAMPS, "FES clamps");
        switchNames.put(QuadBoxSwitches.SW_FES_BRAKES, "FES brakes");
        switchNames.put(QuadBoxSwitches.SW_FES_CHANGER_D, "FES changer");
        switchNames.put(QuadBoxSwitches.SW_FES_LOADER_D, "FES loader");
        switchNames.put(QuadBoxSwitches.SW_SHTR_BRAKES, "Shtr brakes");
        switchNames.put(QuadBoxSwitches.SW_MAIN_48, "Main");
        switchNames.put(QuadBoxSwitches.SW_PURGE_FAN, "MPC fan");
        switchNames.put(QuadBoxSwitches.SW_FES_CAROUSEL_D, "FES carousel");
        switchNames.put(QuadBoxSwitches.SW_FES_HEATER, "FES heater"); 
        switchNames.put(QuadBoxSwitches.SW_SHTR_MOTOR_1, "Shtr Motor 1");
        switchNames.put(QuadBoxSwitches.SW_SHTR_MOTOR_2, "Shtr Motor 2");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_0, "REB PS 0-2");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_1, "REB PS 3-5");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_2, "REB PS 6-8");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_3, "REB PS 9-12");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_4, "REB PS Spare");
        switchNames.put(QuadBoxSwitches.SW_REB_PS_HCU, "REB PS HCU");
        switchNames.put(QuadBoxSwitches.SW_CRYOSTAT_HCU, "Cryostat HCU");
        switchNames.put(QuadBoxSwitches.SW_FES_HCU, "FES HCU");
        switchNames.put(QuadBoxSwitches.SW_SHUTTER_HCU, "Shutter HCU");
        switchNames.put(QuadBoxSwitches.SW_FES_PLC, "FES PLC");
    }
    private static final Map<Integer, Integer> bfrSwitchMap = new HashMap<>();
    static {
        bfrSwitchMap.put(QuadBoxSwitches.DEVC_PDU_5V, QuadBoxSwitches.SW_PDU_5V);
        bfrSwitchMap.put(QuadBoxSwitches.DEVC_PDU_24VC, QuadBoxSwitches.SW_PDU_24VC);
        bfrSwitchMap.put(QuadBoxSwitches.DEVC_PDU_24VD, QuadBoxSwitches.SW_PDU_24VD);
        bfrSwitchMap.put(QuadBoxSwitches.DEVC_PDU_48V, QuadBoxSwitches.SW_PDU_48V);
    }
    private static final Map<Integer, Integer> mainSwitchMap = new HashMap<>();
    static {
        mainSwitchMap.put(QuadBoxSwitches.DEVC_PDU_24VC, QuadBoxSwitches.SW_MAIN_24C);
        mainSwitchMap.put(QuadBoxSwitches.DEVC_PDU_24VD, QuadBoxSwitches.SW_MAIN_24D);
        mainSwitchMap.put(QuadBoxSwitches.DEVC_PDU_48V, QuadBoxSwitches.SW_MAIN_48);
    }
    private static final int STATUS_LABEL_WIDTH = UiUtilities.maxEnumLabelWidth(SwitchState.class);

    private final CommandSender sender;
    private final UiUtilities uiUtils;
    private SystemStatusPanel statusPanel;
    private final JPanel topPanel = new JPanel();
    private final JPanel[] switchPanel = new JPanel[QuadBoxSwitches.NUM_DEVICES];
    private final JLabel[][] switchLabel = new JLabel[QuadBoxSwitches.NUM_DEVICES][];
    private final JLabel[][] switchStatus = new JLabel[QuadBoxSwitches.NUM_DEVICES][];
    private final ButtonGroup[][] switchBG = new ButtonGroup[QuadBoxSwitches.NUM_DEVICES][];
    private final JRadioButton[][] switchOffRB = new JRadioButton[QuadBoxSwitches.NUM_DEVICES][];
    private final JRadioButton[][] switchOnRB = new JRadioButton[QuadBoxSwitches.NUM_DEVICES][];
    private final Map<Integer, Integer> switchLines = new HashMap<>();

    public QuadBoxControlPanel(String agent) {
        sender = new CommandSender(agent, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, "getFullState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((QuadBoxState)reply);
    }

    public void updatePanel(QuadBoxState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {
        //** Create all low-level components

        // State indicator
        statusPanel = new SystemStatusPanel(sender, MonitorControl.NODE_NAME);

        // Switch panel elements
        for (int dvc = 0; dvc < QuadBoxSwitches.NUM_DEVICES; dvc++) {
            int nSwitch = QuadBoxSwitches.NUM_SWITCHES[dvc];
            switchLabel[dvc] = new JLabel[nSwitch + 1];
            switchStatus[dvc] = new JLabel[nSwitch + 1];
            switchBG[dvc] = new ButtonGroup[nSwitch + 1];
            switchOnRB[dvc] = new JRadioButton[nSwitch + 1];
            switchOffRB[dvc] = new JRadioButton[nSwitch + 1];
        }
        int[] lines = new int[QuadBoxSwitches.NUM_DEVICES];
        for (int swId : switchNames.keySet()) {
            int dvc = swId >> 8;
            int line = lines[dvc]++;
            switchLines.put(swId, line);
            switchLabel[dvc][line] = UiUtilities.newLabel(switchNames.get(swId) + ": ", 0);
            switchStatus[dvc][line] = UiUtilities.newLabel("X", STATUS_LABEL_WIDTH);
            JRadioButton on = switchOnRB[dvc][line] = uiUtils.newRadioButton("On", "N" + Integer.toString(swId));
            JRadioButton off = switchOffRB[dvc][line] = uiUtils.newRadioButton("Off", "F" + Integer.toString(swId));
            ButtonGroup bg = switchBG[dvc][line] = new ButtonGroup();
            bg.add(on);
            bg.add(off);
        }

        //** Construct panels

        // Top panel
        GridBagConstraints gbt = new GridBagConstraints();
        topPanel.setLayout(new GridBagLayout());
        gbt.gridx = 0;
        gbt.gridy = 0;
        topPanel.add(statusPanel, gbt);
        
        // Switch panels
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.WEST;
        gbs.insets = new Insets(0, 0, 4, 4);
        for (int dvc = 0; dvc < QuadBoxSwitches.NUM_DEVICES; dvc++) {
            JPanel panel = switchPanel[dvc] = UiUtilities.newBorderedPanel(panelNames[dvc]);
            gbs.gridy = 0;
            gbs.insets.top = 4;
            for (int line = 0; line < switchLabel[dvc].length; line++) {
                if (switchLabel[dvc][line] == null) continue;
                gbs.gridx = 0;
                gbs.insets.left = 4;
                panel.add(switchLabel[dvc][line], gbs);
                gbs.gridx++;
                panel.add(switchStatus[dvc][line], gbs);
                gbs.gridx++;
                panel.add(switchOffRB[dvc][line], gbs);
                gbs.gridx++;
                gbs.insets.left = 0;
                panel.add(switchOnRB[dvc][line], gbs);
                gbs.gridy++;
                gbs.insets.top = 0;
            }
            panel.setMinimumSize(panel.getPreferredSize());
        }

        // Switch panel columns
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        JPanel leftColumn = new JPanel();
        leftColumn.setLayout(new GridBagLayout());
        leftColumn.add(switchPanel[QuadBoxSwitches.DEVC_BFR], gbc);
        gbc.gridy++;
        leftColumn.add(switchPanel[QuadBoxSwitches.DEVC_REB_BULK], gbc);
        gbc.gridy++;
        leftColumn.add(switchPanel[QuadBoxSwitches.DEVC_MAQ20], gbc);
        JPanel centerColumn = new JPanel();
        centerColumn.setLayout(new GridBagLayout());
        gbc.gridy = 0;
        centerColumn.add(switchPanel[QuadBoxSwitches.DEVC_PDU_5V], gbc);
        gbc.gridy++;
        centerColumn.add(switchPanel[QuadBoxSwitches.DEVC_PDU_48V], gbc);
        JPanel rightColumn = new JPanel();
        rightColumn.setLayout(new GridBagLayout());
        gbc.gridy = 0;
        rightColumn.add(switchPanel[QuadBoxSwitches.DEVC_PDU_24VC], gbc);
        gbc.gridy++;
        rightColumn.add(switchPanel[QuadBoxSwitches.DEVC_PDU_24VD], gbc);

        // Whole panel
        setLayout(new GridBagLayout());
        gbc.insets.left = 0;
        gbc.insets.right = 0;
        gbc.insets.bottom = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 3;
        add(topPanel, gbc);
        gbc.gridy++;
        gbc.gridwidth = 1;
        add(leftColumn, gbc);
        gbc.gridx++;
        add(centerColumn, gbc);
        gbc.gridx++;
        add(rightColumn, gbc);
    }

    @Override
    public void handleRadioButton(String name) {
        boolean on = name.charAt(0) == 'N';
        int swId = Integer.valueOf(name.substring(1));
        sender.sendCommand(null, "setSwitchOn", QuadBoxSwitches.getSwitchName(swId), on);
    }

    class UpdatePowerStatus implements Runnable {

        private final QuadBoxState qs;

        UpdatePowerStatus(QuadBoxState qs) {
            this.qs = qs;
        }

        @Override
        public void run() {
            statusPanel.updatePanel(qs.getTickMillis());

            for (int swId : switchNames.keySet()) {
                int devc = swId >> 8;
                int line = switchLines.get(swId);
                SwitchState st = qs.getSwitchState(swId);
                st = st == null ? SwitchState.OFFLINE : st;
                JLabel label = switchStatus[devc][line];
                label.setText(st.toString());
                label.setForeground(st == SwitchState.ON ? UiConstants.GREEN :
                                    st == SwitchState.OFF ? Color.BLACK : UiConstants.BLUE);
                if (st != SwitchState.OFFLINE) {
                    JRadioButton selButton = st == SwitchState.ON ? switchOnRB[devc][line] : switchOffRB[devc][line];
                    selButton.setSelected(true);
                }
                switchStatus[devc][line].setEnabled(true);
                switchOffRB[devc][line].setEnabled(true);
                switchOnRB[devc][line].setEnabled(true);
            }
            for (int devc : bfrSwitchMap.keySet()) {
                SwitchState bfrSwSt = qs.getSwitchState(bfrSwitchMap.get(devc));
                Integer mainSwId = mainSwitchMap.get(devc);
                SwitchState mainSwSt = mainSwId == null ? SwitchState.ON : qs.getSwitchState(mainSwId);
                String suffix = bfrSwSt != SwitchState.ON ? " (BFR off)" : mainSwSt != SwitchState.ON ? " (Main off)" : "";
                Color color = bfrSwSt != SwitchState.ON ? UiConstants.RED :
                              mainSwSt != SwitchState.ON ? UiConstants.PURPLE : UiConstants.BLUE;
                JPanel panel = switchPanel[devc];
                TitledBorder border = (TitledBorder)panel.getBorder();
                border.setTitle(panelNames[devc] + suffix);
                border.setTitleColor(color);
            }
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            statusPanel.disablePanel();
            for (int swId : switchNames.keySet()) {
                int dvc = swId >> 8;
                int line = switchLines.get(swId);
                switchStatus[dvc][line].setEnabled(false);
                switchOffRB[dvc][line].setEnabled(false);
                switchOnRB[dvc][line].setEnabled(false);
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
