package org.lsst.ccs.subsystem.power.ui;

import org.lsst.ccs.subsystem.power.ui.*;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.power.constants.PathFinderQuadBoxSwitches;
import org.lsst.ccs.subsystem.power.data.PathFinderQuadBoxState;
import org.lsst.ccs.subsystem.power.constants.SwitchState;

/**
 *  Implements the REB power supply control & monitoring panel.
 *
 *  @author Owen Saxton
 */

public class PathFinderQuadBoxControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final String[] panelNames = new String[PathFinderQuadBoxSwitches.NUM_DEVICES];
    static {
        panelNames[PathFinderQuadBoxSwitches.DEVC_BFR] = "BFR";
        panelNames[PathFinderQuadBoxSwitches.DEVC_PDU_5V] = "5V Clean PDU";
        panelNames[PathFinderQuadBoxSwitches.DEVC_PDU_24VC] = "24V Clean PDU";
        panelNames[PathFinderQuadBoxSwitches.DEVC_PDU_24VD] = "24V Dirty PDU";
        panelNames[PathFinderQuadBoxSwitches.DEVC_PDU_48V] = "48V Dirty PDU";
    }
    private static final Map<Integer, String> switchNames = new LinkedHashMap<>();
    static {
        switchNames.put(PathFinderQuadBoxSwitches.SW_PDU_5V, "5V Clean PDU");
        switchNames.put(PathFinderQuadBoxSwitches.SW_PDU_24VC, "24V Clean PDU");
        switchNames.put(PathFinderQuadBoxSwitches.SW_PDU_24VD, "24V Dirty PDU");
        switchNames.put(PathFinderQuadBoxSwitches.SW_PDU_48V, "48V Dirty PDU");

        switchNames.put(PathFinderQuadBoxSwitches.SW_HEATER_1, "Heater 1");
        switchNames.put(PathFinderQuadBoxSwitches.SW_HEATER_2, "Heater 2");
 
        switchNames.put(PathFinderQuadBoxSwitches.SW_MAIN_24C, "Main");
        switchNames.put(PathFinderQuadBoxSwitches.SW_ION_PUMP, "Ion pumps");
        switchNames.put(PathFinderQuadBoxSwitches.SW_FES_ENCODER, "FES Encoder");
        switchNames.put(PathFinderQuadBoxSwitches.SW_HCU_FES_SHU, "FES/Shutter HCUs");
        switchNames.put(PathFinderQuadBoxSwitches.SW_BODY_PURGE, "RFG/UTC/BPU MAQ20s");
        switchNames.put(PathFinderQuadBoxSwitches.SW_BPU_MAQ20, "VALVES/PTH MAQ20");
        switchNames.put(PathFinderQuadBoxSwitches.SW_GAUGES, "Gauges");


        switchNames.put(PathFinderQuadBoxSwitches.SW_MAIN_24D, "Main");
        switchNames.put(PathFinderQuadBoxSwitches.SW_PFTURBO, "PF turbo");
        switchNames.put(PathFinderQuadBoxSwitches.SW_PFSCROLL, "PF scroll");
	switchNames.put(PathFinderQuadBoxSwitches.SW_TURBO_HEATER, "PF turbo heater"); 

        switchNames.put(PathFinderQuadBoxSwitches.SW_MAIN_48, "Main");
        switchNames.put(PathFinderQuadBoxSwitches.SW_PURGE_FAN, "Purge fan");
    }
    private static final int[] nameWidths = new int[PathFinderQuadBoxSwitches.NUM_DEVICES];
    static {
        nameWidths[PathFinderQuadBoxSwitches.DEVC_BFR] = 110;
        nameWidths[PathFinderQuadBoxSwitches.DEVC_PDU_5V] = 70;
        nameWidths[PathFinderQuadBoxSwitches.DEVC_PDU_24VC] = 130;
        nameWidths[PathFinderQuadBoxSwitches.DEVC_PDU_24VD] = 105;
        nameWidths[PathFinderQuadBoxSwitches.DEVC_PDU_48V] = 105;
    }
    private static final Map<Integer, Integer> bfrSwitchMap = new HashMap<>();
    static {
        bfrSwitchMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_5V, PathFinderQuadBoxSwitches.SW_PDU_5V);
        bfrSwitchMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_24VC, PathFinderQuadBoxSwitches.SW_PDU_24VC);
        bfrSwitchMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_24VD, PathFinderQuadBoxSwitches.SW_PDU_24VD);
        bfrSwitchMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_48V, PathFinderQuadBoxSwitches.SW_PDU_48V);
    }
    private static final Map<Integer, Integer> mainSwitchMap = new HashMap<>();
    static {
        mainSwitchMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_24VC, PathFinderQuadBoxSwitches.SW_MAIN_24C);
        mainSwitchMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_24VD, PathFinderQuadBoxSwitches.SW_MAIN_24D);
        mainSwitchMap.put(PathFinderQuadBoxSwitches.DEVC_PDU_48V, PathFinderQuadBoxSwitches.SW_MAIN_48);
    }

    private static final int statusLabelWidth = UiUtilities.maxEnumLabelWidth(SwitchState.class);

    private final CommandSender sender;
    private final UiUtilities uiUtils;

    private SystemStatusPanel statusPanel;
    private final JPanel topPanel = new JPanel();
    private final JPanel statePanel = new JPanel();
    private final JLabel stateLabel = new JLabel("System State: ");
    private final JLabel stateStatus = new JLabel("....");
    private final JPanel periodPanel = new JPanel();
    private final JLabel periodLabel = new JLabel("Update Period: ");
    private final JTextField periodTextField = new JTextField("...");
    private final JLabel periodUnitsLabel = new JLabel("sec");
    private final JPanel[] switchPanel = new JPanel[PathFinderQuadBoxSwitches.NUM_DEVICES];
    private final JLabel[][] switchLabel = new JLabel[PathFinderQuadBoxSwitches.NUM_DEVICES][];
    private final JLabel[][] switchStatus = new JLabel[PathFinderQuadBoxSwitches.NUM_DEVICES][];
    private final ButtonGroup[][] switchBG = new ButtonGroup[PathFinderQuadBoxSwitches.NUM_DEVICES][];
    private final JRadioButton[][] switchOffRB = new JRadioButton[PathFinderQuadBoxSwitches.NUM_DEVICES][];
    private final JRadioButton[][] switchOnRB = new JRadioButton[PathFinderQuadBoxSwitches.NUM_DEVICES][];
    private final Map<Integer, Integer> switchLines = new HashMap<>();
    private double powerPeriod = 5;

    public PathFinderQuadBoxControlPanel(String agent) {
	sender = new CommandSender(agent, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    private void initComponents()
    {
        //** Create all low-level components

        // State indicator
        statusPanel = new SystemStatusPanel(sender);

        // State indicator
        Dimension d = stateStatus.getPreferredSize();
        d.width = 70;
        stateStatus.setPreferredSize(d);
        stateStatus.setMinimumSize(d);

        // Update period
        d = periodTextField.getPreferredSize();
        d.width = 50;
        periodTextField.setPreferredSize(d);
        periodTextField.setMinimumSize(d);
        periodTextField.setHorizontalAlignment(JTextField.CENTER);
        periodTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setUpdatePeriod();
            }
        });


        // Switch panel elements                                                                                                                                        
        for (int dvc = 0; dvc < PathFinderQuadBoxSwitches.NUM_DEVICES; dvc++) {
            int nSwitch = PathFinderQuadBoxSwitches.NUM_SWITCHES[dvc];
            switchLabel[dvc] = new JLabel[nSwitch + 1];
            switchStatus[dvc] = new JLabel[nSwitch + 1];
            switchBG[dvc] = new ButtonGroup[nSwitch + 1];
            switchOnRB[dvc] = new JRadioButton[nSwitch + 1];
            switchOffRB[dvc] = new JRadioButton[nSwitch + 1];
        }
        int[] lines = new int[PathFinderQuadBoxSwitches.NUM_DEVICES];
	for (int swId : switchNames.keySet()) {
            int dvc = swId >> 8;
            int line = lines[dvc]++;
            switchLines.put(swId, line);
            switchLabel[dvc][line] = UiUtilities.newLabel(switchNames.get(swId) + ": ", 0);
            switchStatus[dvc][line] = UiUtilities.newLabel("X", statusLabelWidth);
            JRadioButton on = switchOnRB[dvc][line] = uiUtils.newRadioButton("On", "N" + Integer.toString(swId));
            JRadioButton off = switchOffRB[dvc][line] = uiUtils.newRadioButton("Off", "F" + Integer.toString(swId));
            ButtonGroup bg = switchBG[dvc][line] = new ButtonGroup();
            bg.add(on);
            bg.add(off);
        }

        //** Construct panels

        // Power state
        statePanel.add(stateLabel);
        statePanel.add(stateStatus);

        // Update period
        periodPanel.add(periodLabel);
        periodPanel.add(periodTextField);
        periodPanel.add(periodUnitsLabel);

        // Top panel
        GridBagConstraints gbt = new GridBagConstraints();
        topPanel.setLayout(new GridBagLayout());
        gbt.gridx = 0;
        gbt.gridy = 0;
        topPanel.add(statusPanel, gbt);

        // Switch panels
        GridBagConstraints gbs = new GridBagConstraints();
        for (int dvc = 0; dvc < PathFinderQuadBoxSwitches.NUM_DEVICES; dvc++) {
            JPanel panel = switchPanel[dvc] = new JPanel();
            TitledBorder border = new TitledBorder(new LineBorder(Color.BLACK), panelNames[dvc]);
            border.setTitleColor(UiConstants.BLUE);
            panel.setBorder(border);
            panel.setLayout(new GridBagLayout());
            gbs.gridy = 0;
            gbs.insets.top = 0;
            for (int line = 0; line < switchLabel[dvc].length; line++) {
                if (switchLabel[dvc][line] == null) continue;
                gbs.gridx = 0;
                gbs.insets.left = 4;
                gbs.insets.right = 0;
                panel.add(switchLabel[dvc][line], gbs);
                gbs.gridx++;
                gbs.insets.left = 0;
                panel.add(switchStatus[dvc][line], gbs);
                gbs.gridx++;
                gbs.insets.right = 4;
                panel.add(switchOffRB[dvc][line], gbs);
                gbs.gridx++;
                panel.add(switchOnRB[dvc][line], gbs);
                gbs.gridy++;
                gbs.insets.top = -4;
            }
            panel.setMinimumSize(panel.getPreferredSize());
        }

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 3;
        add(topPanel, gbc);
        gbc.gridy++;
        gbc.gridwidth = 1;
        add(switchPanel[PathFinderQuadBoxSwitches.DEVC_BFR], gbc);
        gbc.gridx++;
        add(switchPanel[PathFinderQuadBoxSwitches.DEVC_PDU_5V], gbc);
        gbc.gridx++;
        add(switchPanel[PathFinderQuadBoxSwitches.DEVC_PDU_24VC], gbc);
        gbc.gridy++;
        gbc.gridx = 0;
        add(switchPanel[PathFinderQuadBoxSwitches.DEVC_PDU_24VD], gbc);
        gbc.gridx++;
        add(switchPanel[PathFinderQuadBoxSwitches.DEVC_PDU_48V], gbc);
    }

    private void setUpdatePeriod() {
        try {
            double value = Double.valueOf(periodTextField.getText());
            sender.sendCommand(null, "setUpdatePeriod", (int)(1000 * value));
            periodTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            periodTextField.setText(String.valueOf(powerPeriod));
        }
    }

    private void setSwitchOn(int swId, boolean on) {
        sender.sendCommand(null, "setNamedSwitchOn", PathFinderQuadBoxSwitches.switchIdToName.get(swId), on);
    }

    public void updateControlPanel(PathFinderQuadBoxState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }


    public void initPanel() {
        sender.sendCommand(true, null, "getFullState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((PathFinderQuadBoxState)reply);
    }

    public void updatePanel(PathFinderQuadBoxState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    @Override
    public void handleRadioButton(String name) {
        boolean on = name.charAt(0) == 'N';
        int swId = Integer.valueOf(name.substring(1));
        sender.sendCommand(null, "setNamedSwitchOn", PathFinderQuadBoxSwitches.switchIdToName.get(swId), on);
    }


    class UpdatePowerStatus implements Runnable {

        private final PathFinderQuadBoxState qs;

        UpdatePowerStatus(PathFinderQuadBoxState qs) {
            this.qs = qs;
        }

        @Override
        public void run() {

            statusPanel.updatePanel(qs.getTickMillis());

            stateStatus.setText("RUNNING");
            stateStatus.setForeground(UiConstants.GREEN);

            periodTextField.setEnabled(true);
            powerPeriod = qs.getTickMillis() / 1000.0;
            periodTextField.setText(String.valueOf(powerPeriod));

            for (int swId : switchNames.keySet()) {
                int devc = swId >> 8;
                int line = switchLines.get(swId);
                SwitchState st = qs.getSwitchState(swId);
                st = st == null ? SwitchState.OFFLINE : st;
                JLabel label = switchStatus[devc][line];
                label.setText(st.toString());
                label.setForeground(st == SwitchState.ON ? UiConstants.GREEN :
                                    st == SwitchState.OFF ? Color.BLACK : UiConstants.BLUE);
                if (st != SwitchState.OFFLINE) {
                    JRadioButton selButton = st == SwitchState.ON ? switchOnRB[devc][line] : switchOffRB[devc][line];
                    selButton.setSelected(true);
                }
                switchStatus[devc][line].setEnabled(true);
                switchOffRB[devc][line].setEnabled(true);
                switchOnRB[devc][line].setEnabled(true);
            }
            for (int devc : bfrSwitchMap.keySet()) {
                SwitchState bfrSwSt = qs.getSwitchState(bfrSwitchMap.get(devc));
                Integer mainSwId = mainSwitchMap.get(devc);
                SwitchState mainSwSt = mainSwId == null ? SwitchState.ON : qs.getSwitchState(mainSwId);
                String suffix = bfrSwSt != SwitchState.ON ? " (BFR off)" : mainSwSt != SwitchState.ON ? " (Main off)" : "";
                Color color = bfrSwSt != SwitchState.ON ? UiConstants.RED :
                              mainSwSt != SwitchState.ON ? UiConstants.PURPLE : UiConstants.BLUE;
                JPanel panel = switchPanel[devc];
                TitledBorder border = (TitledBorder)panel.getBorder();
                border.setTitle(panelNames[devc] + suffix);
                border.setTitleColor(color);
            }
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
	    public void run() {
            statusPanel.disablePanel();
            for (int swId : switchNames.keySet()) {
                int dvc = swId >> 8;
                int line = switchLines.get(swId);
                switchStatus[dvc][line].setEnabled(false);
                switchOffRB[dvc][line].setEnabled(false);
                switchOnRB[dvc][line].setEnabled(false);
            }
            repaint();
        }
    }

    private static final long serialVersionUID = 1L;
}
