package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import java.util.HashMap;
import java.util.Map;
import javax.swing.JComponent;
import org.lsst.ccs.drivers.auxelex.RebPS;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.RebPsState;

/**
 *  Implements the REB power supply control & monitoring panel.
 *
 *  @author Owen Saxton
 */
public class RebPsControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final int REB_Q = RebPS.REB_QUANTUM, N_REB = 2 * REB_Q, N_PS = 8,
                             PS_DIG = 0, PS_ANA = 1, PS_OD = 2, PS_CLKH = 3, PS_CLKL = 4, PS_HTR = 5,
                             PS_DPHI = 6, PS_HV = 7;
    private static final String[] psNames = new String[N_PS];
    static {
        psNames[PS_DIG] = "Dig";
        psNames[PS_ANA] = "Ana";
        psNames[PS_OD] = "OD";
        psNames[PS_CLKH] = "ClkH";
        psNames[PS_CLKL] = "ClkL";
        psNames[PS_HTR] = "Htr";
        psNames[PS_DPHI] = "Dphi";
        psNames[PS_HV] = "HVbias";
    }
    private static final Map<Integer, Integer> btnMapSR = new HashMap<>();
    static {
        for (int reb = 0; reb < N_REB; reb++) {
            btnMapSR.put((reb << 4) | RebPS.PS_DIGITAL, (reb << 4) | PS_DIG);
            btnMapSR.put((reb << 4) | RebPS.PS_ANALOG, (reb << 4) | PS_ANA);
            btnMapSR.put((reb << 4) | RebPS.PS_OD, (reb << 4) | PS_OD);
            btnMapSR.put((reb << 4) | RebPS.PS_CLK_HIGH, (reb << 4) | PS_CLKH);
            btnMapSR.put((reb << 4) | RebPS.PS_CLK_LOW, (reb << 4) | PS_CLKL);
            btnMapSR.put((reb << 4) | RebPS.PS_HEATER, (reb << 4) | PS_HTR);
            btnMapSR.put((reb << 4) | RebPS.PS_HV_BIAS, (reb << 4) | PS_HV);
        }
    }
    private static final Map<Integer, Integer> invBtnMapSR = new HashMap<>();
    static {
        for (int ps : btnMapSR.keySet()) {
            invBtnMapSR.put(btnMapSR.get(ps), ps);
        }
    }
    private static final Map<Integer, Integer> btnMapCR = new HashMap<>();
    static {
        for (int reb = 0; reb < N_REB; reb++) {
            if ((reb % REB_Q) != REB_Q - 1) {
                btnMapCR.put((reb << 4) | RebPS.PS_DIGITAL, (reb << 4) | PS_DIG);
            }
            else {
                btnMapCR.put((reb << 4) | RebPS.PS_DIGITAL, ((reb - 2) << 4) | PS_HTR);
            }
            btnMapCR.put((reb << 4) | RebPS.PS_ANALOG, (reb << 4) | PS_ANA);
            btnMapCR.put((reb << 4) | RebPS.PS_OD, (reb << 4) | PS_OD);
            btnMapCR.put((reb << 4) | RebPS.PS_CLK_HIGH, (reb << 4) | PS_CLKH);
            btnMapCR.put((reb << 4) | RebPS.PS_CLK_LOW, (reb << 4) | PS_CLKL);
            btnMapCR.put((reb << 4) | RebPS.PS_DPHI, (reb << 4) | PS_DPHI);
            btnMapCR.put((reb << 4) | RebPS.PS_HV_BIAS, (reb << 4) | PS_HV);
        }
    }
    private static final Map<Integer, Integer> invBtnMapCR = new HashMap<>();
    static {
        for (int ps : btnMapCR.keySet()) {
            invBtnMapCR.put(btnMapCR.get(ps), ps);
        }
    }
    private static final String SOLID_BAR = "\u2588\u2588";

    private final CommandSender sender;
    private final UiUtilities uiUtils;
    private final JPanel pnlSystemState = new JPanel();
    private JLabel lblPowerStateValue;
    private JLabel lblPsIdValue;
    private JButton btnMainPower;
    private TextFieldX tfPowerPeriod;
    private final JPanel pnlPowerControl = new JPanel();
    private final JComponent[][] cmpPsState = new JComponent[N_REB][N_PS];
    private final JButton[] btnSeqOn = new JButton[N_REB];
    private final JButton[] btnSeqOff = new JButton[N_REB];
    private final TextFieldX[] tfBiasDac = new TextFieldX[N_REB];
    private final TextFieldX[] tfDphiDac = new TextFieldX[N_REB];
    private int numReb = 0;
    private boolean gotCR = false; //, gotProto = false;
    //private int[] psNums = psNumsSR;
    //private boolean[] isDac = isDacSR;

    public RebPsControlPanel(String agent) {
        sender = new CommandSender(agent, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        //(new UpdatePowerStatus(new RebPsState(0, PowerChanState.PWR_STATE_OFFLINE, new int[]{},
        //                                      new double[]{}, new double[]{}, 0, 0, ""))).run();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, "getFullState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((RebPsState)reply);
    }

    public void updatePanel(RebPsState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {
        //** Create all low-level components

        // Status line
        lblPowerStateValue = UiUtilities.newLabel("X", UiUtilities.maxLabelWidth(new String[]{"STOPPED", "RUNNING"}, ""));
        lblPsIdValue = UiUtilities.newLabel("MM-9999", 0);
        btnMainPower = uiUtils.newButton(SOLID_BAR, "M", true);
        tfPowerPeriod = uiUtils.newTextFieldX("00000.0", "P", TextFieldX.TYPE_DOUBLE, true);

        // Power control buttons and fields
        for (int reb = 0; reb < N_REB; reb++) {
            btnSeqOn[reb] = uiUtils.newButton("On", "N" + reb, true);
            btnSeqOn[reb].setForeground(UiConstants.GREEN);
            btnSeqOff[reb] = uiUtils.newButton("Off", "F" + reb, true);
            btnSeqOff[reb].setForeground(UiConstants.RED);
            for (int ps = 0; ps < N_PS; ps++) {
                if (ps == PS_HV || ps == PS_DPHI) {
                    cmpPsState[reb][ps] = uiUtils.newButton(SOLID_BAR, "R" + ((reb << 4) | ps), true);
                }
                else {
                    cmpPsState[reb][ps] = UiUtilities.newLabel(SOLID_BAR, 0);
                }
            }
            tfDphiDac[reb] = uiUtils.newTextFieldX("000000", "D" + reb, TextFieldX.TYPE_INT, true);
            tfBiasDac[reb] = uiUtils.newTextFieldX("000000", "B" + reb, TextFieldX.TYPE_INT, true);
        }

        //** Construct panels

        // Status line
        drawStatus();

        // Power supply control
        pnlPowerControl.setLayout(new GridBagLayout());
        drawControl(N_REB, false);

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 10, 6, 10);
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(pnlSystemState, gbc);
        gbc.gridy++;
        add(pnlPowerControl, gbc);
    }

    private void drawStatus() {
        pnlSystemState.setLayout(new GridBagLayout());
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.WEST;
        gbs.insets = new Insets(5, 0, 5, 0);
        gbs.gridx = 0;
        gbs.gridy = 0;
        pnlSystemState.add(UiUtilities.newLabel("System State:", 0), gbs);
        gbs.gridx++;
        gbs.insets.left = 5;
        pnlSystemState.add(lblPowerStateValue, gbs);
        gbs.gridx++;
        gbs.insets.left = 30;
        pnlSystemState.add(UiUtilities.newLabel("PS ID: ", 0), gbs);
        gbs.gridx++;
        gbs.insets.left = 5;
        pnlSystemState.add(lblPsIdValue, gbs);
        gbs.gridx++;
        gbs.insets.left = 30;
        pnlSystemState.add(UiUtilities.newLabel("Main Power:", 0), gbs);
        gbs.gridx++;
        gbs.insets.left = 5;
        pnlSystemState.add(btnMainPower, gbs);
        gbs.gridx++;
        gbs.insets.left = 30;
        pnlSystemState.add(UiUtilities.newLabel("Update Period:", 0), gbs);
        gbs.gridx++;
        gbs.insets.left = 5;
        pnlSystemState.add(tfPowerPeriod, gbs);
        gbs.gridx++;
        pnlSystemState.add(UiUtilities.newLabel("sec", 0), gbs);
    }

    private void drawControl(int nReb, boolean isCR) {
        if (nReb == 0 || nReb == numReb && isCR == gotCR) return;

        numReb = nReb;
        gotCR = isCR;
        pnlPowerControl.removeAll();
        
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.insets = new Insets(0, 6, 4, 6);

        gbp.gridx = 1;
        gbp.gridwidth = 2;
        gbp.gridy = 0;
        pnlPowerControl.add(UiUtilities.newLabel("Sequenced", 0), gbp);
        gbp.gridx += 2;
        gbp.gridwidth = 1;
        for (int ps = 0; ps < N_PS; ps++) {
            if (gotCR || ps != PS_DPHI) {
                pnlPowerControl.add(UiUtilities.newLabel(psNames[ps], 0), gbp);
                gbp.gridx++;
            }
            if (gotCR && ps == PS_DPHI) {
                pnlPowerControl.add(UiUtilities.newLabel("Dphi DAC", 0), gbp);
                gbp.gridx++;
            }
            if (ps == PS_HV) {
                pnlPowerControl.add(UiUtilities.newLabel("Bias DAC", 0), gbp);
                gbp.gridx++;
            }
        }

        gbp.gridx = 0;
        gbp.gridy++;
        for (int reb = 0; reb < numReb; reb++) {
            if (gotCR && (reb % REB_Q) == REB_Q - 1) continue;
            pnlPowerControl.add(UiUtilities.newLabel("REB" + reb + ":", 0), gbp);
            gbp.gridx++;
            pnlPowerControl.add(btnSeqOn[reb], gbp);
            gbp.gridx++;
            pnlPowerControl.add(btnSeqOff[reb], gbp);
            gbp.gridx++;
            for (int ps = 0; ps < N_PS; ps++) {
                if (gotCR || ps != PS_DPHI) {
                    if (ps != PS_HTR || !gotCR || (reb % REB_Q) == 0) {
                        pnlPowerControl.add(cmpPsState[reb][ps], gbp);
                    }
                    gbp.gridx++;
                }
                if (gotCR && ps == PS_DPHI) {
                    pnlPowerControl.add(tfDphiDac[reb], gbp);
                    gbp.gridx++;
                }
                if (ps == PS_HV) {
                    pnlPowerControl.add(tfBiasDac[reb], gbp);
                    gbp.gridx++;
                }
            }
            gbp.gridx = 0;
            gbp.gridy++;
        }
    }

    @Override
    public void handleButton(String name) {
        char type = name.charAt(0);
        if (type == 'M') {
            sender.sendCommand(null, "toggleMainPower");
        }
        else if (type == 'R') {
            int srcPsReb = Integer.valueOf(name.substring(1));
            int dstPsReb = (gotCR ? invBtnMapCR : invBtnMapSR).get(srcPsReb);
            sender.sendCommand(null, "togglePower", dstPsReb >> 4, dstPsReb & 0x0f);
        }
        else {
            int reb = Integer.valueOf(name.substring(1, 2));
            sender.sendCommand(null, "sequencePower", reb, type == 'N');
        }
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        char type = name.charAt(0);
        if (type == 'P') {
            sender.sendCommand(null, "setUpdatePeriod", (int)(1000 * (Double)value));
        }
        else {
            int reb = Integer.valueOf(name.substring(1));
            sender.sendCommand(null, type == 'B' ? "setBiasDac" : "setDphiDac", reb, (Integer)value);
        }
    }

    class UpdatePowerStatus implements Runnable {

        private final RebPsState rs;

        UpdatePowerStatus(RebPsState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            int nReb = rs.getPowerState().length;
            //gotProto = rs.getPsVersion() == RebPs.VERSION_PROTO;
            boolean isCR = rs.getPsType() == RebPS.TYPE_CORNER;
            //psNums = isCR ? psNumsCR : psNumsSR;
            //isDac = isCR ? isDacCR : isDacSR;
            drawControl(nReb, isCR);
            lblPowerStateValue.setText("RUNNING");
            lblPowerStateValue.setForeground(UiConstants.GREEN);

            lblPsIdValue.setText(rs.getPsId());

            int mps = rs.getMainPowerState();
            Color color = mps == PowerChanState.PWR_STATE_ON ? UiConstants.GREEN :
                          mps == PowerChanState.PWR_STATE_OFF ? UiConstants.RED : Color.GRAY;
            btnMainPower.setEnabled(color != Color.GRAY);
            btnMainPower.setForeground(color);

            tfPowerPeriod.update(rs.getTickMillis() / 1000.0, true);

            Map<Integer, Integer> btnMap = isCR ? btnMapCR : btnMapSR;
            int[] state = rs.getPowerState();
            double[] bDacs = rs.getHvBiasDacs();
            double[] dDacs = rs.getDphiDacs();
            for (int reb = 0; reb < nReb; reb++) {
                boolean present = state[reb] >= 0;
                btnSeqOn[reb].setEnabled(present);
                btnSeqOff[reb].setEnabled(present);
                for (int ps = 0; ps < RebPS.NUM_PS; ps++) {
                    color = present ? ((state[reb] & (1 << (ps + 1))) == 0) ?
                              UiConstants.RED : UiConstants.GREEN : Color.GRAY;
                    int btnPsReb = btnMap.get((reb << 4) | ps);
                    cmpPsState[btnPsReb >> 4][btnPsReb & 0x0f].setEnabled(present);
                    cmpPsState[btnPsReb >> 4][btnPsReb & 0x0f].setForeground(color);
                }
                int value = (reb < bDacs.length) ? (int)bDacs[reb] : 0;
                tfBiasDac[reb].update(value, present);
                value = (reb < dDacs.length) ? (int)dDacs[reb] : 0;
                tfDphiDac[reb].update(value, present);
            }
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            lblPowerStateValue.setText("STOPPED");
            lblPowerStateValue.setForeground(UiConstants.RED);
            btnMainPower.setEnabled(false);
            btnMainPower.setForeground(Color.GRAY);
            tfPowerPeriod.setDisabled();
            for (int reb = 0; reb < numReb; reb++) {
                btnSeqOn[reb].setEnabled(false);
                btnSeqOff[reb].setEnabled(false);
                for (int ps = 0; ps < N_PS; ps++) {
                    cmpPsState[reb][ps].setEnabled(false);
                    cmpPsState[reb][ps].setForeground(Color.GRAY);
                }
                tfBiasDac[reb].setDisabled();
                tfDphiDac[reb].setDisabled();
            }
        } 
    }

    private static final long serialVersionUID = 1L;
}
