package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import java.util.HashMap;
import java.util.Map;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.power.constants.ATSChannels;
import org.lsst.ccs.subsystem.power.data.ATSPowerState;

/**
 *  Implements the REB power supply control & monitoring panel.
 *
 *  @author Owen Saxton
 */
public class ATSPowerControlPanel extends javax.swing.JPanel {

    private static final Map<Integer, String> chanNames = new HashMap<>();
    static {
        chanNames.put(ATSChannels.CHAN_DIGITAL, "Dig");
        chanNames.put(ATSChannels.CHAN_ANALOG, "Ana");
        chanNames.put(ATSChannels.CHAN_CLK_HIGH, "ClkH");
        chanNames.put(ATSChannels.CHAN_CLK_LOW, "ClkL");
        chanNames.put(ATSChannels.CHAN_OD, "OD");
        chanNames.put(ATSChannels.CHAN_DPHI, "DPHI");
        chanNames.put(ATSChannels.CHAN_HV_BIAS, "HVbias");
    }
    private static final String SOLID_BAR = "\u2588\u2588\u2588\u2588";

    private final CommandSender sender;
    private SystemStatusPanel sysStatusPnl;
    private JButton btnPowerOn;
    private JButton btnPowerOff;
    private JButton btnDphiOn;
    private JButton btnDphiOff;
    private JTextField tfDphiVal = new JTextField();
    private JButton btnBiasOn;
    private JButton btnBiasOff;
    private JTextField tfBiasVal = new JTextField();
    private final JLabel[] lblChanStatus = new JLabel[ATSChannels.NUM_REB_CHANNELS];
    private JButton btnOtmOn;
    private JButton btnOtmOff;
    private JButton btnFanOn;
    private JButton btnFanOff;
    private JButton btnAuxOn;
    private JButton btnAuxOff;
    private JLabel lblOtmStatus;
    private JLabel lblFanStatus;
    private JLabel lblAuxStatus;
    private double hvBias, dphi;

    public ATSPowerControlPanel(CommandSender sender) {
        this.sender = sender;
        initComponents();
        (new UpdatePowerStatus(new ATSPowerState(0, new Boolean[ATSChannels.NUM_CHANNELS], 0.0, 0.0))).run();
        (new DisableSystem()).run();
    }

    private void initComponents()
    {
        //** Create all low-level components

        // REB power control buttons and fields
        btnPowerOn = newButton("On", UiConstants.GREEN, "powerOn");
        btnPowerOff = newButton("Off", UiConstants.RED, "powerOff");
        btnDphiOn = newButton("On", UiConstants.GREEN, "dphiOn");
        btnDphiOff = newButton("Off", UiConstants.RED, "dphiOff");
        btnBiasOn = newButton("On", UiConstants.GREEN, "hvBiasOn");
        btnBiasOff = newButton("Off", UiConstants.RED, "hvBiasOff");
        btnOtmOn = newButton("On", UiConstants.GREEN, "otmOn");
        btnOtmOff = newButton("Off", UiConstants.RED, "otmOff");
        btnFanOn = newButton("On", UiConstants.GREEN, "fanOn");
        btnFanOff = newButton("Off", UiConstants.RED, "fanOff");
        btnAuxOn = newButton("On", UiConstants.GREEN, "auxOn");
        btnAuxOff = newButton("Off", UiConstants.RED, "auxOff");

        JTextField tf = new JTextField("...");
        Dimension dd = tf.getPreferredSize();
        dd.width = 60;
        tf.setPreferredSize(dd);
        tf.setHorizontalAlignment(JTextField.CENTER);
        tf.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setDphiValue();
            }
        });
        tfDphiVal = tf;

        tf = new JTextField("...");
        tf.setPreferredSize(dd);
        tf.setHorizontalAlignment(JTextField.CENTER);
        tf.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setBiasValue();
            }
        });
        tfBiasVal = tf;

        // Channel status labels
        for (int cs = 0; cs < lblChanStatus.length; cs++) {
            JLabel lbl = new JLabel(SOLID_BAR);  // Blocks
            lbl.setFont(UiConstants.FONT);
            lbl.setFocusable(false);
            lblChanStatus[cs] = lbl;
        }

        JLabel lbl = new JLabel(SOLID_BAR);
        lbl.setFont(UiConstants.FONT);
        lbl.setFocusable(false);
        lblOtmStatus = lbl;

        lbl = new JLabel(SOLID_BAR);
        lbl.setFont(UiConstants.FONT);
        lbl.setFocusable(false);
        lblFanStatus = lbl;

        lbl = new JLabel(SOLID_BAR);
        lbl.setFont(UiConstants.FONT);
        lbl.setFocusable(false);
        lblAuxStatus = lbl;

        //** Construct panels

        // State indicator
        sysStatusPnl = new SystemStatusPanel(sender);

        // Power supply control
        JPanel pnlPowerControl = new JPanel();
        pnlPowerControl.setLayout(new GridBagLayout());
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.insets = new Insets(4, 6, 4, 6);

        gbp.gridx = 0;
        gbp.gridy = 0;
        gbp.gridwidth = 2;
        lbl = new JLabel("Seq. Power");
        lbl.setFont(UiConstants.FONT);
        pnlPowerControl.add(lbl, gbp);
        gbp.gridy = 1;
        gbp.gridwidth = 1;
        gbp.insets.right = 2;
        pnlPowerControl.add(btnPowerOn, gbp);
        gbp.gridx++;
        gbp.insets.left = 2;
        gbp.insets.right = 6;
        pnlPowerControl.add(btnPowerOff, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        gbp.gridwidth = 2;
        gbp.insets.left = 6;
        lbl = new JLabel("DPHI");
        lbl.setFont(UiConstants.FONT);
        pnlPowerControl.add(lbl, gbp);
        gbp.gridy = 1;
        gbp.gridwidth = 1;
        gbp.insets.right = 2;
        pnlPowerControl.add(btnDphiOn, gbp);
        gbp.gridx++;
        gbp.insets.left = 2;
        gbp.insets.right = 6;
        pnlPowerControl.add(btnDphiOff, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        lbl = new JLabel("DPHI Volts");
        lbl.setFont(UiConstants.FONT);
        pnlPowerControl.add(lbl, gbp);
        gbp.gridy = 1;
        pnlPowerControl.add(tfDphiVal, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        gbp.gridwidth = 2;
        gbp.insets.left = 6;
        lbl = new JLabel("HV Bias");
        lbl.setFont(UiConstants.FONT);
        pnlPowerControl.add(lbl, gbp);
        gbp.gridy = 1;
        gbp.gridwidth = 1;
        gbp.insets.right = 2;
        pnlPowerControl.add(btnBiasOn, gbp);
        gbp.gridx++;
        gbp.insets.left = 2;
        gbp.insets.right = 6;
        pnlPowerControl.add(btnBiasOff, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        gbp.insets.left = 6;
        lbl = new JLabel("Bias Volts");
        lbl.setFont(UiConstants.FONT);
        pnlPowerControl.add(lbl, gbp);
        gbp.gridy = 1;
        pnlPowerControl.add(tfBiasVal, gbp);

        // Channel status
        JPanel pnlChanStatus = new JPanel();
        pnlChanStatus.setLayout(new GridBagLayout());
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.insets = new Insets(2, 6, 2, 6);
        gbs.gridx = 0;
        for (int j = 0; j < lblChanStatus.length; j++) {
            gbs.gridy = 0;
            lbl = new JLabel(chanNames.get(j));
            lbl.setFont(UiConstants.FONT);
            pnlChanStatus.add(lbl, gbs);
            gbs.gridy = 1;
            pnlChanStatus.add(lblChanStatus[j], gbs);
            gbs.gridx++;
        }

        // REB power control
        JPanel pnlREBPower = new JPanel();
        TitledBorder border = new TitledBorder(new LineBorder(Color.BLACK), "REB Power");
        border.setTitleColor(UiConstants.BLUE);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleJustification(TitledBorder.CENTER);
        pnlREBPower.setBorder(border);
        pnlREBPower.setLayout(new GridBagLayout());
        GridBagConstraints gbrp = new GridBagConstraints();
        gbrp.insets = new Insets(0, 4, 4, 4);
        gbrp.gridx = 0;
        gbrp.gridy = 0;
        pnlREBPower.add(pnlPowerControl, gbrp);
        gbrp.gridy++;
        pnlREBPower.add(pnlChanStatus, gbrp);

        // Ancillary power control
        JPanel pnlAncPower = new JPanel();
        border = new TitledBorder(new LineBorder(Color.BLACK), "Ancillary Power");
        border.setTitleColor(UiConstants.BLUE);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleJustification(TitledBorder.CENTER);
        pnlAncPower.setBorder(border);
        pnlAncPower.setLayout(new GridBagLayout());
        GridBagConstraints gbap = new GridBagConstraints();
        gbap.insets = new Insets(0, 6, 4, 6);

        gbap.gridx = 0;
        gbap.gridy = 0;
        gbap.gridwidth = 2;
        lbl = new JLabel("OTM");
        lbl.setFont(UiConstants.FONT);
        pnlAncPower.add(lbl, gbap);
        gbap.gridy++;
        gbap.gridwidth = 1;
        gbap.insets.right = 2;
        pnlAncPower.add(btnOtmOn, gbap);
        gbap.gridx++;
        gbap.insets.left = 2;
        gbap.insets.right = 6;
        pnlAncPower.add(btnOtmOff, gbap);
        gbap.gridx--;
        gbap.gridy++;
        gbap.gridwidth = 2;
        gbap.insets.left = 6;
        pnlAncPower.add(lblOtmStatus, gbap);

        gbap.gridx = 2;
        gbap.gridy = 0;
        lbl = new JLabel("Fan");
        lbl.setFont(UiConstants.FONT);
        pnlAncPower.add(lbl, gbap);
        gbap.gridy++;
        gbap.gridwidth = 1;
        gbap.insets.right = 2;
        pnlAncPower.add(btnFanOn, gbap);
        gbap.gridx++;
        gbap.insets.left = 2;
        gbap.insets.right = 6;
        pnlAncPower.add(btnFanOff, gbap);
        gbap.gridx--;
        gbap.gridy++;
        gbap.gridwidth = 2;
        gbap.insets.left = 6;
        pnlAncPower.add(lblFanStatus, gbap);

        gbap.gridx = 4;
        gbap.gridy = 0;
        gbap.gridwidth = 2;
        lbl = new JLabel("Aux");
        lbl.setFont(UiConstants.FONT);
        pnlAncPower.add(lbl, gbap);
        gbap.gridy++;
        gbap.gridwidth = 1;
        gbap.insets.right = 2;
        pnlAncPower.add(btnAuxOn, gbap);
        gbap.gridx++;
        gbap.insets.left = 2;
        gbap.insets.right = 6;
        pnlAncPower.add(btnAuxOff, gbap);
        gbap.gridx--;
        gbap.gridy++;
        gbap.gridwidth = 2;
        gbap.insets.left = 6;
        pnlAncPower.add(lblAuxStatus, gbap);

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(4, 0, 4, 0);
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(sysStatusPnl, gbc);
        gbc.gridy++;
        add(pnlREBPower, gbc);
        gbc.gridy++;
        add(pnlAncPower, gbc);
    }

    private JButton newButton(String text, Color color, String command) {
        JButton btn = new JButton(text);
        btn.setFont(UiConstants.FONT);
        btn.setForeground(color);
        btn.setFocusable(false);
        btn.setActionCommand(command);
        btn.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                sender.sendCommand(null, ((JButton)evt.getSource()).getActionCommand());
            }
        });
        return btn;
    }

    private void setBiasValue() {
        try {
            double value = Double.valueOf(tfBiasVal.getText());
            sender.sendCommand(null, "setHvBias", value);
            //tfBiasVal.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfBiasVal.setText(String.format("%.1f", hvBias));
        }
    }

    private void setDphiValue() {
        try {
            double value = Double.valueOf(tfDphiVal.getText());
            sender.sendCommand(null, "setDphi", value);
            //tfDphiVal.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfDphiVal.setText(String.format("%.1f", dphi));
        }
    }

    public void updateControlPanel(ATSPowerState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdatePowerStatus implements Runnable {

        private final ATSPowerState rs;

        UpdatePowerStatus(ATSPowerState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            sysStatusPnl.updateStatus(rs.getTickMillis());

            btnPowerOn.setEnabled(true);
            btnPowerOff.setEnabled(true);
            btnDphiOn.setEnabled(true);
            btnDphiOff.setEnabled(true);
            btnBiasOn.setEnabled(true);
            btnBiasOff.setEnabled(true);

            hvBias = rs.getHvBias();
            tfBiasVal.setText(String.valueOf(hvBias));
            tfBiasVal.setEnabled(true);

            dphi = rs.getDphi();
            tfDphiVal.setText(String.valueOf(dphi));
            tfDphiVal.setEnabled(true);

            Boolean[] state = rs.getPowerState();
            for (int j = 0; j < lblChanStatus.length; j++) {
                lblChanStatus[j].setForeground(state[j] == null ? UiConstants.BLUE :
                                               state[j] ? UiConstants.GREEN : UiConstants.RED);
                lblChanStatus[j].setEnabled(true);
            }
            
            btnOtmOn.setEnabled(true);
            btnOtmOff.setEnabled(true);
            btnFanOn.setEnabled(true);
            btnFanOff.setEnabled(true);
            btnAuxOn.setEnabled(true);
            btnAuxOff.setEnabled(true);
            lblOtmStatus.setForeground(state[ATSChannels.CHAN_OTM] == null ? UiConstants.BLUE :
                                       state[ATSChannels.CHAN_OTM] ? UiConstants.GREEN : UiConstants.RED);
            lblOtmStatus.setEnabled(true);
            lblFanStatus.setForeground(state[ATSChannels.CHAN_FAN] == null ? UiConstants.BLUE :
                                       state[ATSChannels.CHAN_FAN] ? UiConstants.GREEN : UiConstants.RED);
            lblFanStatus.setEnabled(true);
            lblAuxStatus.setForeground(state[ATSChannels.CHAN_AUX] == null ? UiConstants.BLUE :
                                       state[ATSChannels.CHAN_AUX] ? UiConstants.GREEN : UiConstants.RED);
            lblAuxStatus.setEnabled(true);
            repaint();
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            sysStatusPnl.disableSystem();
            btnPowerOn.setEnabled(false);
            btnPowerOff.setEnabled(false);
            btnDphiOn.setEnabled(false);
            btnDphiOff.setEnabled(false);
            btnBiasOn.setEnabled(false);
            btnBiasOff.setEnabled(false);
            tfBiasVal.setEnabled(false);
            tfDphiVal.setEnabled(false);
            for (JLabel lbl : lblChanStatus) {
                lbl.setEnabled(false);
            }
            btnOtmOn.setEnabled(false);
            btnOtmOff.setEnabled(false);
            btnFanOn.setEnabled(false);
            btnFanOff.setEnabled(false);
            btnAuxOn.setEnabled(false);
            btnAuxOff.setEnabled(false);
            lblOtmStatus.setEnabled(false);
            lblFanStatus.setEnabled(false);
            lblAuxStatus.setEnabled(false);
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
