package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.power.constants.QuadBoxSwitches;
import org.lsst.ccs.subsystem.power.data.QuadBoxState;
import org.lsst.ccs.subsystem.power.constants.SwitchState;

/**
 *  Implements the REB power supply control & monitoring panel.
 *
 *  @author Owen Saxton
 */
public class QuadBoxControlPanel extends JPanel {

    private static final String[] panelNames = new String[QuadBoxSwitches.NUM_DEVICES];
    static {
        panelNames[QuadBoxSwitches.DEVC_BFR] = "BFR";
        panelNames[QuadBoxSwitches.DEVC_PDU_5V] = "5V Clean PDU";
        panelNames[QuadBoxSwitches.DEVC_PDU_24VC] = "24V Clean PDU";
        panelNames[QuadBoxSwitches.DEVC_PDU_24VD] = "24V Dirty PDU";
        panelNames[QuadBoxSwitches.DEVC_PDU_48V] = "48V Dirty PDU";
        panelNames[QuadBoxSwitches.DEVC_REB_BULK] = "REB Bulk PS";
    }
    private static final Map<Integer, String> switchNames = new LinkedHashMap<>();
    static {
        switchNames.put(QuadBoxSwitches.SW_PDU_5V, "5V Clean PDU");
        switchNames.put(QuadBoxSwitches.SW_PDU_24VC, "24V Clean PDU");
        switchNames.put(QuadBoxSwitches.SW_PDU_24VD, "24V Dirty PDU");
        switchNames.put(QuadBoxSwitches.SW_PDU_48V, "48V Dirty PDU");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_PS_0, "REB PS 0-2");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_PS_1, "REB PS 3-5");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_PS_2, "REB PS 6-8");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_PS_3, "REB PS 9-12");
        switchNames.put(QuadBoxSwitches.SW_TRIM_HEATERS, "Trim Heaters");
        switchNames.put(QuadBoxSwitches.SW_HEATER_1, "Heater 1");
        switchNames.put(QuadBoxSwitches.SW_HEATER_2, "Heater 2");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_PS_4, "REB PS spare");
        switchNames.put(QuadBoxSwitches.SW_OTM_0_A, "OTM 0-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_0_B, "OTM 0-B");
        switchNames.put(QuadBoxSwitches.SW_OTM_1_A, "OTM 1-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_1_B, "OTM 1-B");
        switchNames.put(QuadBoxSwitches.SW_OTM_2_A, "OTM 2-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_2_B, "OTM 2-B");
        switchNames.put(QuadBoxSwitches.SW_OTM_3_A, "OTM 3-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_3_B, "OTM 3-B");
        switchNames.put(QuadBoxSwitches.SW_OTM_4_A, "OTM 4-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_4_B, "OTM 4-B");
        switchNames.put(QuadBoxSwitches.SW_OTM_5_A, "OTM 5-A");
        switchNames.put(QuadBoxSwitches.SW_OTM_5_B, "OTM 5-B");
        switchNames.put(QuadBoxSwitches.SW_MAIN_24C, "Main");
        switchNames.put(QuadBoxSwitches.SW_HCU_PWR_CRY, "Power/Cryo HCUs");
        switchNames.put(QuadBoxSwitches.SW_HCU_FES_SHU, "FES/Shutter HCUs");
        switchNames.put(QuadBoxSwitches.SW_ION_PUMPS, "Ion pumps");
        switchNames.put(QuadBoxSwitches.SW_BODY_PURGE, "Body purge");
        switchNames.put(QuadBoxSwitches.SW_BPU_MAQ20, "BPU MAQ20");
        switchNames.put(QuadBoxSwitches.SW_GAUGES, "Gauges");
        //switchNames.put(QuadBoxSwitches.SW_UNUSED_24C, "Unused");
        switchNames.put(QuadBoxSwitches.SW_FES_CAROUSEL, "FES carousel");
        switchNames.put(QuadBoxSwitches.SW_FES_CHANGER_C, "FES changer");
        switchNames.put(QuadBoxSwitches.SW_FES_LOADER_C, "FES loader");
        switchNames.put(QuadBoxSwitches.SW_SHUTTER_1, "Shutter 1");
        switchNames.put(QuadBoxSwitches.SW_SHUTTER_2, "Shutter 2");
        switchNames.put(QuadBoxSwitches.SW_MAIN_24D, "Main");
        switchNames.put(QuadBoxSwitches.SW_CRYO_TURBO, "Cryo turbo");
        switchNames.put(QuadBoxSwitches.SW_HEX_TURBO, "Hex turbo");
        switchNames.put(QuadBoxSwitches.SW_FES_CLAMPS, "FES clamps");
        switchNames.put(QuadBoxSwitches.SW_FES_BRAKES, "FES brake");
        switchNames.put(QuadBoxSwitches.SW_FES_CHANGER_D, "FES changer");
        switchNames.put(QuadBoxSwitches.SW_FES_LOADER_D, "FES loader");
        switchNames.put(QuadBoxSwitches.SW_SHUTTER_BRAKE, "Shutter brake");
        switchNames.put(QuadBoxSwitches.SW_MAIN_48, "Main");
        switchNames.put(QuadBoxSwitches.SW_PURGE_FAN, "Purge fan");
        switchNames.put(QuadBoxSwitches.SW_FES_CAROUSEL_D, "FES carousel");
        switchNames.put(QuadBoxSwitches.SW_FES_HEATER, "FES heater"); 
        switchNames.put(QuadBoxSwitches.SW_SHUTTER_1_D, "Shutter 1");
        switchNames.put(QuadBoxSwitches.SW_SHUTTER_2_D , "Shutter 2");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_0, "REB PS 0-2");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_1, "REB PS 3-5");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_2, "REB PS 6-8");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_3, "REB PS 9-12");
        switchNames.put(QuadBoxSwitches.SW_REB_BULK_4, "REB PS Spare");
    }
    private static final int[] nameWidths = new int[QuadBoxSwitches.NUM_DEVICES];
    static {
        nameWidths[QuadBoxSwitches.DEVC_BFR] = 110;
        nameWidths[QuadBoxSwitches.DEVC_PDU_5V] = 70;
        nameWidths[QuadBoxSwitches.DEVC_PDU_24VC] = 130;
        nameWidths[QuadBoxSwitches.DEVC_PDU_24VD] = 105;
        nameWidths[QuadBoxSwitches.DEVC_PDU_48V] = 105;
        nameWidths[QuadBoxSwitches.DEVC_REB_BULK] = 110;
    }
    private static final Map<Integer, Integer> bfrSwitchMap = new HashMap<>();
    static {
        bfrSwitchMap.put(QuadBoxSwitches.DEVC_PDU_5V, QuadBoxSwitches.SW_PDU_5V);
        bfrSwitchMap.put(QuadBoxSwitches.DEVC_PDU_24VC, QuadBoxSwitches.SW_PDU_24VC);
        bfrSwitchMap.put(QuadBoxSwitches.DEVC_PDU_24VD, QuadBoxSwitches.SW_PDU_24VD);
        bfrSwitchMap.put(QuadBoxSwitches.DEVC_PDU_48V, QuadBoxSwitches.SW_PDU_48V);
    }
    private static final Map<Integer, Integer> mainSwitchMap = new HashMap<>();
    static {
        mainSwitchMap.put(QuadBoxSwitches.DEVC_PDU_24VC, QuadBoxSwitches.SW_MAIN_24C);
        mainSwitchMap.put(QuadBoxSwitches.DEVC_PDU_24VD, QuadBoxSwitches.SW_MAIN_24D);
        mainSwitchMap.put(QuadBoxSwitches.DEVC_PDU_48V, QuadBoxSwitches.SW_MAIN_48);
    }

    private final CommandSender sender;
    private final JPanel topPanel = new JPanel();
    private final JPanel statePanel = new JPanel();
    private final JLabel stateLabel = new JLabel("System State: ");
    private final JLabel stateStatus = new JLabel("....");
    private final JPanel periodPanel = new JPanel();
    private final JLabel periodLabel = new JLabel("Update Period: ");
    private final JTextField periodTextField = new JTextField("...");
    private final JLabel periodUnitsLabel = new JLabel("sec");
    private final JPanel[] switchPanel = new JPanel[QuadBoxSwitches.NUM_DEVICES];
    private final JLabel[][] switchLabel = new JLabel[QuadBoxSwitches.NUM_DEVICES][];
    private final JLabel[][] switchStatus = new JLabel[QuadBoxSwitches.NUM_DEVICES][];
    private final ButtonGroup[][] switchBG = new ButtonGroup[QuadBoxSwitches.NUM_DEVICES][];
    private final JRadioButton[][] switchOffRB = new JRadioButton[QuadBoxSwitches.NUM_DEVICES][];
    private final JRadioButton[][] switchOnRB = new JRadioButton[QuadBoxSwitches.NUM_DEVICES][];
    private final Map<Integer, Integer> switchLines = new HashMap<>();
    private double powerPeriod = 5;

    public QuadBoxControlPanel(CommandSender sender) {
        this.sender = sender;
        initComponents();
        (new UpdatePowerStatus(new QuadBoxState())).run();
        (new DisableSystem()).run();
    }

    private void initComponents()
    {
        //** Create all low-level components

        // State indicator
        Dimension d = stateStatus.getPreferredSize();
        d.width = 70;
        stateStatus.setPreferredSize(d);
        stateStatus.setMinimumSize(d);

        // Update period
        d = periodTextField.getPreferredSize();
        d.width = 50;
        periodTextField.setPreferredSize(d);
        periodTextField.setMinimumSize(d);
        periodTextField.setHorizontalAlignment(JTextField.CENTER);
        periodTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setUpdatePeriod();
            }
        });

        // Switch panel elements
        for (int dvc = 0; dvc < QuadBoxSwitches.NUM_DEVICES; dvc++) {
            int nSwitch = QuadBoxSwitches.NUM_SWITCHES[dvc];
            switchLabel[dvc] = new JLabel[nSwitch + 1];
            switchStatus[dvc] = new JLabel[nSwitch + 1];
            switchBG[dvc] = new ButtonGroup[nSwitch + 1];
            switchOnRB[dvc] = new JRadioButton[nSwitch + 1];
            switchOffRB[dvc] = new JRadioButton[nSwitch + 1];
        }
        int[] lines = new int[QuadBoxSwitches.NUM_DEVICES];
        for (int swId : switchNames.keySet()) {
            int dvc = swId >> 8;
            int line = lines[dvc]++;
            switchLines.put(swId, line);
            //int sw = (byte)swId;   // Sign extend the byte value
            JLabel label = switchLabel[dvc][line] = new JLabel(switchNames.get(swId));
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = nameWidths[dvc];
            label.setPreferredSize(d);
            label.setMinimumSize(d);
            JLabel status = switchStatus[dvc][line] = new JLabel("XXX");
            status.setFont(UiConstants.FONT);
            d = status.getPreferredSize();
            d.width = 64;
            status.setPreferredSize(d);
            status.setMinimumSize(d);
            JRadioButton on = switchOnRB[dvc][line] = new JRadioButton("On");
            on.setFont(UiConstants.FONT);
            on.setFocusable(false);
            on.setName(Integer.toString(swId));
            on.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), true);
                }
            });
            JRadioButton off = switchOffRB[dvc][line] = new JRadioButton("Off");
            off.setFont(UiConstants.FONT);
            off.setFocusable(false);
            off.setName(Integer.toString(swId));
            off.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), false);
                }
            });
            ButtonGroup bg = switchBG[dvc][line] = new ButtonGroup();
            bg.add(on);
            bg.add(off);
        }

        //** Construct panels

        // Power state
        statePanel.add(stateLabel);
        statePanel.add(stateStatus);

        // Update period
        periodPanel.add(periodLabel);
        periodPanel.add(periodTextField);
        periodPanel.add(periodUnitsLabel);

        // Top panel
        GridBagConstraints gbt = new GridBagConstraints();
        topPanel.setLayout(new GridBagLayout());
        gbt.gridx = 0;
        gbt.gridy = 0;
        topPanel.add(statePanel, gbt);
        gbt.insets.left = 80;
        gbt.gridx++;
        topPanel.add(periodPanel, gbt);
        
        // Switch panels
        GridBagConstraints gbs = new GridBagConstraints();
        for (int dvc = 0; dvc < QuadBoxSwitches.NUM_DEVICES; dvc++) {
            JPanel panel = switchPanel[dvc] = new JPanel();
            TitledBorder border = new TitledBorder(new LineBorder(Color.BLACK), panelNames[dvc]);
            border.setTitleColor(UiConstants.BLUE);
            panel.setBorder(border);
            panel.setLayout(new GridBagLayout());
            gbs.gridy = 0;
            gbs.insets.top = 0;
            for (int line = 0; line < switchLabel[dvc].length; line++) {
                if (switchLabel[dvc][line] == null) continue;
                gbs.gridx = 0;
                gbs.insets.left = 4;
                gbs.insets.right = 0;
                panel.add(switchLabel[dvc][line], gbs);
                gbs.gridx++;
                gbs.insets.left = 0;
                panel.add(switchStatus[dvc][line], gbs);
                gbs.gridx++;
                gbs.insets.right = 4;
                panel.add(switchOffRB[dvc][line], gbs);
                gbs.gridx++;
                panel.add(switchOnRB[dvc][line], gbs);
                gbs.gridy++;
                gbs.insets.top = -4;
            }
            panel.setMinimumSize(panel.getPreferredSize());
        }

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 3;
        add(topPanel, gbc);
        gbc.gridy++;
        gbc.gridwidth = 1;
        add(switchPanel[QuadBoxSwitches.DEVC_BFR], gbc);
        gbc.gridx++;
        add(switchPanel[QuadBoxSwitches.DEVC_PDU_5V], gbc);
        gbc.gridx++;
        add(switchPanel[QuadBoxSwitches.DEVC_PDU_24VC], gbc);
        gbc.gridy++;
        gbc.gridx = 0;
        add(switchPanel[QuadBoxSwitches.DEVC_REB_BULK], gbc);
        gbc.gridx++;
        add(switchPanel[QuadBoxSwitches.DEVC_PDU_24VD], gbc);
        gbc.gridx++;
        add(switchPanel[QuadBoxSwitches.DEVC_PDU_48V], gbc);
    }

    private void setUpdatePeriod() {
        try {
            double value = Double.valueOf(periodTextField.getText());
            sender.sendCommand(null, "setUpdatePeriod", (int)(1000 * value));
            periodTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            periodTextField.setText(String.valueOf(powerPeriod));
        }
    }

    private void setSwitchOn(int swId, boolean on) {
        sender.sendCommand(null, "setNamedSwitchOn", QuadBoxSwitches.switchIdToName.get(swId), on);
    }

    public void updateControlPanel(QuadBoxState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdatePowerStatus implements Runnable {

        private final QuadBoxState qs;

        UpdatePowerStatus(QuadBoxState qs) {
            this.qs = qs;
        }

        @Override
        public void run() {
            stateStatus.setText("RUNNING");
            stateStatus.setForeground(UiConstants.GREEN);

            periodTextField.setEnabled(true);
            powerPeriod = qs.getTickMillis() / 1000.0;
            periodTextField.setText(String.valueOf(powerPeriod));

            for (int swId : switchNames.keySet()) {
                int devc = swId >> 8;
                int line = switchLines.get(swId);
                SwitchState st = qs.getSwitchState(swId);
                st = st == null ? SwitchState.OFFLINE : st;
                JLabel label = switchStatus[devc][line];
                label.setText(st.toString());
                label.setForeground(st == SwitchState.ON ? UiConstants.GREEN :
                                    st == SwitchState.OFF ? Color.BLACK : UiConstants.BLUE);
                if (st != SwitchState.OFFLINE) {
                    JRadioButton selButton = st == SwitchState.ON ? switchOnRB[devc][line] : switchOffRB[devc][line];
                    selButton.setSelected(true);
                }
                switchStatus[devc][line].setEnabled(true);
                switchOffRB[devc][line].setEnabled(true);
                switchOnRB[devc][line].setEnabled(true);
            }
            for (int devc : bfrSwitchMap.keySet()) {
                SwitchState bfrSwSt = qs.getSwitchState(bfrSwitchMap.get(devc));
                Integer mainSwId = mainSwitchMap.get(devc);
                SwitchState mainSwSt = mainSwId == null ? SwitchState.ON : qs.getSwitchState(mainSwId);
                String suffix = bfrSwSt != SwitchState.ON ? " (BFR off)" : mainSwSt != SwitchState.ON ? " (Main off)" : "";
                Color color = bfrSwSt != SwitchState.ON ? UiConstants.RED :
                              mainSwSt != SwitchState.ON ? UiConstants.PURPLE : UiConstants.BLUE;
                JPanel panel = switchPanel[devc];
                TitledBorder border = (TitledBorder)panel.getBorder();
                border.setTitle(panelNames[devc] + suffix);
                border.setTitleColor(color);
            }
            repaint();
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            stateStatus.setText("STOPPED");
            stateStatus.setForeground(UiConstants.RED);
            periodTextField.setEnabled(false);
            for (int swId : switchNames.keySet()) {
                int dvc = swId >> 8;
                int line = switchLines.get(swId);
                switchStatus[dvc][line].setEnabled(false);
                switchOffRB[dvc][line].setEnabled(false);
                switchOnRB[dvc][line].setEnabled(false);
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
