package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.subsystem.power.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.power.data.ATSPowerState;

/**
 *  Implements the REB power supply control & monitoring panel.
 *
 *  @author Owen Saxton
 */
public class ATSPowerControlPanel extends javax.swing.JPanel {

    private static final Font FONT = new Font("Tahoma", 1, 12);
    private static final Color RED = new Color(180, 0, 0),
                               GREEN = new Color(0, 180, 0),
                               BLUE = new Color(0, 150, 180),
                               YELLOW = new Color(200, 180, 0),
                               GREY = Color.GRAY;
    private static final Map<Integer, String> chanNames = new HashMap<>();
    static {
        chanNames.put(ATSPowerState.CHAN_DIGITAL, "Dig");
        chanNames.put(ATSPowerState.CHAN_ANALOG, "Ana");
        chanNames.put(ATSPowerState.CHAN_CLK_HIGH, "ClkH");
        chanNames.put(ATSPowerState.CHAN_CLK_LOW, "ClkL");
        chanNames.put(ATSPowerState.CHAN_OD, "OD");
        chanNames.put(ATSPowerState.CHAN_HEATER, "Htr");
        chanNames.put(ATSPowerState.CHAN_DPHI, "DPHI");
        chanNames.put(ATSPowerState.CHAN_HV_BIAS, "HVbias");
    }

    private final CommandSender sender;
    private final JLabel lblPowerStateValue = new JLabel("....");
    private final JTextField tfPowerPeriod = new JTextField("...");
    private JButton btnPowerOn;
    private JButton btnPowerOff;
    private JButton btnBiasOn;
    private JButton btnBiasOff;
    private JTextField tfBiasVal = new JTextField();
    private JTextField tfDphiVal = new JTextField();
    private final JButton[] btnChanStatus = new JButton[ATSPowerState.NUM_CHANNELS];
    private double powerPeriod = 5, hvBias, dphi;

    public ATSPowerControlPanel(CommandSender sender) {
        this.sender = sender;
        initComponents();
        (new UpdatePowerStatus(new ATSPowerState(0, new Boolean[ATSPowerState.NUM_CHANNELS], 0.0, 0.0))).run();
        (new DisableSystem()).run();
    }

    private void initComponents()
    {
        //** Create all low-level components

        // State indicator
        Dimension ds = lblPowerStateValue.getPreferredSize();
        ds.width = 80;
        lblPowerStateValue.setPreferredSize(ds);
        lblPowerStateValue.setMinimumSize(ds);

        // Update period
        Dimension dp = tfPowerPeriod.getPreferredSize();
        dp.width = 50;
        tfPowerPeriod.setPreferredSize(dp);
        tfPowerPeriod.setMinimumSize(dp);
        tfPowerPeriod.setHorizontalAlignment(JTextField.CENTER);
        tfPowerPeriod.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setUpdatePeriod();
            }
        });

        // Power control buttons and fields
        JButton btn = new JButton("On");
        btn.setForeground(GREEN);
        btn.setFocusable(false);
        btn.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                sender.sendCommand(null, "powerOn");
            }
        });
        btnPowerOn = btn;

        btn = new JButton("Off");
        btn.setForeground(RED);
        btn.setFocusable(false);
        btn.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                sender.sendCommand(null, "powerOff");
            }
        });
        btnPowerOff = btn;

        btn = new JButton("On");
        btn.setForeground(GREEN);
        btn.setFocusable(false);
        btn.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                sender.sendCommand(null, "hvBiasOn");
            }
        });
        btnBiasOn = btn;

        btn = new JButton("Off");
        btn.setForeground(RED);
        btn.setFocusable(false);
        btn.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                sender.sendCommand(null, "hvBiasOff");
            }
        });
        btnBiasOff = btn;

        JTextField tf = new JTextField("...");
        Dimension dd = tf.getPreferredSize();
        dd.width = 60;
        tf.setPreferredSize(dd);
        tf.setMinimumSize(dd);
        tf.setHorizontalAlignment(JTextField.CENTER);
        tf.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setBiasValue();
            }
        });
        tfBiasVal = tf;

        tf = new JTextField("...");
        tf.setPreferredSize(dd);
        tf.setMinimumSize(dd);
        tf.setHorizontalAlignment(JTextField.CENTER);
        tf.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setDphiValue();
            }
        });
        tfDphiVal = tf;

        // Channel status buttons
        StringBuilder blanks = new StringBuilder();
        for (int j = 0; j < 6; j++) {
            blanks.append('\u00a0');
        }
        for (int cs = 0; cs < btnChanStatus.length; cs++) {
//            btn = new JButton(blanks.toString());
            btn = new JButton("   ");
            btn.setName(String.format("%02d", cs));
            btn.setFocusable(false);
            btn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                }
            });
            btnChanStatus[cs] = btn;
        }

        //** Construct panels

        // Power state
        JPanel pnlPowerState = new JPanel();
        pnlPowerState.add(new JLabel("System State: "));
        pnlPowerState.add(lblPowerStateValue);

        // Update period
        JPanel pnlPowerPeriod = new JPanel();
        pnlPowerPeriod.add(new JLabel("Update Period: "));
        pnlPowerPeriod.add(tfPowerPeriod);
        pnlPowerPeriod.add(new JLabel("sec"));

        // Power supply control
        JPanel pnlPowerControl = new JPanel();
        pnlPowerControl.setLayout(new GridBagLayout());
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.insets = new Insets(4, 6, 4, 6);

        gbp.gridx = 0;
        gbp.gridy = 0;
        gbp.gridwidth = 2;
        pnlPowerControl.add(new JLabel("Seq. Power"), gbp);
        gbp.gridy = 1;
        gbp.gridwidth = 1;
        pnlPowerControl.add(btnPowerOn, gbp);
        gbp.gridx++;
        pnlPowerControl.add(btnPowerOff, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        gbp.gridwidth = 2;
        pnlPowerControl.add(new JLabel("HV Bias"), gbp);
        gbp.gridy = 1;
        gbp.gridwidth = 1;
        pnlPowerControl.add(btnBiasOn, gbp);
        gbp.gridx++;
        pnlPowerControl.add(btnBiasOff, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        pnlPowerControl.add(new JLabel("Bias Volts"), gbp);
        gbp.gridy = 1;
        pnlPowerControl.add(tfBiasVal, gbp);

        gbp.gridx++;
        gbp.gridy = 0;
        pnlPowerControl.add(new JLabel("DPHI Volts"), gbp);
        gbp.gridy = 1;
        pnlPowerControl.add(tfDphiVal, gbp);

        // Channel status
        JPanel pnlChanStatus = new JPanel();
        pnlChanStatus.setLayout(new GridBagLayout());
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.insets = new Insets(4, 6, 4, 6);
        gbs.gridx = 0;
        for (int j = 0; j < btnChanStatus.length; j++) {
            gbs.gridy = 0;
            pnlChanStatus.add(new JLabel(chanNames.get(j)), gbs);
            gbs.gridy = 1;
            pnlChanStatus.add(btnChanStatus[j], gbs);
            gbs.gridx++;
        }

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 10, 6, 10);
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(pnlPowerState, gbc);
        gbc.gridx++;
        add(pnlPowerPeriod, gbc);
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.gridy++;
        gbc.gridx = 0;
        gbc.gridwidth = 2;
        add(pnlPowerControl, gbc);
        gbc.gridy++;
        add(pnlChanStatus, gbc);
    }

    private void setUpdatePeriod() {
        try {
            double value = Double.valueOf(tfPowerPeriod.getText());
            sender.sendCommand(null, "setUpdatePeriod", (int)(1000 * value));
            //tfPowerPeriod.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfPowerPeriod.setText(String.valueOf(powerPeriod));
        }
    }

    private void setBiasValue() {
        try {
            double value = Double.valueOf(tfBiasVal.getText());
            sender.sendCommand(null, "setHvBias", value);
            //tfBiasVal.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfBiasVal.setText(String.format("%.1f", hvBias));
        }
    }

    private void setDphiValue() {
        try {
            double value = Double.valueOf(tfDphiVal.getText());
            sender.sendCommand(null, "setDphi", value);
            //tfDphiVal.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfDphiVal.setText(String.format("%.1f", dphi));
        }
    }

    public void updateControlPanel(ATSPowerState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdatePowerStatus implements Runnable {

        private final ATSPowerState rs;

        UpdatePowerStatus(ATSPowerState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            lblPowerStateValue.setText("RUNNING");
            lblPowerStateValue.setForeground(GREEN);

            powerPeriod = rs.getTickMillis() / 1000.0;
            tfPowerPeriod.setText(String.valueOf(powerPeriod));
            tfPowerPeriod.setEnabled(true);

            btnPowerOn.setEnabled(true);
            btnPowerOff.setEnabled(true);
            btnBiasOn.setEnabled(true);
            btnBiasOff.setEnabled(true);

            hvBias = rs.getHvBias();
            tfBiasVal.setText(String.valueOf(hvBias));
            tfBiasVal.setEnabled(true);

            dphi = rs.getDphi();
            tfDphiVal.setText(String.valueOf(dphi));
            tfDphiVal.setEnabled(true);

            Boolean[] state = rs.getPowerState();
            for (int j = 0; j < btnChanStatus.length; j++) {
                btnChanStatus[j].setBackground(state[j] == null ? BLUE : state[j] ? GREEN : RED);
                btnChanStatus[j].setEnabled(true);
            }
            
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            lblPowerStateValue.setText("STOPPED");
            lblPowerStateValue.setForeground(RED);
            tfPowerPeriod.setEnabled(false);
            btnPowerOn.setEnabled(false);
            btnPowerOff.setEnabled(false);
            btnBiasOn.setEnabled(false);
            btnBiasOff.setEnabled(false);
            tfBiasVal.setEnabled(false);
            tfDphiVal.setEnabled(false);
            for (JButton btn : btnChanStatus) {
                btn.setBackground(GREY);
                btn.setEnabled(false);
            }
        } 
    }

    private static final long serialVersionUID = 1L;
}
