package org.lsst.ccs.subsystem.power.ui.jas;

import java.awt.BorderLayout;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.swing.JComponent;
import javax.swing.JPanel;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.EmbeddedObjectDeserializationException;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.OptionalPage;
import org.lsst.ccs.gconsole.plugins.monitor.SectionedTableView;
import org.lsst.ccs.gconsole.plugins.monitor.TabbedView;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;
import org.lsst.ccs.subsystem.power.data.QuadBoxState;
import org.lsst.ccs.subsystem.power.ui.QuadBoxControlPanel;

/**
 * @author tonyj
 */
@Plugin(name = "Quadbox Control Plugin", description = "Quadbox Control Plugin")
public class QuadBoxControlPlugin extends ConsolePlugin {

    @Override
    public void initialize() {
        Console.getConsole().getOptionalPageService().add(new QuadboxControlOptionalPage());
    }
    
    static class QuadboxControlOptionalPage implements OptionalPage {

        private final CommandSender cmdSender = new CommandSender();
        private final Map<String, BusListener> busListeners = new ConcurrentHashMap<>();
        
        @Override
        public boolean isAutoOpen() {
            return false;
        }

        @Override
        public String getPage(AgentInfo agent) {
            return agent != null && agent.hasAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT)
                     ? getPageName(agent) : null;
        }
        
        @Override
        public boolean agentDisconnected(AgentInfo agent) {
            //System.out.println("agentDisconnected called");
            BusListener listener = busListeners.get(getPageName(agent));
            if (listener == null) return true;
            listener.controls.disableSystem();
            return false;
        }
        
        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
            //System.out.println("open called");
            String pageName = getPageName(agent);
            String agentName = agent.getName();
            BusListener listener = busListeners.get(pageName);
            if (listener != null) {
                initControls(agentName, listener.controls);
                return existingComponent;
            }

            JPanel panel = new JPanel();
            panel.setLayout(new java.awt.BorderLayout());

            // Create the monitoring panel
            TabbedView view = new TabbedView();
            view.setViewFactory(name -> new SectionedTableView());            
            MonitorFilter monitorFilter = new MonitorFilter(agent.getName());
            view.setFilter(monitorFilter);
            view.install();
            panel.add(view.getPanel(), BorderLayout.CENTER);

            // Create the control panel
            QuadBoxControlPanel controls = new QuadBoxControlPanel(cmdSender);
            controls.setSubsystems(agent.getName());
            panel.add(controls, BorderLayout.SOUTH);

            // Create the bus listener
            listener = new BusListener(controls);
            AgentMessagingLayer aml = Console.getConsole().getMessagingAccess();
            aml.addStatusMessageListener(listener, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName)
                                                              && msg instanceof StatusSubsystemData);
            busListeners.put(pageName, listener);

            // Initialize the control panel
            initControls(agentName, controls);
            return panel;
        }
        
        @Override
        public void closed(String pageName) {
            //System.out.println("closed called");
            BusListener listener = busListeners.remove(pageName);
            if (listener == null) {
                throw new RuntimeException("No page exists with the name " + pageName);
            }
            Console.getConsole().getMessagingAccess().removeStatusMessageListener(listener);
        }

        private String getPageName(AgentInfo agent) {
            return "CCS Subsystems/" + agent.getName();
        }

        private void initControls(String agentName, QuadBoxControlPanel controls) {
            Object response = cmdSender.sendCommand(agentName, null, "getFullState");
            if (response instanceof QuadBoxState) {
                controls.updateControlPanel((QuadBoxState)response);
            }
        }

    }

    static class BusListener implements StatusMessageListener {

        private final QuadBoxControlPanel controls;
        
        BusListener(QuadBoxControlPanel controls) {
            this.controls = controls;
        }
        
        @Override
        public void onStatusMessage(StatusMessage msg) {
            try {
                StatusSubsystemData sd = (StatusSubsystemData)msg;
                if (sd.getDataKey().equals(QuadBoxState.KEY)) {
                    controls.updateControlPanel((QuadBoxState)((KeyValueData)sd.getSubsystemData()).getValue());
                }
            }
            catch (EmbeddedObjectDeserializationException x) {
            }
        }

    }
    
}
