package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.auxelex.RebPs;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.RebPsState;
import org.lsst.ccs.subsystem.power.ui.jas.CommandSender;

/**
 *  Implements the REB power supply control & monitoring panel.
 *
 *  @author Owen Saxton
 */
public class RebPsControlPanel extends javax.swing.JPanel {

    private static final Font FONT = new Font("Tahoma", 1, 12);
    private static final Color RED = new Color(180, 0, 0),
                               GREEN = new Color(0, 180, 0),
                               YELLOW = new Color(200, 180, 0),
                               GREY = Color.GRAY;
    private static final int REB_Q = RebPs.REB_QUANTUM, N_REB = 2 * REB_Q, N_PS = 8,
                             PS_DIG = 0, PS_ANA = 1, PS_OD = 2, PS_CLKH = 3, PS_CLKL = 4, PS_HTR = 5,
                             PS_DPHI = 6, PS_HV = 7;
    //private static final int[] psNumsSR = {RebPs.PS_DIGITAL, RebPs.PS_ANALOG, RebPs.PS_OD, RebPs.PS_CLK_LOW,
    //                                       RebPs.PS_CLK_HIGH, RebPs.PS_HEATER, RebPs.PS_HV_BIAS};
    //private static final boolean[] isDacSR = {false, false, false, false, false, false, false, true};
    //private static final int[] psNumsCR = {RebPs.PS_DIGITAL, RebPs.PS_ANALOG, RebPs.PS_OD, RebPs.PS_CLK_LOW,
    //                                       RebPs.PS_CLK_HIGH, RebPs.PS_HEATER, RebPs.PS_DPHI,RebPs.PS_HV_BIAS};
    //private static final boolean[] isDacCR = {false, false, false, false, false, false, true, true};
    //private static final Map<Integer, Integer> psNumMapSR = new HashMap<>();
    //private static final Map<Integer, Integer> psNumMapCR = new HashMap<>();
    private static final Map<Integer, Integer> btnMapSR = new HashMap<>();
    static {
        for (int reb = 0; reb < N_REB; reb++) {
            btnMapSR.put((reb << 4) | RebPs.PS_DIGITAL, (reb << 4) | PS_DIG);
            btnMapSR.put((reb << 4) | RebPs.PS_ANALOG, (reb << 4) | PS_ANA);
            btnMapSR.put((reb << 4) | RebPs.PS_OD, (reb << 4) | PS_OD);
            btnMapSR.put((reb << 4) | RebPs.PS_CLK_HIGH, (reb << 4) | PS_CLKH);
            btnMapSR.put((reb << 4) | RebPs.PS_CLK_LOW, (reb << 4) | PS_CLKL);
            btnMapSR.put((reb << 4) | RebPs.PS_HEATER, (reb << 4) | PS_HTR);
            btnMapSR.put((reb << 4) | RebPs.PS_HV_BIAS, (reb << 4) | PS_HV);
        }
    }
    private static final Map<Integer, Integer> invBtnMapSR = new HashMap<>();
    static {
        for (int ps : btnMapSR.keySet()) {
            invBtnMapSR.put(btnMapSR.get(ps), ps);
        }
    }
    private static final Map<Integer, Integer> btnMapCR = new HashMap<>();
    static {
        for (int reb = 0; reb < N_REB; reb++) {
            if ((reb % REB_Q) != REB_Q - 1) {
                btnMapCR.put((reb << 4) | RebPs.PS_DIGITAL, (reb << 4) | PS_DIG);
            }
            else {
                btnMapCR.put((reb << 4) | RebPs.PS_DIGITAL, ((reb - 2) << 4) | PS_HTR);
            }
            btnMapCR.put((reb << 4) | RebPs.PS_ANALOG, (reb << 4) | PS_ANA);
            btnMapCR.put((reb << 4) | RebPs.PS_OD, (reb << 4) | PS_OD);
            btnMapCR.put((reb << 4) | RebPs.PS_CLK_HIGH, (reb << 4) | PS_CLKH);
            btnMapCR.put((reb << 4) | RebPs.PS_CLK_LOW, (reb << 4) | PS_CLKL);
            btnMapCR.put((reb << 4) | RebPs.PS_DPHI, (reb << 4) | PS_DPHI);
            btnMapCR.put((reb << 4) | RebPs.PS_HV_BIAS, (reb << 4) | PS_HV);
        }
    }
    private static final Map<Integer, Integer> invBtnMapCR = new HashMap<>();
    static {
        for (int ps : btnMapCR.keySet()) {
            invBtnMapCR.put(btnMapCR.get(ps), ps);
        }
    }
    private final CommandSender gui;
    private final JPanel pnlPowerState = new JPanel();
    private final JLabel lblPowerState = new JLabel("System State: ");
    private final JLabel lblPowerStateValue = new JLabel("....");
    private final JPanel pnlPsId = new JPanel();
    private final JLabel lblPsId = new JLabel("PS ID: ");
    private final JLabel lblPsIdValue = new JLabel("....");
    private final JPanel pnlMainPower = new JPanel();
    private final JLabel lblMainPower = new JLabel("Main Power: ");
    private final JButton btnMainPower = new JButton("");
    private final JPanel pnlPowerPeriod = new JPanel();
    private final JLabel lblPowerPeriod = new JLabel("Update Period: ");
    private final JTextField tfPowerPeriod = new JTextField("...");
    private final JLabel lblPowerPeriodUnits = new JLabel("sec");
    private final JPanel pnlPowerControl = new JPanel();
    private final JLabel[] lblPsNames = new JLabel[N_PS];
    private final JLabel[] lblRebName = new JLabel[N_REB];
    private final JButton[][] btnToggle = new JButton[N_REB][N_PS];
    private final JButton[] btnSeqOn = new JButton[N_REB];
    private final JButton[] btnSeqOff = new JButton[N_REB];
    private final JLabel lblBiasDac = new JLabel("Bias DAC");
    private final JTextField[] tfBiasDac = new JTextField[N_REB];
    private final JLabel lblDphiDac = new JLabel("Dphi DAC");
    private final JTextField[] tfDphiDac = new JTextField[N_REB];
    private final int biasDacs[] = new int[N_REB];
    private final int dphiDacs[] = new int[N_REB];
    private int numReb = 0;
    private boolean gotCR = false; //, gotProto = false;
    private String powerName;
    private double powerPeriod = 5;
    //private int[] psNums = psNumsSR;
    //private boolean[] isDac = isDacSR;

    public RebPsControlPanel(CommandSender gui) {
        //for (int j = 0; j < N_PS; j++) {
        //    psNumMapSR.put(psNumsSR[j], j);
        //    psNumMapCR.put(psNumsCR[j], j);
        //}
        this.gui = gui;
        initComponents();
        (new UpdatePowerStatus(new RebPsState(0, PowerChanState.PWR_STATE_OFFLINE, new int[]{},
                                              new double[]{}, new double[]{}, 0, 0, ""))).run();
        (new DisableSystem()).run();
    }

    public void setSubsystems(String powerName) {
        this.powerName = powerName;
    }

    private void initComponents()
    {
        //** Create all low-level components

        // State indicator
        Dimension ds = lblPowerStateValue.getPreferredSize();
        ds.width = 70;
        lblPowerStateValue.setPreferredSize(ds);
        lblPowerStateValue.setMinimumSize(ds);

        // Power supply ID
        Dimension di = lblPsIdValue.getPreferredSize();
        di.width = 50;
        lblPsIdValue.setPreferredSize(di);
        lblPsIdValue.setMinimumSize(di);
        
        // Main power control
        Dimension dc = btnMainPower.getPreferredSize();
        dc.height = 16;
        btnMainPower.setPreferredSize(dc);
        btnMainPower.setMinimumSize(dc);
        btnMainPower.setFocusable(false);
        btnMainPower.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
               toggleMainPower();
            }
        });

        // Update period
        Dimension dp = tfPowerPeriod.getPreferredSize();
        dp.width = 50;
        tfPowerPeriod.setPreferredSize(dp);
        tfPowerPeriod.setMinimumSize(dp);
        tfPowerPeriod.setHorizontalAlignment(JTextField.CENTER);
        tfPowerPeriod.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setUpdatePeriod();
            }
        });

        // Power control labels
        for (int reb = 0; reb < N_REB; reb++) {
            lblRebName[reb] = new JLabel("REB" + reb);
        }

        lblPsNames[PS_DIG] = new JLabel("Dig");
        lblPsNames[PS_ANA] = new JLabel("Ana");
        lblPsNames[PS_OD] = new JLabel("OD");
        lblPsNames[PS_CLKH] = new JLabel("ClkH");
        lblPsNames[PS_CLKL] = new JLabel("ClkL");
        lblPsNames[PS_HTR] = new JLabel("Htr");
        lblPsNames[PS_DPHI] = new JLabel("Dphi");
        lblPsNames[PS_HV] = new JLabel("HVbias");

        // Power control buttons and fields
        for (int reb = 0; reb < N_REB; reb++) {
            JButton btn = new JButton("On");
            btn.setForeground(GREEN);
            btn.setName(String.valueOf(reb));
            btn.setFocusable(false);
            btn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JButton)evt.getSource()).getName();
                    sequencePower(Integer.valueOf(name), true);
                }
            });
            btnSeqOn[reb] = btn;

            btn = new JButton("Off");
            btn.setForeground(RED);
            btn.setName(String.valueOf(reb));
            btn.setFocusable(false);
            btn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JButton)evt.getSource()).getName();
                    sequencePower(Integer.valueOf(name), false);
                }
            });
            btnSeqOff[reb] = btn;

            for (int ps = 0; ps < N_PS; ps++) {
                btn = new JButton(" ");
                btn.setName(String.valueOf((reb << 4) | ps));
                btn.setFocusable(false);
                btn.addActionListener(new ActionListener() {
                    @Override
                    public void actionPerformed(ActionEvent evt) {
                       int srcPsReb = Integer.valueOf(((JButton)evt.getSource()).getName());
                       int dstPsReb = (gotCR ? invBtnMapCR : invBtnMapSR).get(srcPsReb);
                       togglePower(dstPsReb >> 4, dstPsReb & 0x0f);
                    }
                });
                btnToggle[reb][ps] = btn;
            }

            JTextField tf = new JTextField("...");
            tf.setName("D" + reb);
            Dimension dd = tf.getPreferredSize();
            dd.width = 60;
            tf.setPreferredSize(dd);
            tf.setMinimumSize(dd);
            tf.setHorizontalAlignment(JTextField.CENTER);
            tf.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JTextField)evt.getSource()).getName();
                    setDphiDac(Integer.valueOf(name.substring(1)));
                }
            });
            tfDphiDac[reb] = tf;

            tf = new JTextField("...");
            tf.setName("B" + reb);
            tf.setPreferredSize(dd);
            tf.setMinimumSize(dd);
            tf.setHorizontalAlignment(JTextField.CENTER);
            tf.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JTextField)evt.getSource()).getName();
                    setBiasDac(Integer.valueOf(name.substring(1)));
                }
            });
            tfBiasDac[reb] = tf;
        }

        //** Construct panels

        // Power state
        pnlPowerState.add(lblPowerState);
        pnlPowerState.add(lblPowerStateValue);

        // Power supply ID
        pnlPsId.add(lblPsId);
        pnlPsId.add(lblPsIdValue);

        // Main power control
        pnlMainPower.add(lblMainPower);
        pnlMainPower.add(btnMainPower);

        // Update period
        pnlPowerPeriod.add(lblPowerPeriod);
        pnlPowerPeriod.add(tfPowerPeriod);
        pnlPowerPeriod.add(lblPowerPeriodUnits);

        // Power supply control
        pnlPowerControl.setLayout(new GridBagLayout());
        drawControl(N_REB, false);

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 10, 6, 10);
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(pnlPowerState, gbc);
        gbc.gridx++;
        add(pnlPsId, gbc);
        gbc.gridx++;
        add(pnlMainPower, gbc);
        gbc.gridx++;
        add(pnlPowerPeriod, gbc);
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.gridy++;
        gbc.gridx = 0;
        gbc.gridwidth = 4;
        add(pnlPowerControl, gbc);
    }

    private void drawControl(int nReb, boolean isCR) {
        if (nReb == 0 || nReb == numReb && isCR == gotCR) return;

        numReb = nReb;
        gotCR = isCR;
        pnlPowerControl.removeAll();
        
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.insets = new Insets(4, 6, 4, 6);

        gbp.gridx = 1;
        gbp.gridwidth = 2;
        gbp.gridy = 0;
        pnlPowerControl.add(new JLabel("Sequenced"), gbp);
        gbp.gridx += 2;
        gbp.gridwidth = 1;
        for (int ps = 0; ps < N_PS; ps++) {
            if (gotCR || ps != PS_DPHI) {
                pnlPowerControl.add(lblPsNames[ps], gbp);
                gbp.gridx++;
            }
            if (gotCR && ps == PS_DPHI) {
                pnlPowerControl.add(lblDphiDac, gbp);
                gbp.gridx++;
            }
            if (ps == PS_HV) {
                pnlPowerControl.add(lblBiasDac, gbp);
                gbp.gridx++;
            }
        }

        gbp.gridx = 0;
        gbp.gridy++;
        for (int reb = 0; reb < numReb; reb++) {
            if (gotCR && (reb % REB_Q) == REB_Q - 1) continue;
            pnlPowerControl.add(lblRebName[reb], gbp);
            gbp.gridx++;
            pnlPowerControl.add(btnSeqOn[reb], gbp);
            gbp.gridx++;
            pnlPowerControl.add(btnSeqOff[reb], gbp);
            gbp.gridx++;
            for (int ps = 0; ps < N_PS; ps++) {
                if (gotCR || ps != PS_DPHI) {
                    if (ps != PS_HTR || !gotCR || (reb % REB_Q) == 0) {
                        pnlPowerControl.add(btnToggle[reb][ps], gbp);
                    }
                    gbp.gridx++;
                }
                if (gotCR && ps == PS_DPHI) {
                    pnlPowerControl.add(tfDphiDac[reb], gbp);
                    gbp.gridx++;
                }
                if (ps == PS_HV) {
                    pnlPowerControl.add(tfBiasDac[reb], gbp);
                    gbp.gridx++;
                }
            }
            gbp.gridx = 0;
            gbp.gridy++;
        }
    }

    private void setUpdatePeriod() {
        try {
            double value = Double.valueOf(tfPowerPeriod.getText());
            gui.sendCommand(powerName, null, "setUpdatePeriod", (int)(1000 * value));
            tfPowerPeriod.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfPowerPeriod.setText(String.valueOf(powerPeriod));
        }
    }

    private void toggleMainPower() {
        gui.sendCommand(powerName, null, "toggleMainPower");
    }

    private void sequencePower(int reb, boolean on) {
        gui.sendCommand(powerName, null, "sequencePower", reb, on);
    }

    private void togglePower(int reb, int ps) {
        //if (!gotProto && !isDac[ps]) return;
        gui.sendCommand(powerName, null, "togglePower", reb, ps);
    }

    private void setBiasDac(int reb) {
        try {
            double value = Integer.valueOf(tfBiasDac[reb].getText());
            gui.sendCommand(powerName, null, "setBiasDac", reb, value);
            tfBiasDac[reb].setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfBiasDac[reb].setText(String.valueOf(biasDacs[reb]));
        }
    }

    private void setDphiDac(int reb) {
        try {
            double value = Integer.valueOf(tfDphiDac[reb].getText());
            gui.sendCommand(powerName, null, "setDphiDac", reb, value);
            tfDphiDac[reb].setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfDphiDac[reb].setText(String.valueOf(dphiDacs[reb]));
        }
    }

    public void updateControlPanel(RebPsState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdatePowerStatus implements Runnable {

        private final RebPsState rs;

        UpdatePowerStatus(RebPsState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            int nReb = rs.getPowerState().length;
            //gotProto = rs.getPsVersion() == RebPs.VERSION_PROTO;
            boolean isCR = rs.getPsType() == RebPs.TYPE_CORNER;
            //psNums = isCR ? psNumsCR : psNumsSR;
            //isDac = isCR ? isDacCR : isDacSR;
            drawControl(nReb, isCR);
            lblPowerStateValue.setText("RUNNING");
            lblPowerStateValue.setForeground(GREEN);

            lblPsIdValue.setText(rs.getPsId());

            int mps = rs.getMainPowerState();
            Color color = mps == PowerChanState.PWR_STATE_ON ? GREEN :
                          mps == PowerChanState.PWR_STATE_OFF ? RED : GREY;
            btnMainPower.setEnabled(color != GREY);
            btnMainPower.setBackground(color);

            tfPowerPeriod.setEnabled(true);
            powerPeriod = rs.getTickMillis() / 1000.0;
            tfPowerPeriod.setText(String.valueOf(powerPeriod));

            Map<Integer, Integer> btnMap = isCR ? btnMapCR : btnMapSR;
            int[] state = rs.getPowerState();
            double[] bDacs = rs.getHvBiasDacs();
            double[] dDacs = rs.getDphiDacs();
            for (int reb = 0; reb < nReb; reb++) {
                boolean present = state[reb] >= 0;
                btnSeqOn[reb].setEnabled(present);
                btnSeqOff[reb].setEnabled(present);
                for (int ps = 0; ps < RebPs.NUM_PS; ps++) {
                    color = present ? ((state[reb] & (1 << (ps + 1))) == 0) ? RED : GREEN : GREY;
                    int btnPsReb = btnMap.get((reb << 4) | ps);
                    btnToggle[btnPsReb >> 4][btnPsReb & 0x0f].setEnabled(present);
                    btnToggle[btnPsReb >> 4][btnPsReb & 0x0f].setBackground(color);
                }
                tfBiasDac[reb].setEnabled(present);
                int value = (reb < bDacs.length) ? (int)bDacs[reb] : 0;
                tfBiasDac[reb].setText(String.valueOf(value));
                biasDacs[reb] = value;
                tfDphiDac[reb].setEnabled(present);
                value = (reb < dDacs.length) ? (int)dDacs[reb] : 0;
                tfDphiDac[reb].setText(String.valueOf(value));
                dphiDacs[reb] = value;
            }
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            lblPowerStateValue.setText("STOPPED");
            lblPowerStateValue.setForeground(RED);
            btnMainPower.setEnabled(false);
            btnMainPower.setBackground(GREY);
            tfPowerPeriod.setEnabled(false);
            for (int reb = 0; reb < numReb; reb++) {
                btnSeqOn[reb].setEnabled(false);
                btnSeqOff[reb].setEnabled(false);
                for (int ps = 0; ps < N_PS; ps++) {
                    btnToggle[reb][ps].setEnabled(false);
                    btnToggle[reb][ps].setBackground(GREY);
                }
                tfBiasDac[reb].setEnabled(false);
                tfDphiDac[reb].setEnabled(false);
            }
        } 
    }

    private static final long serialVersionUID = 1L;
}
