package org.lsst.ccs.subsystem.power.ui.jas;

import java.awt.BorderLayout;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.swing.JComponent;
import javax.swing.JPanel;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.EmbeddedObjectDeserializationException;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.OptionalPage;
import org.lsst.ccs.gconsole.plugins.monitor.SectionedTableView;
import org.lsst.ccs.gconsole.plugins.monitor.TabbedView;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.monitor.ui.CommandSender;
import org.lsst.ccs.subsystem.power.data.PowerSupplyAgentProperties;
import org.lsst.ccs.subsystem.power.data.RebPsFullState;
import org.lsst.ccs.subsystem.power.data.RebPsState;
import org.lsst.ccs.subsystem.power.ui.RebPsControlPanel;

/**
 *
 * @author tonyj
 */
@Plugin(name = "CCS Reb Power Supply Plugin New", description = "CCS Reb Power Supply Plugin New")
public class RebPsControlPluginNew extends ConsolePlugin {

    private OptionalPage rebPsOptionalPage;

    @Override
    public void initialize() {
        rebPsOptionalPage = new RebPsControlPluginNew.RebPsControlOptionalPage(Console.getConsole());
        Console.getConsole().addOptionalPage(rebPsOptionalPage);
    }
    
    static class RebPsControlOptionalPage implements OptionalPage {


        private final PrivateCommandSender cmdSender;
        private final Map<String,BusListener> busListeners = new ConcurrentHashMap<>();
        
        RebPsControlOptionalPage(Console console) {
            cmdSender = new PrivateCommandSender(Console.getConsole());            
        }

        @Override
        public boolean isAutoOpen() {
            return false;
        }

        @Override
        public String getPath() {
            return "CCS Subsystems/Reb PS";
        }

        @Override
        public String getPage(AgentInfo agent) {
            return getPageNameForAgent(agent);
        }

        private String getPageNameForAgent(AgentInfo agent) {
            String pageName = null;
            if (agent.hasAgentProperty(PowerSupplyAgentProperties.REB_POWER_SUPPLY_TYPE_AGENT_PROPERTY)) {
                String agentName = agent.getName();
                String title = agentName.replace("ccs-", "").replace("ccs", "").toUpperCase() + " Control";
                pageName = title;
            }
            return pageName;            
        }
        
        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
//            String agentName = agent.getName();
//            RebPsControlGui gui = new RebPsControlGui(agentName);
//            gui.initGui();
//            return gui.getGuiLayout();
            JPanel panel = new JPanel();
            panel.setLayout(new java.awt.BorderLayout());
            //Create the monitoring panel
            TabbedView view = new TabbedView();
            view.setViewFactory(name -> new SectionedTableView());            
            RebPsMonitorFilter monitorFilter = new RebPsMonitorFilter(agent.getName());
            view.setFilter(monitorFilter);
            view.install();
            panel.add(view.getPanel(), BorderLayout.CENTER);
            
            


            RebPsControlPanel controls = new RebPsControlPanel(cmdSender);
            controls.setSubsystems(agent.getName());
            
            String pageName = getPageNameForAgent(agent);
            if ( busListeners.containsKey(pageName) ) {
                throw new RuntimeException("Page with name "+pageName+" already exists for the Reb Ps plugin.");
            }
            BusListener busListener = new BusListener(controls, agent.getName(), cmdSender);
            Console.getConsole().getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(busListener);
            Console.getConsole().getMessagingAccess().addStatusMessageListener(busListener);
            busListeners.put(pageName, busListener);
            
            panel.add(controls, BorderLayout.SOUTH);
             
            return panel;
        }

        @Override
        public void closed(String page) {
            BusListener busListener = busListeners.remove(page);
            if ( busListener != null ) {
                Console.getConsole().getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(busListener);
                Console.getConsole().getMessagingAccess().addStatusMessageListener(busListener);                
            }
        }
        
    }

    
    static class BusListener implements AgentPresenceListener, StatusMessageListener {
        
        private final RebPsControlPanel controls;
        private final String agentName;
        private final CommandSender cmdSender;
        
        BusListener(RebPsControlPanel controls, String agentName, CommandSender cmdSender) {
            this.controls = controls;
            this.agentName = agentName;
            this.cmdSender = cmdSender;
        }
        
        @Override
        public void onStatusMessage(StatusMessage msg) {
            if (msg.getOriginAgentInfo().getName().equals(agentName)) {

                if (msg instanceof StatusSubsystemData) {
                    try {
                        StatusSubsystemData sd = (StatusSubsystemData) msg;
                        if (sd.getDataKey().equals(RebPsState.KEY)) {
                            controls.updateControlPanel(0, (RebPsState) ((KeyValueData) sd.getSubsystemData()).getValue());
                        }
                    } catch (EmbeddedObjectDeserializationException x) {
                    }
                }

            }
        }

        @Override
        public void disconnecting(AgentInfo agent) {
            if ( agent.getName().equals(agentName) ) {
                //The id is not used internally
                controls.disableSystem(0);
            } 
        }

        @Override
        public void connecting(AgentInfo agent) {
            if ( agent.getName().equals(agentName) ) {
                Object response = cmdSender.sendCommand(agentName, null, "getFullState");
                if (response instanceof RebPsFullState) {
                    RebPsFullState r = (RebPsFullState) response;
                    controls.updateControlPanel(0, r.getRebPsState());
                }
            } 
        }

    }
    
    
    //This class will have to be removed when we replace the old monitoring
    //with the new classes. It is needed to keep the RebPsControlPanel class
    //work with both implementations.
    static class PrivateCommandSender implements CommandSender {
        
        private final Console console;
        
        PrivateCommandSender(Console console) {
            this.console = console;
        }

        @Override
        public Object sendCommand(String dest, String target, String cmnd, Object... args) {
            try {
                return sendCommandRaw(dest, target, cmnd, args);
            } catch (Exception e) {
                console.getLoggerUI().error("Command error: ", e);
                return null;
            }
        }
        
        @Override
        public Object sendCommandRaw(String dest, String target, String cmnd,
                Object... args) throws Exception {
            String dst = dest + (target == null ? "" : "/" + target);
            dst += "/"+cmnd;
            return console.sendCommand(Duration.ofSeconds(1),dst, args);
        }

    }
    

}
