package org.lsst.ccs.subsystem.power.ui;

import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JWindow;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.subsystem.monitor.ui.MonitorGui;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 **  Implements the VST power control panel.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class PowerControlPanel extends MonitorGui.Panel
                               implements PowerStatusPanel.Parent {

    private static final String NB_SPACE = "\u00a0";
    private final JPanel btnPanel = new JPanel();
    private final JButton btCfgPower = new JButton("Configure...");
    private final JButton btCfgSave = new JButton("Save Config");
    private final JButton btPwrOn = new JButton("Turn On");
    private final JButton btPwrOff = new JButton("Turn Off");
    private final JButton btQuit = new JButton("Quit");

    private final JLabel lbMessage = new JLabel(NB_SPACE);
    
    private final JPanel mainPanel = new JPanel();

    private final PowerStatusPanel pwrPanel;
    private final PowerControlGui gui;
    private final Logger log;
    private String vstPower;
    private boolean commandOK = false;
    private static final long serialVersionUID = 1L;

   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  gui     The controlling GUI
    **
    **  @param  noQuit  If true, don't add Quit button
    **
    ***************************************************************************
    */
    public PowerControlPanel(PowerControlGui gui, boolean noQuit) {
        this.gui = gui;
        this.log = gui.getLogger();
        pwrPanel = new PowerStatusPanel(this);
        initButtonPanel(noQuit);
        initMainPanel();
        initTopPanel();
    }

   /**
    ***************************************************************************
    **
    **  Gets the power state from the power subsystem.
    **
    **  @return  The power state
    **
    ***************************************************************************
    */
    @Override
    public List<PowerChanState> getPowerState() {
        if (vstPower != null) {
            Object resp = sendCommand(null, "getPowerState");
            return (List<PowerChanState>)resp;
        }
        else {
            display("VST power subsystem not running");
            return null;
        }
    }

   /**
    ***************************************************************************
    **
    **  Resizes the panel if necessary.
    **
    ***************************************************************************
    */
    @Override
    public void powerResized() {
        Component f = getTopFrame();
        f.setSize(f.getPreferredSize());
    }

   /**
    ***************************************************************************
    **
    **  Enables the power subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void enableSystem(String name, int id) {
        pwrPanel.updatePowerStatus(vstPower = name);
    }

   /**
    ***************************************************************************
    **
    **  Disables the power subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void disableSystem(String name, int id) {
        pwrPanel.updatePowerStatus(vstPower = null);
    }

   /**
    ***************************************************************************
    **
    **  Processes status data messages from the VST power subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void handleStatusData(String name, int id, long time, String key,
                                 Object d) {
        if (key.equals(PowerChanState.KEY)) {
            Object ps = ((KeyValueData)d).getValue();
            pwrPanel.updatePowerState((List<PowerChanState>)ps);
        }
    }

   /**
    ***************************************************************************
    **
    **  Initializes the button sub-panel.
    **
    ***************************************************************************
    */
    private void initButtonPanel(boolean noQuit) {

        btnPanel.setLayout(new GridBagLayout());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;

        btnPanel.add(btCfgPower, gbc);
        btCfgPower.setFocusable(false);
        btCfgPower.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                cfgPower();
            }
        });

        gbc.gridy++;
        btnPanel.add(btCfgSave, gbc);
        btCfgSave.setFocusable(false);
        btCfgSave.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                cfgSave();
            }
        });

        gbc.gridy++;
        btnPanel.add(btPwrOn, gbc);
        btPwrOn.setFocusable(false);
        btPwrOn.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                powerOn();
            }
        });

        gbc.gridy++;
        btnPanel.add(btPwrOff, gbc);
        btPwrOff.setFocusable(false);
        btPwrOff.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                powerOff();
            }
        });

        if (!noQuit) {
            gbc.gridy++;
            btnPanel.add(btQuit, gbc);
            btQuit.setFocusable(false);
            btQuit.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    quit();
                }
            });
        }
    }

   /**
    ***************************************************************************
    **
    **  Initializes the main panel.
    **
    ***************************************************************************
    */
    private void initMainPanel() {
        mainPanel.setBorder(BorderFactory.createTitledBorder(""));
        mainPanel.setLayout(new GridBagLayout());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(10, 10, 10, 10);
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridx = 0;
        gbc.gridy = 0;

        mainPanel.add(pwrPanel, gbc);
        gbc.gridx++;
        mainPanel.add(btnPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy++;
        gbc.gridwidth = 2;
        gbc.anchor = GridBagConstraints.NORTH;
        mainPanel.add(lbMessage, gbc);
    }

   /**
    ***************************************************************************
    **
    **  Initializes the top-level panel.
    **
    ***************************************************************************
    */
    private void initTopPanel() {
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTH;
        add(mainPanel, gbc);
    }

   /**
    ***************************************************************************
    **
    **  Configures the power.
    **
    ***************************************************************************
    */
    private void cfgPower() {
        Power[] power = (Power[])sendCommand(null, "getPowerConfig");
        if (power == null) {
            display("Cannot get power configuration");
            return;
        }
        if (PowerConfigPanel.edit(SwingUtilities.getWindowAncestor(this),
                                  power)) {
            sendCommand(null, "setPowerConfig", (Object)power);
        }
    }

   /**
    ***************************************************************************
    **
    **  Saves the configuration.
    **
    ***************************************************************************
    */
    private void cfgSave() {
        sendCommand(null, "saveNamedConfig", "");
    }

   /**
    ***************************************************************************
    **
    **  Turns on the power.
    **
    ***************************************************************************
    */
    private void powerOn() {
        sendCommand(null, "powerOn");
        if (commandOK) {
            pwrPanel.updatePowerState(null);
        }
    }

   /**
    ***************************************************************************
    **
    **  Turns off the power.
    **
    ***************************************************************************
    */
    private void powerOff() {
        sendCommand(null, "powerOff");
        if (commandOK) {
            pwrPanel.updatePowerState(null);
        }
    }

   /**
    ***************************************************************************
    **
    **  Stops the subsystem and exits.
    **
    ***************************************************************************
    */
    private void quit() {
        sendCommand(null, "shutdown");
        try {
            Thread.sleep(200);   // Wait for command to be sent
        }
        catch (InterruptedException e) {
        }
        System.exit(0);
    }

   /**
    ***************************************************************************
    **
    **  Sends a command to the power subsystem.
    **
    ***************************************************************************
    */
    private Object sendCommand(String child, String name, Object... params)
    {
        commandOK = false;
        if (vstPower == null) {
            display("VST power system not running");
            return false;
        }
        try {
            Object resp = gui.sendCommandRaw(vstPower, child, name, params);
            display(NB_SPACE);
            commandOK = true;
            return resp;
        }
        catch (Exception e) {
            log.error("Command error: " + e);
            String text = e.getMessage();
            display(text == null || text.isEmpty() ? e.toString() : text);
            return null;
        }
    }

   /**
    ***************************************************************************
    **
    **  Displays a message.
    **
    ***************************************************************************
    */
    private void display(String message) {
        lbMessage.setText(message);
    }

   /**
    ***************************************************************************
    **
    **  Gets the top-most container of this panel.
    **
    ***************************************************************************
    */
    private Component getTopFrame() {
        Component c = this;
        while (!(c instanceof JFrame) && !(c instanceof JWindow)
                && !(c instanceof JDialog) && c.getParent() != null) {
            c = c.getParent();
        }
        return c;
    }

}
