package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.rebps.RebPs;
import org.lsst.ccs.subsystem.monitor.ui.CommandSender;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.RebPsState;

/**
 ******************************************************************************
 **
 **  Implements the REB power supply control & monitoring panel.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RebPsControlPanel extends javax.swing.JPanel {

    private final static Font FONT = new Font("Tahoma", 1, 12);
    private final static Color RED = new Color(180, 0, 0),
                               GREEN = new Color(0, 180, 0),
                               YELLOW = new Color(200, 180, 0),
                               GREY = Color.GRAY;
    private final static int N_REB = RebPs.NUM_REBS, N_PS = RebPs.NUM_PS + 1;
    private final static int[] psNumsSR = {-1, RebPs.PS_DIGITAL, RebPs.PS_ANALOG,
                                           RebPs.PS_OD, RebPs.PS_CLK_LOW,
                                           RebPs.PS_CLK_HIGH, RebPs.PS_HEATER,
                                           RebPs.PS_HV_BIAS};
    private final static int[] psNumsCR = {-1, RebPs.PS_DIGITAL, RebPs.PS_ANALOG,
                                           RebPs.PS_OD, RebPs.PS_CLK_LOW,
                                           RebPs.PS_CLK_HIGH, RebPs.PS_DPHI,
                                           RebPs.PS_HV_BIAS};
    private final static Map<Integer, Integer> psNumMapSR = new HashMap<>();
    private final static Map<Integer, Integer> psNumMapCR = new HashMap<>();
    private final static String[] psNamesSR = {"Master", "Dig", "Ana", "OD",
                                               "ClkL", "ClkH", "Htr", "HVbias"};
    private final static String[] psNamesCR = {"Master", "Dig", "Ana", "OD",
                                               "ClkL", "ClkH", "Dphi", "HVbias"};
    private final CommandSender gui;
    private final JPanel pnlPowerState = new JPanel();
    private final JLabel lblPowerState = new JLabel("System State: ");
    private final JLabel lblPowerStateValue = new JLabel("....");
    private final JPanel pnlPsId = new JPanel();
    private final JLabel lblPsId = new JLabel("PS ID: ");
    private final JLabel lblPsIdValue = new JLabel("....");
    private final JPanel pnlMainPower = new JPanel();
    private final JLabel lblMainPower = new JLabel("Main Power: ");
    private final JButton btnMainPower = new JButton("");
    private final JPanel pnlPowerPeriod = new JPanel();
    private final JLabel lblPowerPeriod = new JLabel("Update Period: ");
    private final JTextField tfPowerPeriod = new JTextField("...");
    private final JLabel lblPowerPeriodUnits = new JLabel("sec");
    private final JPanel pnlPowerControl = new JPanel();
    private final JLabel[] lblPsNames = new JLabel[N_PS];
    private final JButton[][] btnToggle = new JButton[N_REB][N_PS];
    private final JButton[] btnSeqOn = new JButton[N_REB];
    private final JButton[] btnSeqOff = new JButton[N_REB];
    private final JTextField[] tfBiasDac = new JTextField[N_REB];
    private final JLabel lblDphi = new JLabel("Dphi DAC");
    private final JTextField[] tfDphiDac = new JTextField[N_REB];
    private final int biasDacs[] = new int[N_REB];
    private final int dphiDacs[] = new int[N_REB];
    private boolean gotCR = false;
    private String powerName;
    private double powerPeriod = 5;
    private int[] psNums = psNumsSR;

    public RebPsControlPanel(CommandSender gui) {
        for (int j = 0; j < N_PS; j++) {
            psNumMapSR.put(psNumsSR[j], j);
            psNumMapCR.put(psNumsCR[j], j);
        }
        this.gui = gui;
        initComponents();
        (new UpdatePowerStatus(new RebPsState(0, PowerChanState.PWR_STATE_OFFLINE,
                                              new int[]{}, new double[]{},
                                              new double[]{}, ""))).run();
        (new DisableSystem()).run();
    }

    public void setSubsystems(String powerName) {
        this.powerName = powerName;
    }

    private void initComponents()
    {
        //** Create all low-level components

        // State indicator
        Dimension ds = lblPowerStateValue.getPreferredSize();
        ds.width = 60;
        lblPowerStateValue.setPreferredSize(ds);
        lblPowerStateValue.setMinimumSize(ds);

        // Power supply ID
        Dimension di = lblPsIdValue.getPreferredSize();
        di.width = 50;
        lblPsIdValue.setPreferredSize(di);
        lblPsIdValue.setMinimumSize(di);
        
        // Main power control
        Dimension dc = btnMainPower.getPreferredSize();
        dc.height = 16;
        btnMainPower.setPreferredSize(dc);
        btnMainPower.setMinimumSize(dc);
        btnMainPower.setFocusable(false);
        btnMainPower.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
               toggleMainPower();
            }
        });

        // Update period
        Dimension dp = tfPowerPeriod.getPreferredSize();
        dp.width = 50;
        tfPowerPeriod.setPreferredSize(dp);
        tfPowerPeriod.setMinimumSize(dp);
        tfPowerPeriod.setHorizontalAlignment(JTextField.CENTER);
        tfPowerPeriod.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setUpdatePeriod();
            }
        });

        // Power control buttons and fields
        for (int j = 0; j < N_PS; j++) {
            lblPsNames[j] = new JLabel(psNamesSR[j]);
        }

        for (int reb = 0; reb < N_REB; reb++) {
            JButton btn = new JButton("On");
            btn.setForeground(GREEN);
            btn.setName(reb + "");
            btn.setFocusable(false);
            btn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JButton)evt.getSource()).getName();
                    sequencePower(Integer.valueOf(name), true);
                }
            });
            btnSeqOn[reb] = btn;

            btn = new JButton("Off");
            btn.setForeground(RED);
            btn.setName(String.valueOf(reb));
            btn.setFocusable(false);
            btn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JButton)evt.getSource()).getName();
                    sequencePower(Integer.valueOf(name), false);
                }
            });
            btnSeqOff[reb] = btn;

            for (int ps = 0; ps < N_PS; ps++) {
                btn = new JButton(" ");
                btn.setName(String.format("%02d", 10 * reb + ps));
                btn.setFocusable(false);
                btn.addActionListener(new ActionListener() {
                    @Override
                    public void actionPerformed(ActionEvent evt) {
                       String name = ((JButton)evt.getSource()).getName();
                       togglePower(Integer.valueOf(name.substring(0, 1)),
                                   Integer.valueOf(name.substring(1, 2)));
                    }
                });
                btnToggle[reb][ps] = btn;
            }

            JTextField tf = new JTextField("...");
            tf.setName("B" + reb);
            Dimension dd = tf.getPreferredSize();
            dd.width = 60;
            tf.setPreferredSize(dd);
            tf.setMinimumSize(dd);
            tf.setHorizontalAlignment(JTextField.CENTER);
            tf.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JTextField)evt.getSource()).getName();
                    setBiasDac(Integer.valueOf(name.substring(1)));
                }
            });
            tfBiasDac[reb] = tf;

            tf = new JTextField("...");
            tf.setName("D" + reb);
            tf.setPreferredSize(dd);
            tf.setMinimumSize(dd);
            tf.setHorizontalAlignment(JTextField.CENTER);
            tf.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JTextField)evt.getSource()).getName();
                    setDphiDac(Integer.valueOf(name.substring(1)));
                }
            });
            tfDphiDac[reb] = tf;
        }

        //** Construct panels

        // Power state
        pnlPowerState.add(lblPowerState);
        pnlPowerState.add(lblPowerStateValue);

        // Power supply ID
        pnlPsId.add(lblPsId);
        pnlPsId.add(lblPsIdValue);

        // Main power control
        pnlMainPower.add(lblMainPower);
        pnlMainPower.add(btnMainPower);

        // Update period
        pnlPowerPeriod.add(lblPowerPeriod);
        pnlPowerPeriod.add(tfPowerPeriod);
        pnlPowerPeriod.add(lblPowerPeriodUnits);

        // Power supply control
        pnlPowerControl.setLayout(new GridBagLayout());
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.insets = new Insets(4, 6, 4, 6);
        gbp.gridx = 1;
        gbp.gridwidth = 2;
        gbp.gridy = 0;
        pnlPowerControl.add(new JLabel("Sequenced"), gbp);
        gbp.gridx += 2;
        gbp.gridwidth = 1;
        for (int j = 0; j < N_PS; j++) {
            pnlPowerControl.add(lblPsNames[j], gbp);
            gbp.gridx++;
        }
        pnlPowerControl.add(new JLabel("Bias DAC"), gbp);
        gbp.gridx = 0;
        gbp.gridy++;
        for (int reb = 0; reb < N_REB; reb++) {
            pnlPowerControl.add(new JLabel("REB " + reb), gbp);
            gbp.gridx++;
            pnlPowerControl.add(btnSeqOn[reb], gbp);
            gbp.gridx++;
            pnlPowerControl.add(btnSeqOff[reb], gbp);
            gbp.gridx++;
            for (int ps = 0; ps < N_PS; ps++) {
                pnlPowerControl.add(btnToggle[reb][ps], gbp);
                gbp.gridx++;
            }
            pnlPowerControl.add(tfBiasDac[reb], gbp);
            gbp.gridx = 0;
            gbp.gridy++;
        }

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 10, 6, 10);
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(pnlPowerState, gbc);
        gbc.gridx++;
        add(pnlPsId, gbc);
        gbc.gridx++;
        add(pnlMainPower, gbc);
        gbc.gridx++;
        add(pnlPowerPeriod, gbc);
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.gridy++;
        gbc.gridx = 0;
        gbc.gridwidth = 4;
        add(pnlPowerControl, gbc);
    }

    private void reshape(boolean isCR) {
        if (isCR == gotCR) return;
        gotCR = isCR;

        String[] psNames = gotCR ? psNamesCR : psNamesSR;
        for (int j = 0; j < N_PS; j++) {
            lblPsNames[j].setText(psNames[j]);
        }

        GridBagConstraints gbp = new GridBagConstraints();
        gbp.insets = new Insets(4, 6, 4, 6);
        if (gotCR) {
            gbp.gridx = N_PS + 4;
            gbp.gridy = 0;
            pnlPowerControl.add(lblDphi, gbp);
            for (int j = 0; j < N_REB - 1; j++) {
                gbp.gridy++;
                pnlPowerControl.add(tfDphiDac[j], gbp);
            }
            for (int j = 3; j < N_PS; j++) {
                pnlPowerControl.remove(btnToggle[N_REB - 1][j]);
            }
            pnlPowerControl.remove(tfBiasDac[N_REB - 1]);
        }
        else {
            pnlPowerControl.remove(lblDphi);
            for (int j = 0; j < N_REB - 1; j++) {
                pnlPowerControl.remove(tfDphiDac[j]);
            }
            gbp.gridx = 6;
            gbp.gridy = N_REB;
            for (int j = 3; j < N_PS; j++) {
                pnlPowerControl.add(btnToggle[N_REB - 1][j], gbp);
                gbp.gridx++;
            }
            pnlPowerControl.add(tfBiasDac[N_REB - 1], gbp);
        }
    }

    private void setUpdatePeriod() {
        try {
            double value = Double.valueOf(tfPowerPeriod.getText());
            gui.sendCommand(powerName, null, "setUpdatePeriod", (int)(1000 * value));
            tfPowerPeriod.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfPowerPeriod.setText(String.valueOf(powerPeriod));
        }
    }

    private void toggleMainPower() {
        gui.sendCommand(powerName, null, "toggleMainPower");
    }

    private void sequencePower(int reb, boolean on) {
        gui.sendCommand(powerName, null, "sequencePower", reb, on);
    }

    private void togglePower(int reb, int ps) {
        gui.sendCommand(powerName, null, "togglePower", reb, psNums[ps]);
    }

    private void setBiasDac(int reb) {
        try {
            double value = Integer.valueOf(tfBiasDac[reb].getText());
            gui.sendCommand(powerName, null, "setBiasDac", reb, value);
            tfBiasDac[reb].setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfBiasDac[reb].setText(String.valueOf(biasDacs[reb]));
        }
    }

    private void setDphiDac(int reb) {
        try {
            double value = Integer.valueOf(tfDphiDac[reb].getText());
            gui.sendCommand(powerName, null, "setDphiDac", reb, value);
            tfDphiDac[reb].setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfDphiDac[reb].setText(String.valueOf(dphiDacs[reb]));
        }
    }

    public void updateControlPanel(int id, RebPsState rs) {
        SwingUtilities.invokeLater(new UpdatePowerStatus(rs));
    }

    public void disableSystem(int id) {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdatePowerStatus implements Runnable {

        private final RebPsState rs;

        UpdatePowerStatus(RebPsState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            Map<Integer, Integer> psNumMap;
            boolean isCR = rs.getpsId().startsWith("CR");
            if (isCR) {
                psNums = psNumsCR;
                psNumMap = psNumMapCR;
            }
            else {
                psNums = psNumsSR;
                psNumMap = psNumMapSR;
            }
            reshape(isCR);
            lblPowerStateValue.setText("RUNNING");
            lblPowerStateValue.setForeground(GREEN);

            lblPsIdValue.setText(rs.getpsId());

            btnMainPower.setEnabled(true);
            int mps = rs.getMainPowerState();
            Color color = mps == PowerChanState.PWR_STATE_ON ? GREEN
                            : mps == PowerChanState.PWR_STATE_OFF ? RED : GREY;
            btnMainPower.setBackground(color);

            tfPowerPeriod.setEnabled(true);
            powerPeriod = rs.getTickMillis() / 1000.0;
            tfPowerPeriod.setText(String.valueOf(powerPeriod));

            int[] state = rs.getPowerState();
            double[] bDacs = rs.getHvBiasDacs();
            double[] dDacs = rs.getDphiDacs();
            for (int reb = 0; reb < N_REB; reb++) {
                btnSeqOn[reb].setEnabled(true);
                btnSeqOff[reb].setEnabled(true);
                boolean present = reb < state.length;
                for (int ps = 0; ps < N_PS; ps++) {
                    color = present ? ((state[reb] & (1 << ps)) == 0) ? RED : GREEN : GREY;
                    int j = psNumMap.get(ps - 1);
                    btnToggle[reb][j].setEnabled(true);
                    btnToggle[reb][j].setBackground(color);
                }
                tfBiasDac[reb].setEnabled(true);
                int value = (reb < bDacs.length) ? (int)bDacs[reb] : 0;
                tfBiasDac[reb].setText(String.valueOf(value));
                biasDacs[reb] = value;
                tfDphiDac[reb].setEnabled(true);
                value = (reb < dDacs.length) ? (int)dDacs[reb] : 0;
                tfDphiDac[reb].setText(String.valueOf(value));
                dphiDacs[reb] = value;
            }
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            lblPowerStateValue.setText("STOPPED");
            lblPowerStateValue.setForeground(RED);
            btnMainPower.setEnabled(false);
            btnMainPower.setBackground(GREY);
            tfPowerPeriod.setEnabled(false);
            for (int reb = 0; reb < N_REB; reb++) {
                btnSeqOn[reb].setEnabled(false);
                btnSeqOff[reb].setEnabled(false);
                for (int ps = 0; ps < N_PS; ps++) {
                    btnToggle[reb][ps].setEnabled(false);
                    btnToggle[reb][ps].setBackground(GREY);
                }
                tfBiasDac[reb].setEnabled(false);
                tfDphiDac[reb].setEnabled(false);
            }
        } 
    }

    private static final long serialVersionUID = 1L;
}
